use crate::base_1d::BarcodeBase1D;
use crate::error::{BarcodeError, Result};
use super::rss_utils::{get_widths, calculate_gtin_check_digit, pad_left, parse_int64};

// ISO 24724 Limited tables
const LEFT_REG_OFFSETS: &[i32] = &[0, 183064, 820064, 1000776, 1491021, 1979845, 1996939];

const T_EVEN_LTD: &[i32] = &[28, 728, 6454, 203, 2408, 1, 16632];
const MODULES_ODD_LTD: &[i32] = &[17, 13, 9, 15, 11, 19, 7];
const MODULES_EVEN_LTD: &[i32] = &[9, 13, 17, 11, 15, 7, 19];
const WIDEST_ODD_LTD: &[i32] = &[6, 5, 3, 5, 4, 8, 1];
const WIDEST_EVEN_LTD: &[i32] = &[3, 4, 6, 4, 5, 1, 8];

const CHECKSUM_WEIGHT_LTD: &[i32] = &[
    1, 3, 9, 27, 81, 65, 17, 51, 64, 14,
    42, 37, 22, 66, 20, 60, 2, 6, 18, 54,
    73, 41, 34, 13, 39, 28, 84, 74,
];

const FINDER_PATTERN_LTD: &[i32] = &[
    1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 3, 3, 1, 1,
    1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 3, 2, 1, 1,
    1, 1, 1, 1, 1, 1, 1, 1, 1, 3, 3, 1, 1, 1,
    1, 1, 1, 1, 1, 1, 1, 2, 1, 1, 3, 2, 1, 1,
    1, 1, 1, 1, 1, 1, 1, 2, 1, 2, 3, 1, 1, 1,
    1, 1, 1, 1, 1, 1, 1, 3, 1, 1, 3, 1, 1, 1,
    1, 1, 1, 1, 1, 2, 1, 1, 1, 1, 3, 2, 1, 1,
    1, 1, 1, 1, 1, 2, 1, 1, 1, 2, 3, 1, 1, 1,
    1, 1, 1, 1, 1, 2, 1, 2, 1, 1, 3, 1, 1, 1,
    1, 1, 1, 1, 1, 3, 1, 1, 1, 1, 3, 1, 1, 1,
    1, 1, 1, 2, 1, 1, 1, 1, 1, 1, 3, 2, 1, 1,
    1, 1, 1, 2, 1, 1, 1, 1, 1, 2, 3, 1, 1, 1,
    1, 1, 1, 2, 1, 1, 1, 2, 1, 1, 3, 1, 1, 1,
    1, 1, 1, 2, 1, 2, 1, 1, 1, 1, 3, 1, 1, 1,
    1, 1, 1, 3, 1, 1, 1, 1, 1, 1, 3, 1, 1, 1,
    1, 2, 1, 1, 1, 1, 1, 1, 1, 1, 3, 2, 1, 1,
    1, 2, 1, 1, 1, 1, 1, 1, 1, 2, 3, 1, 1, 1,
    1, 2, 1, 1, 1, 1, 1, 2, 1, 1, 3, 1, 1, 1,
    1, 2, 1, 1, 1, 2, 1, 1, 1, 1, 3, 1, 1, 1,
    1, 2, 1, 2, 1, 1, 1, 1, 1, 1, 3, 1, 1, 1,
    1, 3, 1, 1, 1, 1, 1, 1, 1, 1, 3, 1, 1, 1,
    1, 1, 1, 1, 1, 1, 1, 1, 2, 1, 3, 2, 1, 1,
    1, 1, 1, 1, 1, 1, 1, 1, 2, 2, 3, 1, 1, 1,
    1, 1, 1, 1, 1, 1, 1, 2, 2, 1, 3, 1, 1, 1,
    1, 1, 1, 1, 1, 2, 1, 1, 2, 1, 3, 1, 1, 1,
    1, 1, 1, 2, 1, 1, 1, 1, 2, 1, 3, 1, 1, 1,
    1, 2, 1, 1, 1, 1, 1, 1, 2, 1, 3, 1, 1, 1,
    1, 1, 1, 1, 1, 1, 2, 1, 1, 1, 3, 2, 1, 1,
    1, 1, 1, 1, 1, 1, 2, 1, 1, 2, 3, 1, 1, 1,
    1, 1, 1, 2, 1, 1, 2, 1, 1, 1, 3, 1, 1, 1,
    1, 2, 1, 1, 1, 1, 2, 1, 1, 1, 3, 1, 1, 1,
    1, 1, 1, 1, 2, 1, 1, 1, 1, 1, 3, 2, 1, 1,
    1, 1, 1, 1, 2, 1, 1, 1, 1, 2, 3, 1, 1, 1,
    1, 1, 1, 1, 2, 1, 1, 2, 1, 1, 3, 1, 1, 1,
    1, 1, 1, 1, 2, 1, 2, 1, 1, 1, 3, 1, 1, 1,
    1, 1, 1, 1, 2, 2, 1, 1, 1, 1, 3, 1, 1, 1,
    1, 2, 1, 1, 2, 1, 1, 1, 1, 1, 3, 1, 1, 1,
    1, 1, 1, 1, 1, 1, 2, 1, 2, 1, 3, 1, 1, 1,
    1, 1, 1, 1, 1, 1, 2, 2, 1, 1, 3, 1, 1, 1,
    1, 1, 1, 2, 1, 1, 2, 1, 1, 1, 3, 1, 1, 1,
    1, 1, 1, 1, 1, 2, 2, 1, 1, 1, 3, 1, 1, 1,
    1, 1, 1, 2, 2, 1, 1, 1, 1, 1, 3, 1, 1, 1,
    1, 2, 1, 1, 1, 1, 2, 1, 1, 1, 3, 1, 1, 1,
    1, 1, 2, 1, 1, 1, 1, 1, 1, 1, 3, 2, 1, 1,
    1, 1, 2, 1, 1, 1, 1, 1, 1, 2, 3, 1, 1, 1,
    1, 1, 2, 1, 1, 1, 1, 2, 1, 1, 3, 1, 1, 1,
    1, 1, 2, 1, 1, 2, 1, 1, 1, 1, 3, 1, 1, 1,
    1, 1, 2, 2, 1, 1, 1, 1, 1, 1, 3, 1, 1, 1,
    1, 2, 2, 1, 1, 1, 1, 1, 1, 1, 3, 1, 1, 1,
    1, 1, 1, 1, 2, 1, 1, 1, 2, 1, 3, 1, 1, 1,
    1, 1, 1, 1, 2, 1, 1, 2, 2, 1, 3, 1, 1, 1,
    1, 1, 1, 2, 2, 1, 1, 1, 2, 1, 3, 1, 1, 1,
    1, 1, 1, 1, 2, 2, 1, 1, 2, 1, 3, 1, 1, 1,
    1, 2, 1, 1, 2, 1, 1, 1, 2, 1, 3, 1, 1, 1,
    1, 1, 1, 1, 2, 1, 2, 1, 2, 1, 3, 1, 1, 1,
    1, 1, 1, 1, 2, 1, 2, 2, 1, 1, 3, 1, 1, 1,
    1, 1, 1, 2, 2, 1, 2, 1, 1, 1, 3, 1, 1, 1,
    1, 2, 1, 1, 2, 1, 2, 1, 1, 1, 3, 1, 1, 1,
    1, 1, 1, 1, 3, 1, 1, 1, 2, 1, 3, 1, 1, 1,
    1, 2, 1, 1, 2, 2, 1, 1, 1, 1, 3, 1, 1, 1,
    1, 2, 1, 1, 1, 1, 2, 1, 2, 1, 3, 1, 1, 1,
    1, 1, 2, 1, 1, 1, 2, 1, 1, 1, 3, 1, 1, 1,
    1, 1, 2, 1, 2, 1, 1, 1, 1, 1, 3, 1, 1, 1,
    1, 1, 2, 2, 1, 1, 1, 1, 1, 1, 3, 1, 1, 1,
    1, 2, 2, 1, 2, 1, 1, 1, 1, 1, 3, 1, 1, 1,
    1, 2, 2, 1, 1, 1, 1, 1, 2, 1, 3, 1, 1, 1,
    1, 2, 2, 1, 1, 1, 2, 1, 1, 1, 3, 1, 1, 1,
    1, 1, 2, 1, 2, 1, 2, 1, 1, 1, 3, 1, 1, 1,
    1, 1, 2, 1, 2, 2, 1, 1, 1, 1, 3, 1, 1, 1,
    1, 2, 2, 1, 2, 1, 2, 1, 1, 1, 3, 1, 1, 1,
    1, 1, 1, 1, 2, 1, 2, 1, 2, 2, 3, 1, 1, 1,
    1, 1, 1, 1, 2, 2, 2, 1, 2, 1, 3, 1, 1, 1,
    1, 1, 1, 2, 2, 1, 2, 1, 2, 1, 3, 1, 1, 1,
    1, 1, 1, 1, 2, 2, 2, 2, 1, 1, 3, 1, 1, 1,
    1, 2, 1, 1, 2, 1, 2, 1, 2, 1, 3, 1, 1, 1,
    1, 1, 1, 1, 3, 1, 2, 1, 2, 1, 3, 1, 1, 1,
    1, 2, 1, 1, 2, 2, 2, 1, 1, 1, 3, 1, 1, 1,
    1, 2, 1, 1, 3, 1, 2, 1, 1, 1, 3, 1, 1, 1,
    1, 1, 2, 1, 2, 1, 2, 1, 2, 1, 3, 1, 1, 1,
    1, 1, 2, 1, 2, 2, 2, 1, 1, 1, 3, 1, 1, 1,
    1, 1, 2, 2, 2, 1, 2, 1, 1, 1, 3, 1, 1, 1,
    1, 2, 2, 1, 2, 1, 2, 1, 2, 1, 3, 1, 1, 1,
    1, 1, 2, 2, 2, 1, 2, 2, 1, 1, 3, 1, 1, 1,
    1, 2, 2, 1, 2, 2, 2, 1, 1, 1, 3, 1, 1, 1,
    1, 2, 2, 2, 2, 1, 2, 1, 1, 1, 3, 1, 1, 1,
    1, 1, 1, 1, 3, 1, 2, 1, 2, 2, 3, 1, 1, 1,
    1, 1, 1, 1, 3, 1, 2, 2, 2, 1, 3, 1, 1, 1,
    1, 1, 1, 2, 3, 1, 2, 1, 2, 1, 3, 1, 1, 1,
    1, 1, 1, 1, 3, 2, 2, 1, 2, 1, 3, 1, 1, 1,
    1, 2, 1, 1, 3, 1, 2, 1, 2, 1, 3, 1, 1, 1,
];

fn determine_group_ltd(reg_value: i32) -> usize {
    let mut group = 0;
    if reg_value > 183063 { group = 1; }
    if reg_value > 820063 { group = 2; }
    if reg_value > 1000775 { group = 3; }
    if reg_value > 1491020 { group = 4; }
    if reg_value > 1979844 { group = 5; }
    if reg_value > 1996938 { group = 6; }
    group
}

/// GS1 DataBar Limited barcode encoder.
pub struct GS1DataBarLimited {
    pub base_1d: BarcodeBase1D,
    gtin14: String,
}

impl GS1DataBarLimited {
    pub fn new(output_format: &str) -> Self {
        let mut s = Self {
            base_1d: BarcodeBase1D::new(output_format),
            gtin14: String::new(),
        };
        s.base_1d.show_text = false;
        s
    }

    pub fn gtin14(&self) -> &str { &self.gtin14 }
    pub fn human_readable(&self) -> String { format!("(01){}", self.gtin14) }

    pub fn draw(&mut self, code: &str, width: i32, height: i32) -> Result<()> {
        let pattern = self.encode(code)?;
        if self.base_1d.base.is_svg_output() {
            let display = "";
            return self.base_1d.draw_svg_bars(&pattern, width, height, display);
        }
        #[cfg(any(feature = "png", feature = "jpeg"))]
        {
            self.base_1d.render_bars_to_png(&pattern, width, height, "")
        }
        #[cfg(not(any(feature = "png", feature = "jpeg")))]
        Err(BarcodeError::FormatError("PNG/JPEG features not enabled".into()))
    }

    pub fn encode(&mut self, code: &str) -> Result<Vec<i32>> {
        for c in code.chars() {
            if !c.is_ascii_digit() {
                return Err(BarcodeError::InvalidData(
                    "GS1 DataBar Limited requires numeric digits only".into(),
                ));
            }
        }
        if code.len() < 8 || code.len() > 13 {
            return Err(BarcodeError::InvalidData("input must be 8-13 digits".into()));
        }

        let padded = pad_left(code, 13);
        if padded.as_bytes()[0] != b'0' && padded.as_bytes()[0] != b'1' {
            return Err(BarcodeError::InvalidData(
                "GS1 DataBar Limited requires leading digit 0 or 1 after padding".into(),
            ));
        }

        let accum = parse_int64(&padded);
        let left_reg = (accum / 2013571) as i32;
        let right_reg = (accum % 2013571) as i32;

        let left_group = determine_group_ltd(left_reg);
        let right_group = determine_group_ltd(right_reg);

        let left_reg_adj = if left_group > 0 {
            left_reg - LEFT_REG_OFFSETS[left_group]
        } else {
            left_reg
        };
        let right_reg_adj = if right_group > 0 {
            right_reg - LEFT_REG_OFFSETS[right_group]
        } else {
            right_reg
        };

        let left_odd = left_reg_adj / T_EVEN_LTD[left_group];
        let left_even = left_reg_adj % T_EVEN_LTD[left_group];
        let right_odd = right_reg_adj / T_EVEN_LTD[right_group];
        let right_even = right_reg_adj % T_EVEN_LTD[right_group];

        // Generate width patterns (14 elements each)
        let mut left_widths = [0i32; 14];
        let mut right_widths = [0i32; 14];

        let w = get_widths(left_odd, MODULES_ODD_LTD[left_group], 7, WIDEST_ODD_LTD[left_group], 1);
        for i in 0..7 { left_widths[i * 2] = w[i]; }
        let w = get_widths(left_even, MODULES_EVEN_LTD[left_group], 7, WIDEST_EVEN_LTD[left_group], 0);
        for i in 0..7 { left_widths[i * 2 + 1] = w[i]; }
        let w = get_widths(right_odd, MODULES_ODD_LTD[right_group], 7, WIDEST_ODD_LTD[right_group], 1);
        for i in 0..7 { right_widths[i * 2] = w[i]; }
        let w = get_widths(right_even, MODULES_EVEN_LTD[right_group], 7, WIDEST_EVEN_LTD[right_group], 0);
        for i in 0..7 { right_widths[i * 2 + 1] = w[i]; }

        // Checksum (modulo 89)
        let mut checksum = 0;
        for i in 0..14 {
            checksum += CHECKSUM_WEIGHT_LTD[i] * left_widths[i];
            checksum += CHECKSUM_WEIGHT_LTD[i + 14] * right_widths[i];
        }
        checksum %= 89;

        let check_start = checksum as usize * 14;
        let check_elements = &FINDER_PATTERN_LTD[check_start..check_start + 14];

        // Build total widths (46 elements)
        let mut total_widths = vec![0i32; 46];
        total_widths[0] = 1;
        total_widths[1] = 1;
        total_widths[44] = 1;
        total_widths[45] = 1;

        total_widths[2..16].copy_from_slice(&left_widths);
        total_widths[16..30].copy_from_slice(&check_elements[..14]);
        total_widths[30..44].copy_from_slice(&right_widths);

        let cd = calculate_gtin_check_digit(&padded);
        self.gtin14 = format!("{}{}", padded, cd);

        Ok(total_widths)
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_encode_basic() {
        let mut g = GS1DataBarLimited::new("svg");
        let widths = g.encode("0000000000000").unwrap();
        assert_eq!(widths.len(), 46);
    }

    #[test]
    fn test_encode_short() {
        let mut g = GS1DataBarLimited::new("svg");
        let widths = g.encode("12345678").unwrap();
        assert_eq!(widths.len(), 46);
    }

    #[test]
    fn test_encode_too_short() {
        let mut g = GS1DataBarLimited::new("svg");
        assert!(g.encode("1234567").is_err());
    }

    #[test]
    fn test_gtin14() {
        let mut g = GS1DataBarLimited::new("svg");
        g.encode("0000000000000").unwrap();
        assert_eq!(g.gtin14().len(), 14);
    }
}
