use crate::base_1d::BarcodeBase1D;
use crate::error::{BarcodeError, Result};
use super::rss_utils::{get_widths, bin2pat};

/// GS1 DataBar Expanded symbol type.
#[derive(Clone, Copy, PartialEq)]
pub enum ExpandedSymbolType {
    Unstacked = 0,
    Stacked = 1,
}

// Encoding modes
const ENC_NUMERIC: i32 = 0;
const ENC_ALPHA: i32 = 1;
const ENC_ISO: i32 = 2;
const ENC_INVALID: i32 = 3;
const ENC_ANY_ENC: i32 = 4;
const ENC_ALPHA_OR_ISO: i32 = 5;

// ISO 24724 Tables
const G_SUM_EXP: &[i32] = &[0, 348, 1388, 2948, 3988];
const T_EVEN_EXP: &[i32] = &[4, 20, 52, 104, 204];
const MODULES_ODD_EXP: &[i32] = &[12, 10, 8, 6, 4];
const MODULES_EVEN_EXP: &[i32] = &[5, 7, 9, 11, 13];
const WIDEST_ODD_EXP: &[i32] = &[7, 5, 4, 3, 1];
const WIDEST_EVEN_EXP: &[i32] = &[2, 4, 5, 6, 8];

#[rustfmt::skip]
const CHECKSUM_WEIGHT_EXP: &[i32] = &[
    1, 3, 9, 27, 81, 32, 96, 77, 20, 60, 180, 118, 143, 7, 21, 63, 189,
    145, 13, 39, 117, 140, 209, 205, 193, 157, 49, 147, 19, 57, 171, 91,
    62, 186, 136, 197, 169, 85, 44, 132, 185, 133, 188, 142, 4, 12, 36,
    108, 113, 128, 173, 97, 80, 29, 87, 50, 150, 28, 84, 41, 123, 158, 52,
    156, 46, 138, 203, 187, 139, 206, 196, 166, 76, 17, 51, 153, 37, 111,
    122, 155, 43, 129, 176, 106, 107, 110, 119, 146, 16, 48, 144, 10, 30,
    90, 59, 177, 109, 116, 137, 200, 178, 112, 125, 164, 70, 210, 208, 202,
    184, 130, 179, 115, 134, 191, 151, 31, 93, 68, 204, 190, 148, 22, 66,
    198, 172, 94, 71, 2, 6, 18, 54, 162, 64, 192, 154, 40, 120, 149, 25,
    75, 14, 42, 126, 167, 79, 26, 78, 23, 69, 207, 199, 175, 103, 98, 83,
    38, 114, 131, 182, 124, 161, 61, 183, 127, 170, 88, 53, 159, 55, 165,
    73, 8, 24, 72, 5, 15, 45, 135, 194, 160, 58, 174, 100, 89,
];

#[rustfmt::skip]
const FINDER_PATTERN_EXP: &[i32] = &[
    1, 8, 4, 1, 1, 1, 1, 4, 8, 1, 3, 6, 4, 1, 1, 1, 1, 4, 6, 3, 3, 4, 6, 1,
    1, 1, 1, 6, 4, 3, 3, 2, 8, 1, 1, 1, 1, 8, 2, 3, 2, 6, 5, 1, 1, 1, 1, 5,
    6, 2, 2, 2, 9, 1, 1, 1, 1, 9, 2, 2,
];

#[rustfmt::skip]
const FINDER_SEQUENCE: &[i32] = &[
    1, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 4, 3, 0, 0, 0, 0, 0, 0, 0, 0, 1, 6,
    3, 8, 0, 0, 0, 0, 0, 0, 0, 1, 10, 3, 8, 5, 0, 0, 0, 0, 0, 0, 1, 10, 3,
    8, 7, 12, 0, 0, 0, 0, 0, 1, 10, 3, 8, 9, 12, 11, 0, 0, 0, 0, 1, 2, 3,
    4, 5, 6, 7, 8, 0, 0, 0, 1, 2, 3, 4, 5, 6, 7, 10, 9, 0, 0, 1, 2, 3, 4,
    5, 6, 7, 10, 11, 12, 0, 1, 2, 3, 4, 5, 8, 7, 10, 9, 12, 11,
];

#[rustfmt::skip]
const WEIGHT_ROWS: &[i32] = &[
    0, 1, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 6,
    3, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 9, 10, 3, 4,
    13, 14, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 17, 18, 3, 4, 13,
    14, 7, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 17, 18, 3, 4, 13, 14,
    11, 12, 21, 22, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 17, 18, 3, 4, 13, 14,
    15, 16, 21, 22, 19, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 5, 6, 7,
    8, 9, 10, 11, 12, 13, 14, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 5, 6, 7, 8,
    9, 10, 11, 12, 17, 18, 15, 16, 0, 0, 0, 0, 0, 1, 2, 3, 4, 5, 6, 7, 8,
    9, 10, 11, 12, 17, 18, 19, 20, 21, 22, 0, 0, 0, 1, 2, 3, 4, 5, 6, 7, 8,
    13, 14, 11, 12, 17, 18, 15, 16, 21, 22, 19, 20,
];

fn iso_char_bits(c: u8) -> Option<&'static str> {
    match c {
        b'!' => Some("11101000"),
        b'"' => Some("11101001"),
        b'%' => Some("11101010"),
        b'&' => Some("11101011"),
        b'\'' => Some("11101100"),
        b'(' => Some("11101101"),
        b')' => Some("11101110"),
        b'*' => Some("11101111"),
        b'+' => Some("11110000"),
        b',' => Some("11110001"),
        b'-' => Some("11110010"),
        b'.' => Some("11110011"),
        b'/' => Some("11110100"),
        b':' => Some("11110101"),
        b';' => Some("11110110"),
        b'<' => Some("11110111"),
        b'=' => Some("11111000"),
        b'>' => Some("11111001"),
        b'?' => Some("11111010"),
        b'_' => Some("11111011"),
        b' ' => Some("11111100"),
        _ => None,
    }
}

fn int_to_bits(value: i32, num_bits: i32) -> String {
    let mut result = Vec::with_capacity(num_bits as usize);
    for j in 0..num_bits {
        if value & (1 << (num_bits - 1 - j)) != 0 {
            result.push(b'1');
        } else {
            result.push(b'0');
        }
    }
    String::from_utf8(result).unwrap()
}

fn calculate_remainder(binary_string_length: usize) -> usize {
    let mut remainder = 12 - (binary_string_length % 12);
    if remainder == 12 {
        remainder = 0;
    }
    if binary_string_length < 36 {
        remainder = 36 - binary_string_length;
    }
    remainder
}

fn bin2pat_from_black(binary: &str) -> String {
    if binary.is_empty() {
        return String::new();
    }
    let pattern = bin2pat(binary);
    if binary.as_bytes()[0] == b'0' {
        format!("0{}", pattern)
    } else {
        pattern
    }
}

/// GS1 DataBar Expanded barcode encoder.
pub struct GS1DataBarExpanded {
    pub base_1d: BarcodeBase1D,
    pub symbol_type: ExpandedSymbolType,
    pub num_columns: i32,
    human_readable: String,
    patterns: Vec<String>,
    row_heights: Vec<i32>,
    row_count: usize,
    // internal
    binary_string: String,
    general_field: String,
    general_field_type: Vec<i32>,
    enc_method_result: i32,
}

impl GS1DataBarExpanded {
    pub fn new(output_format: &str, symbol_type: ExpandedSymbolType, num_columns: i32) -> Self {
        let mut s = Self {
            base_1d: BarcodeBase1D::new(output_format),
            symbol_type,
            num_columns,
            human_readable: String::new(),
            patterns: Vec::new(),
            row_heights: Vec::new(),
            row_count: 0,
            binary_string: String::new(),
            general_field: String::new(),
            general_field_type: Vec::new(),
            enc_method_result: 0,
        };
        s.base_1d.show_text = false;
        s
    }

    pub fn human_readable(&self) -> &str { &self.human_readable }
    pub fn patterns(&self) -> &[String] { &self.patterns }
    pub fn row_heights(&self) -> &[i32] { &self.row_heights }
    pub fn row_count(&self) -> usize { self.row_count }

    pub fn draw(&mut self, code: &str, width: i32, height: i32) -> Result<()> {
        self.encode(code)?;
        if self.base_1d.base.is_svg_output() {
            return super::databar_render::draw_multi_row_svg(
                &mut self.base_1d.base, &self.patterns, &self.row_heights, width, height,
            );
        }
        #[cfg(any(feature = "png", feature = "jpeg"))]
        {
            super::databar_render::draw_multi_row_png(
                &mut self.base_1d.base, &self.patterns, &self.row_heights, width, height,
            )
        }
        #[cfg(not(any(feature = "png", feature = "jpeg")))]
        Err(BarcodeError::FormatError("PNG/JPEG features not enabled".into()))
    }

    pub fn encode(&mut self, code: &str) -> Result<Vec<i32>> {
        let content = code.replace("{AI}", "");
        if content.is_empty() {
            return Err(BarcodeError::EmptyCode);
        }

        self.binary_string = "0".to_string(); // Linkage flag
        self.calculate_binary_string(&content, code)?;

        let data_chars = self.binary_string.len() / 12;

        // Calculate vs values (12-bit segments)
        let mut vs = vec![0i32; data_chars];
        for (i, vs_val) in vs.iter_mut().enumerate().take(data_chars) {
            let mut v = 0;
            for j in 0..12 {
                if self.binary_string.as_bytes()[i * 12 + j] == b'1' {
                    v += 2048 >> j;
                }
            }
            *vs_val = v;
        }

        // Determine groups and calculate widths
        let mut group = vec![0i32; data_chars];
        let mut char_widths = vec![[0i32; 8]; data_chars];

        for i in 0..data_chars {
            group[i] = if vs[i] <= 347 {
                1
            } else if vs[i] <= 1387 {
                2
            } else if vs[i] <= 2947 {
                3
            } else if vs[i] <= 3987 {
                4
            } else {
                5
            };

            let gi = (group[i] - 1) as usize;
            let vo = (vs[i] - G_SUM_EXP[gi]) / T_EVEN_EXP[gi];
            let ve = (vs[i] - G_SUM_EXP[gi]) % T_EVEN_EXP[gi];

            let w = get_widths(vo, MODULES_ODD_EXP[gi], 4, WIDEST_ODD_EXP[gi], 0);
            char_widths[i][0] = w[0];
            char_widths[i][2] = w[1];
            char_widths[i][4] = w[2];
            char_widths[i][6] = w[3];

            let w = get_widths(ve, MODULES_EVEN_EXP[gi], 4, WIDEST_EVEN_EXP[gi], 1);
            char_widths[i][1] = w[0];
            char_widths[i][3] = w[1];
            char_widths[i][5] = w[2];
            char_widths[i][7] = w[3];
        }

        // Checksum (modulo 211)
        let mut checksum = 0i32;
        for i in 0..data_chars {
            let row = WEIGHT_ROWS[(((data_chars as i32 - 2) / 2) * 21 + i as i32) as usize];
            for j in 0..8 {
                checksum += char_widths[i][j] * CHECKSUM_WEIGHT_EXP[(row * 8 + j as i32) as usize];
            }
        }

        let check_char = 211 * ((data_chars as i32 + 1) - 4) + (checksum % 211);

        let cg = if check_char <= 347 {
            1
        } else if check_char <= 1387 {
            2
        } else if check_char <= 2947 {
            3
        } else if check_char <= 3987 {
            4
        } else {
            5
        };

        let c_odd = (check_char - G_SUM_EXP[cg - 1]) / T_EVEN_EXP[cg - 1];
        let c_even = (check_char - G_SUM_EXP[cg - 1]) % T_EVEN_EXP[cg - 1];

        let mut check_widths = [0i32; 8];
        let w = get_widths(c_odd, MODULES_ODD_EXP[cg - 1], 4, WIDEST_ODD_EXP[cg - 1], 0);
        check_widths[0] = w[0];
        check_widths[2] = w[1];
        check_widths[4] = w[2];
        check_widths[6] = w[3];
        let w = get_widths(c_even, MODULES_EVEN_EXP[cg - 1], 4, WIDEST_EVEN_EXP[cg - 1], 1);
        check_widths[1] = w[0];
        check_widths[3] = w[1];
        check_widths[5] = w[2];
        check_widths[7] = w[3];

        // Build elements array
        let dc1 = data_chars as i32 + 1;
        let pattern_width = (((dc1 / 2) + (dc1 & 1)) * 5 + dc1 * 8 + 4) as usize;
        let mut elements = vec![0i32; pattern_width];
        elements[0] = 1;
        elements[1] = 1;
        elements[pattern_width - 2] = 1;
        elements[pattern_width - 1] = 1;

        // Finder patterns
        let finder_count = (dc1 / 2 + (dc1 & 1)) as usize;
        for i in 0..finder_count {
            let k = ((((dc1 - 2) / 2 + (dc1 & 1)) - 1) * 11 + i as i32) as usize;
            for j in 0..5 {
                elements[21 * i + j + 10] =
                    FINDER_PATTERN_EXP[((FINDER_SEQUENCE[k] - 1) * 5 + j as i32) as usize];
            }
        }

        // Check character at position 2
        elements[2..10].copy_from_slice(&check_widths);

        // Odd-index data characters (forward)
        let mut i = 1;
        while i < data_chars {
            for j in 0..8 {
                elements[((i - 1) / 2) * 21 + 23 + j] = char_widths[i][j];
            }
            i += 2;
        }

        // Even-index data characters (reversed)
        i = 0;
        while i < data_chars {
            for j in 0..8 {
                elements[(i / 2) * 21 + 15 + j] = char_widths[i][7 - j];
            }
            i += 2;
        }

        // Generate patterns
        self.generate_patterns(&elements, pattern_width, data_chars as i32);

        Ok(elements)
    }

    fn calculate_binary_string(&mut self, source: &str, original_code: &str) -> Result<()> {
        let ai_tag = "{AI}";

        // Determine encoding method
        let mut encoding_method;
        if source.len() >= 16 && source.as_bytes()[0] == b'0' && source.as_bytes()[1] == b'1' {
            encoding_method = 1;
            self.human_readable = format!("(01){}", &original_code[2..]);
            self.replace_ai_tags(ai_tag);
        } else {
            encoding_method = 2;
            self.human_readable = original_code.to_string();
            self.replace_ai_tags(ai_tag);
        }

        // Check for compressed methods > 2
        if source.len() >= 20
            && encoding_method == 1
            && source.as_bytes()[2] == b'9'
            && source.as_bytes()[16] == b'3'
        {
            self.check_compressed_methods(source, encoding_method);
            encoding_method = self.enc_method_result;
        }

        // Method header and read position
        let read_posn = self.apply_method_header(encoding_method, source.len());

        // Verify compressed data is numeric
        for i in 0..read_posn {
            let c = source.as_bytes()[i];
            if !c.is_ascii_digit() && c != b'[' && c != b']' {
                return Err(BarcodeError::InvalidData(
                    "invalid characters in input data".into(),
                ));
            }
        }

        // Encode compressed data field
        self.encode_compressed(encoding_method, source);

        // General purpose data field
        self.general_field = source[read_posn..].to_string();
        self.general_field_type = vec![ENC_ISO; self.general_field.len()];

        let mut last_mode = ENC_NUMERIC;
        if !self.general_field.is_empty() {
            last_mode = self.encode_general_field(last_mode);
        }

        // Length check
        if self.binary_string.len() > 252 {
            return Err(BarcodeError::InvalidData("input too long".into()));
        }

        // Padding
        let remainder = calculate_remainder(self.binary_string.len());
        let mut padstring = String::new();
        let mut pad_i = remainder as i32;
        if !self.general_field.is_empty() && last_mode == ENC_NUMERIC {
            padstring.push_str("0000");
            pad_i = remainder as i32 - 4;
        }
        while pad_i > 0 {
            padstring.push_str("00100");
            pad_i -= 5;
        }
        if remainder > 0 && remainder <= padstring.len() {
            self.binary_string.push_str(&padstring[..remainder]);
        } else if remainder > 0 {
            self.binary_string.push_str(&padstring);
        }

        // Patch variable length symbol bit field
        let data_char_count = self.binary_string.len() / 12 + 1;
        let patch0 = if data_char_count & 1 != 0 { "1" } else { "0" };
        let patch1 = if self.binary_string.len() > 156 { "1" } else { "0" };
        let patch = format!("{}{}", patch0, patch1);

        let bs = self.binary_string.clone();
        match encoding_method {
            1 => {
                self.binary_string = format!("{}{}{}", &bs[..2], patch, &bs[4..]);
            }
            2 => {
                self.binary_string = format!("{}{}{}", &bs[..3], patch, &bs[5..]);
            }
            5 | 6 => {
                self.binary_string = format!("{}{}{}", &bs[..6], patch, &bs[8..]);
            }
            _ => {}
        }

        Ok(())
    }

    fn replace_ai_tags(&mut self, ai_tag: &str) {
        loop {
            let p = self.human_readable.find(ai_tag);
            match p {
                None => break,
                Some(p) => {
                    let ai_start = p + ai_tag.len();
                    if ai_start + 2 > self.human_readable.len() {
                        break;
                    }
                    let ai_code = self.human_readable[ai_start..ai_start + 2].to_string();
                    let rest = self.human_readable[ai_start + 2..].to_string();
                    self.human_readable =
                        format!("{}({}){}", &self.human_readable[..p], ai_code, rest);
                }
            }
        }
    }

    fn check_compressed_methods(&mut self, source: &str, base_method: i32) {
        self.enc_method_result = base_method;
        let s = source.as_bytes();

        if source.len() >= 26 && s[17] == b'1' && s[18] == b'0' {
            let mut weight = 0.0f64;
            for i in 0..6 {
                weight = weight * 10.0 + (s[20 + i] - b'0') as f64;
            }
            if weight < 99999.0 {
                if s[19] == b'3' && source.len() == 26 && weight / 1000.0 <= 32.767 {
                    self.enc_method_result = 3;
                    self.human_readable =
                        format!("(01){}(3103){}", &source[2..16], &source[20..]);
                }
                if source.len() == 34 {
                    self.set_weight_date_method(source, weight, "310", &[7, 9, 11, 13]);
                }
            }
        }

        if source.len() >= 26 && s[17] == b'2' && s[18] == b'0' {
            let mut weight = 0.0f64;
            for i in 0..6 {
                weight = weight * 10.0 + (s[20 + i] - b'0') as f64;
            }
            if weight < 99999.0 {
                if (s[19] == b'2' || s[19] == b'3') && source.len() == 26 {
                    if s[19] == b'3' {
                        if weight / 1000.0 <= 22.767 {
                            self.enc_method_result = 4;
                            self.human_readable =
                                format!("(01){}(3203){}", &source[2..16], &source[20..]);
                        }
                    } else if weight / 100.0 <= 99.99 {
                        self.enc_method_result = 4;
                        self.human_readable =
                            format!("(01){}(3202){}", &source[2..16], &source[20..]);
                    }
                }
                if source.len() == 34 {
                    self.set_weight_date_method(source, weight, "320", &[8, 10, 12, 14]);
                }
            }
        }

        if s[17] == b'9' {
            if s[18] == b'2' && s[19] >= b'0' && s[19] <= b'3' {
                self.enc_method_result = 5;
                self.human_readable = format!(
                    "(01){}(392{}){}",
                    &source[2..16],
                    s[19] as char,
                    &source[20..]
                );
            }
            if s[18] == b'3' && s[19] >= b'0' && s[19] <= b'3' {
                self.enc_method_result = 6;
                self.human_readable = format!(
                    "(01){}(393{}){}",
                    &source[2..16],
                    s[19] as char,
                    &source[20..]
                );
            }
        }
    }

    fn set_weight_date_method(
        &mut self,
        source: &str,
        _weight: f64,
        ai_prefix: &str,
        methods: &[i32],
    ) {
        if source.len() != 34 {
            return;
        }
        let s = source.as_bytes();
        if s[26] == b'1' {
            let date_ai_map: &[(u8, i32)] =
                &[(b'1', methods[0]), (b'3', methods[1]), (b'5', methods[2]), (b'7', methods[3])];
            let date_ais: &[(i32, &str)] = &[
                (methods[0], "11"),
                (methods[1], "13"),
                (methods[2], "15"),
                (methods[3], "17"),
            ];

            for &(ch, method) in date_ai_map {
                if s[27] == ch {
                    self.enc_method_result = method;
                    let date_ai = date_ais.iter().find(|&&(m, _)| m == method).map(|&(_, ai)| ai).unwrap_or("");
                    self.human_readable = format!(
                        "(01){}({}{}){}({}){}",
                        &source[2..16],
                        ai_prefix,
                        s[19] as char,
                        &source[20..26],
                        date_ai,
                        &source[28..]
                    );
                    break;
                }
            }
        }
    }

    fn apply_method_header(&mut self, encoding_method: i32, source_len: usize) -> usize {
        let (header, rp) = match encoding_method {
            1 => ("1XX", 16i32),
            2 => ("00XX", 0),
            3 => ("0100", -1),
            4 => ("0101", -1),
            5 => ("01100XX", 20),
            6 => ("01101XX", 23),
            7 => ("0111000", -1),
            8 => ("0111001", -1),
            9 => ("0111010", -1),
            10 => ("0111011", -1),
            11 => ("0111100", -1),
            12 => ("0111101", -1),
            13 => ("0111110", -1),
            14 => ("0111111", -1),
            _ => ("", 0),
        };
        self.binary_string.push_str(header);
        if rp < 0 {
            source_len
        } else {
            rp as usize
        }
    }

    fn encode_compressed(&mut self, encoding_method: i32, source: &str) {
        let s = source.as_bytes();
        match encoding_method {
            1 => {
                let group_val = (s[2] - b'0') as i32;
                self.binary_string.push_str(&int_to_bits(group_val, 4));
                for i in 1..=4 {
                    let gv = 100 * (s[i * 3] - b'0') as i32
                        + 10 * (s[i * 3 + 1] - b'0') as i32
                        + (s[i * 3 + 2] - b'0') as i32;
                    self.binary_string.push_str(&int_to_bits(gv, 10));
                }
            }
            3 | 4 => {
                for i in 1..=4 {
                    let gv = 100 * (s[i * 3] - b'0') as i32
                        + 10 * (s[i * 3 + 1] - b'0') as i32
                        + (s[i * 3 + 2] - b'0') as i32;
                    self.binary_string.push_str(&int_to_bits(gv, 10));
                }
                let mut gv = 0i32;
                for i in 0..6 {
                    gv = gv * 10 + (s[20 + i] - b'0') as i32;
                }
                if encoding_method == 4 && s[19] == b'3' {
                    gv += 10000;
                }
                self.binary_string.push_str(&int_to_bits(gv, 15));
            }
            7..=14 => {
                for i in 1..=4 {
                    let gv = 100 * (s[i * 3] - b'0') as i32
                        + 10 * (s[i * 3 + 1] - b'0') as i32
                        + (s[i * 3 + 2] - b'0') as i32;
                    self.binary_string.push_str(&int_to_bits(gv, 10));
                }
                let mut gv = (s[19] - b'0') as i32;
                for i in 0..5 {
                    gv = gv * 10 + (s[21 + i] - b'0') as i32;
                }
                self.binary_string.push_str(&int_to_bits(gv, 20));
                let gv2 = if source.len() == 34 {
                    ((10 * (s[28] - b'0') as i32 + (s[29] - b'0') as i32) * 384)
                        + ((10 * (s[30] - b'0') as i32 + (s[31] - b'0') as i32) - 1) * 32
                        + 10 * (s[32] - b'0') as i32
                        + (s[33] - b'0') as i32
                } else {
                    38400
                };
                self.binary_string.push_str(&int_to_bits(gv2, 16));
            }
            5 | 6 => {
                for i in 1..=4 {
                    let gv = 100 * (s[i * 3] - b'0') as i32
                        + 10 * (s[i * 3 + 1] - b'0') as i32
                        + (s[i * 3 + 2] - b'0') as i32;
                    self.binary_string.push_str(&int_to_bits(gv, 10));
                }
                let dec = match s[19] {
                    b'0' => "00",
                    b'1' => "01",
                    b'2' => "10",
                    b'3' => "11",
                    _ => "00",
                };
                self.binary_string.push_str(dec);
                if encoding_method == 6 {
                    let mut gv = 0i32;
                    for i in 0..3 {
                        gv = gv * 10 + (s[20 + i] - b'0') as i32;
                    }
                    self.binary_string.push_str(&int_to_bits(gv, 10));
                }
            }
            _ => {}
        }
    }

    #[allow(clippy::needless_range_loop)]
    fn encode_general_field(&mut self, mut last_mode: i32) -> i32 {
        let gf = self.general_field.clone();
        let gf_bytes = gf.as_bytes();
        let n = gf.len();

        // Classify each character
        for i in 0..n {
            let c = gf_bytes[i];
            if !(b' '..=b'z').contains(&c) {
                self.general_field_type[i] = ENC_INVALID;
            } else {
                self.general_field_type[i] = ENC_ISO;
            }
            if c == b'#' || c == b'$' || c == b'@' || c == b'\\' || c == b'^' || c == b'`' {
                self.general_field_type[i] = ENC_INVALID;
            }
            if c.is_ascii_uppercase() || c == b'*' || c == b',' || c == b'-' || c == b'.' || c == b'/' {
                self.general_field_type[i] = ENC_ALPHA_OR_ISO;
            }
            if c.is_ascii_digit() || c == b'[' {
                self.general_field_type[i] = ENC_ANY_ENC;
            }
        }

        // Propagate ISO/ALPHA_OR_ISO to following FNC1
        for i in 0..n.saturating_sub(1) {
            if self.general_field_type[i] == ENC_ISO && gf_bytes[i + 1] == b'[' {
                self.general_field_type[i + 1] = ENC_ISO;
            }
        }
        for i in 0..n.saturating_sub(1) {
            if self.general_field_type[i] == ENC_ALPHA_OR_ISO && gf_bytes[i + 1] == b'[' {
                self.general_field_type[i + 1] = ENC_ALPHA_OR_ISO;
            }
        }

        // Apply rules
        let latch = self.apply_general_field_rules();

        // Set initial mode
        if self.general_field_type[0] == ENC_ALPHA {
            self.binary_string.push_str("0000");
            last_mode = ENC_ALPHA;
        }
        if self.general_field_type[0] == ENC_ISO {
            self.binary_string.push_str("0000");
            self.binary_string.push_str("00100");
            last_mode = ENC_ISO;
        }

        // Encode characters
        let mut i = 0;
        loop {
            let mode = self.general_field_type[i];

            if mode == ENC_NUMERIC {
                if last_mode != ENC_NUMERIC {
                    self.binary_string.push_str("000");
                }
                let d1 = if gf_bytes[i] != b'[' {
                    (gf_bytes[i] - b'0') as i32
                } else {
                    10
                };
                let d2 = if gf_bytes[i + 1] != b'[' {
                    (gf_bytes[i + 1] - b'0') as i32
                } else {
                    10
                };
                let value = 11 * d1 + d2 + 8;
                self.binary_string.push_str(&int_to_bits(value, 7));
                i += 2;
                last_mode = ENC_NUMERIC;
            } else if mode == ENC_ALPHA {
                if i != 0 {
                    if last_mode == ENC_NUMERIC {
                        self.binary_string.push_str("0000");
                    }
                    if last_mode == ENC_ISO {
                        self.binary_string.push_str("00100");
                    }
                }
                let c = gf_bytes[i];
                if c.is_ascii_digit() {
                    self.binary_string.push_str(&int_to_bits(c as i32 - 43, 5));
                }
                if c.is_ascii_uppercase() {
                    self.binary_string.push_str(&int_to_bits(c as i32 - 33, 6));
                }
                last_mode = ENC_ALPHA;
                match c {
                    b'[' => {
                        self.binary_string.push_str("01111");
                        last_mode = ENC_NUMERIC;
                    }
                    b'*' => self.binary_string.push_str("111010"),
                    b',' => self.binary_string.push_str("111011"),
                    b'-' => self.binary_string.push_str("111100"),
                    b'.' => self.binary_string.push_str("111101"),
                    b'/' => self.binary_string.push_str("111110"),
                    _ => {}
                }
                i += 1;
            } else if mode == ENC_ISO {
                if i != 0 {
                    if last_mode == ENC_NUMERIC {
                        self.binary_string.push_str("0000");
                        self.binary_string.push_str("00100");
                    }
                    if last_mode == ENC_ALPHA {
                        self.binary_string.push_str("00100");
                    }
                }
                let c = gf_bytes[i];
                if c.is_ascii_digit() {
                    self.binary_string.push_str(&int_to_bits(c as i32 - 43, 5));
                }
                if c.is_ascii_uppercase() {
                    self.binary_string.push_str(&int_to_bits(c as i32 - 1, 7));
                }
                if c.is_ascii_lowercase() {
                    self.binary_string.push_str(&int_to_bits(c as i32 - 7, 7));
                }
                last_mode = ENC_ISO;
                if c == b'[' {
                    self.binary_string.push_str("01111");
                    last_mode = ENC_NUMERIC;
                }
                if let Some(v) = iso_char_bits(c) {
                    self.binary_string.push_str(v);
                }
                i += 1;
            } else {
                i += 1;
            }

            let mut current_length = i;
            if latch {
                current_length += 1;
            }
            if current_length >= n {
                break;
            }
        }

        // Handle single remaining digit (latch case)
        let remainder = calculate_remainder(self.binary_string.len());
        if latch {
            if last_mode == ENC_NUMERIC {
                if (4..=6).contains(&remainder) {
                    let value = (gf_bytes[i] - b'0') as i32 + 1;
                    self.binary_string.push_str(&int_to_bits(value, 4));
                } else {
                    let d1 = (gf_bytes[i] - b'0') as i32;
                    let d2 = 10;
                    let value = 11 * d1 + d2 + 8;
                    self.binary_string.push_str(&int_to_bits(value, 7));
                }
            } else {
                let value = gf_bytes[i] as i32 - 43;
                self.binary_string.push_str(&int_to_bits(value, 5));
            }
        }

        last_mode
    }

    fn apply_general_field_rules(&mut self) -> bool {
        let n = self.general_field.len();
        if n == 0 {
            return false;
        }

        // Build blocks
        let mut block_len = vec![1i32];
        let mut block_type = vec![self.general_field_type[0]];
        for i in 1..n {
            if self.general_field_type[i] == self.general_field_type[i - 1] {
                *block_len.last_mut().unwrap() += 1;
            } else {
                block_len.push(1);
                block_type.push(self.general_field_type[i]);
            }
        }
        let mut block_count = block_len.len();

        // Apply rules
        for i in 0..block_count {
            let cur = block_type[i];
            let nxt = if i < block_count - 1 { block_type[i + 1] } else { -1 };

            if cur == ENC_ISO && i != block_count - 1 {
                if nxt == ENC_ANY_ENC && block_len[i + 1] >= 4 {
                    block_type[i + 1] = ENC_NUMERIC;
                }
                if nxt == ENC_ANY_ENC && block_len[i + 1] < 4 {
                    block_type[i + 1] = ENC_ISO;
                }
                if nxt == ENC_ALPHA_OR_ISO && block_len[i + 1] >= 5 {
                    block_type[i + 1] = ENC_ALPHA;
                }
                if nxt == ENC_ALPHA_OR_ISO && block_len[i + 1] < 5 {
                    block_type[i + 1] = ENC_ISO;
                }
            }

            if cur == ENC_ALPHA_OR_ISO {
                block_type[i] = ENC_ALPHA;
            }

            if cur == ENC_ALPHA && i != block_count - 1 {
                if nxt == ENC_ANY_ENC && block_len[i + 1] >= 6 {
                    block_type[i + 1] = ENC_NUMERIC;
                }
                if nxt == ENC_ANY_ENC && block_len[i + 1] < 6 {
                    if i == block_count - 2 && block_len[i + 1] >= 4 {
                        block_type[i + 1] = ENC_NUMERIC;
                    } else {
                        block_type[i + 1] = ENC_ALPHA;
                    }
                }
            }

            if cur == ENC_ANY_ENC {
                block_type[i] = ENC_NUMERIC;
            }
        }

        // Merge adjacent same-type blocks
        if block_count > 1 {
            let mut i = 1;
            while i < block_len.len() {
                if block_type[i - 1] == block_type[i] {
                    block_len[i - 1] += block_len[i];
                    block_len.remove(i);
                    block_type.remove(i);
                    if i > 1 {
                        i -= 1;
                    }
                } else {
                    i += 1;
                }
            }
            block_count = block_len.len();
        }

        // Odd-size numeric blocks (not the last)
        for i in 0..block_count.saturating_sub(1) {
            if block_type[i] == ENC_NUMERIC && (block_len[i] & 1) != 0 {
                block_len[i] -= 1;
                block_len[i + 1] += 1;
            }
        }

        // Expand back to per-character types
        let mut j = 0;
        for i in 0..block_count {
            for _ in 0..block_len[i] {
                self.general_field_type[j] = block_type[i];
                j += 1;
            }
        }

        block_type[block_count - 1] == ENC_NUMERIC && (block_len[block_count - 1] & 1) != 0
    }

    fn generate_patterns(&mut self, elements: &[i32], pattern_width: usize, data_chars: i32) {
        if self.symbol_type == ExpandedSymbolType::Unstacked {
            self.row_count = 1;
            self.row_heights = vec![-1];
            let mut pattern = "0".to_string();
            for &elem in elements.iter().take(pattern_width) {
                pattern.push((elem as u8 + b'0') as char);
            }
            self.patterns = vec![pattern];
        } else {
            self.generate_stacked(elements, pattern_width, data_chars);
        }
    }

    #[allow(clippy::too_many_arguments)]
    #[allow(clippy::needless_range_loop)]
    fn generate_stacked(&mut self, elements: &[i32], pattern_width: usize, data_chars: i32) {
        let dc1 = data_chars + 1;
        let codeblocks = (dc1 / 2 + dc1 % 2) as usize;
        let mut blocks_per_row = self.num_columns as usize;
        if !(1..=10).contains(&blocks_per_row) {
            blocks_per_row = 2;
        }

        let stack_rows = codeblocks.div_ceil(blocks_per_row);
        let row_count = stack_rows * 4 - 3;
        self.row_heights = vec![0i32; row_count];
        self.patterns = vec![String::new(); row_count];
        let mut symbol_row = 0;
        let mut current_block = 0;

        for current_row in 1..=stack_rows {
            let mut sub_elements = vec![0i32; 235];
            let mut special_case_row = false;

            sub_elements[0] = 1;
            sub_elements[1] = 1;
            let mut elements_in_sub = 2;

            let mut reader = 0;
            let mut left_to_right;

            loop {
                if (blocks_per_row & 1) != 0
                    || (current_row & 1) != 0
                    || (current_row == stack_rows
                        && codeblocks != current_row * blocks_per_row
                        && (((current_row * blocks_per_row) - codeblocks) & 1) != 0)
                {
                    left_to_right = true;
                    let base_i = 2 + current_block * 21;
                    for j in 0..21 {
                        if base_i + j < pattern_width {
                            sub_elements[j + reader * 21 + 2] = elements[base_i + j];
                            elements_in_sub += 1;
                        }
                    }
                } else {
                    left_to_right = false;
                    if current_row * blocks_per_row < codeblocks {
                        let base_i = 2 + ((current_row * blocks_per_row) - reader - 1) * 21;
                        for j in 0..21 {
                            if base_i + j < pattern_width {
                                sub_elements[(20 - j) + reader * 21 + 2] = elements[base_i + j];
                                elements_in_sub += 1;
                            }
                        }
                    } else {
                        let k_off = (current_row * blocks_per_row) - codeblocks;
                        let l_off = (current_row * blocks_per_row) - reader - 1;
                        let base_i = 2 + (l_off - k_off) * 21;
                        for j in 0..21 {
                            if base_i + j < pattern_width {
                                sub_elements[(20 - j) + reader * 21 + 2] = elements[base_i + j];
                                elements_in_sub += 1;
                            }
                        }
                    }
                }

                reader += 1;
                current_block += 1;
                if reader >= blocks_per_row || current_block >= codeblocks {
                    break;
                }
            }

            // Row stop
            sub_elements[elements_in_sub] = 1;
            sub_elements[elements_in_sub + 1] = 1;
            elements_in_sub += 2;

            // Build pattern
            self.row_heights[symbol_row] = -1;
            let mut black = true;

            if (current_row & 1) != 0 {
                self.patterns[symbol_row] = "0".to_string();
                black = false;
            } else if current_row == stack_rows
                && codeblocks != current_row * blocks_per_row
                && (((current_row * blocks_per_row) - codeblocks) & 1) != 0
            {
                special_case_row = true;
                sub_elements[0] = 2;
                self.patterns[symbol_row] = "0".to_string();
                black = false;
            } else {
                self.patterns[symbol_row] = String::new();
            }

            // Build separator binary and pattern string
            let mut writer = 0;
            let mut sep_bin = String::new();
            for i in 0..elements_in_sub {
                self.patterns[symbol_row]
                    .push((sub_elements[i] as u8 + b'0') as char);
                for _ in 0..sub_elements[i] {
                    if black {
                        sep_bin.push('0');
                    } else {
                        sep_bin.push('1');
                    }
                }
                black = !black;
                writer += sub_elements[i] as usize;
            }

            // Fix separator binary
            if writer > 4 {
                let mut sep_bytes: Vec<u8> = sep_bin.into_bytes();
                sep_bytes[0] = b'0';
                sep_bytes[1] = b'0';
                sep_bytes[2] = b'0';
                sep_bytes[3] = b'0';
                if sep_bytes.len() > writer {
                    sep_bytes.truncate(writer);
                }
                sep_bin = String::from_utf8(sep_bytes).unwrap();
            }

            for j in 0..reader {
                let k = if special_case_row {
                    49 * j + 19
                } else {
                    49 * j + 18
                };
                let mut sep_bytes: Vec<u8> = sep_bin.into_bytes();
                if left_to_right {
                    for ii in 0..15 {
                        let idx = ii + k;
                        if idx >= 1
                            && idx < sep_bytes.len()
                            && sep_bytes[idx - 1] == b'1'
                            && sep_bytes[idx] == b'1'
                        {
                            sep_bytes[idx] = b'0';
                        }
                    }
                } else {
                    for ii in (0..15).rev() {
                        let idx = ii + k;
                        if idx + 1 < sep_bytes.len()
                            && sep_bytes[idx + 1] == b'1'
                            && sep_bytes[idx] == b'1'
                        {
                            sep_bytes[idx] = b'0';
                        }
                    }
                }
                sep_bin = String::from_utf8(sep_bytes).unwrap();
            }

            let sep_pattern = bin2pat_from_black(&sep_bin);

            // Separator rows
            if current_row != 1 {
                let mut mid_sep = "05".to_string();
                let mut cnt = 5;
                while cnt < 49 * blocks_per_row {
                    mid_sep.push_str("11");
                    cnt += 2;
                }
                self.patterns[symbol_row - 2] = mid_sep;
                self.row_heights[symbol_row - 2] = 1;
                self.patterns[symbol_row - 1] = sep_pattern.clone();
                self.row_heights[symbol_row - 1] = 1;
            }

            if current_row != stack_rows {
                self.patterns[symbol_row + 1] = sep_pattern;
                self.row_heights[symbol_row + 1] = 1;
            }

            symbol_row += 4;
        }

        self.row_count = row_count;
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_encode_basic() {
        let mut g = GS1DataBarExpanded::new("svg", ExpandedSymbolType::Unstacked, 2);
        let elements = g.encode("{FNC1}0104912345123459").unwrap();
        assert!(!elements.is_empty());
        assert_eq!(g.row_count(), 1);
    }

    #[test]
    fn test_encode_empty() {
        let mut g = GS1DataBarExpanded::new("svg", ExpandedSymbolType::Unstacked, 2);
        assert!(g.encode("").is_err());
    }

    #[test]
    fn test_encode_stacked() {
        let mut g = GS1DataBarExpanded::new("svg", ExpandedSymbolType::Stacked, 2);
        let elements = g.encode("{FNC1}0104912345123459").unwrap();
        assert!(!elements.is_empty());
        assert!(g.row_count() > 1);
    }
}
