use crate::base_1d::BarcodeBase1D;
use crate::error::{BarcodeError, Result};
use super::rss_utils::{get_widths, calculate_gtin_check_digit, bin2pat, pad_left, parse_int64};

/// GS1 DataBar 14 symbol type.
#[derive(Clone, Copy, PartialEq)]
pub enum SymbolType14 {
    Omnidirectional = 0,
    Stacked = 1,
    StackedOmnidirectional = 2,
}

// ISO 24724 Table 3, 4
const G_SUM_TABLE_14: &[i32] = &[0, 161, 961, 2015, 2715, 0, 336, 1036, 1516];
const T_TABLE_14: &[i32] = &[1, 10, 34, 70, 126, 4, 20, 48, 81];
const MODULES_ODD_14: &[i32] = &[12, 10, 8, 6, 4, 5, 7, 9, 11];
const MODULES_EVEN_14: &[i32] = &[4, 6, 8, 10, 12, 10, 8, 6, 4];
const WIDEST_ODD_14: &[i32] = &[8, 6, 4, 3, 1, 2, 4, 6, 8];
const WIDEST_EVEN_14: &[i32] = &[1, 3, 5, 6, 8, 7, 5, 3, 1];

const CHECKSUM_WEIGHT_14: &[i32] = &[
    1, 3, 9, 27, 2, 6, 18, 54,
    4, 12, 36, 29, 8, 24, 72, 58,
    16, 48, 65, 37, 32, 17, 51, 74,
    64, 34, 23, 69, 49, 68, 46, 59,
];

const FINDER_PATTERN_14: &[i32] = &[
    3, 8, 2, 1, 1,
    3, 5, 5, 1, 1,
    3, 3, 7, 1, 1,
    3, 1, 9, 1, 1,
    2, 7, 4, 1, 1,
    2, 5, 6, 1, 1,
    2, 3, 8, 1, 1,
    1, 5, 7, 1, 1,
    1, 3, 9, 1, 1,
];

/// GS1 DataBar 14 (RSS-14) barcode encoder.
pub struct GS1DataBar14 {
    pub base_1d: BarcodeBase1D,
    pub symbol_type: SymbolType14,
    gtin14: String,
    patterns: Vec<String>,
    row_heights: Vec<i32>,
    row_count: usize,
}

impl GS1DataBar14 {
    pub fn new(output_format: &str, symbol_type: SymbolType14) -> Self {
        let mut s = Self {
            base_1d: BarcodeBase1D::new(output_format),
            symbol_type,
            gtin14: String::new(),
            patterns: Vec::new(),
            row_heights: Vec::new(),
            row_count: 0,
        };
        s.base_1d.show_text = false;
        s
    }

    pub fn gtin14(&self) -> &str { &self.gtin14 }
    pub fn human_readable(&self) -> String { format!("(01){}", self.gtin14) }
    pub fn patterns(&self) -> &[String] { &self.patterns }
    pub fn row_heights(&self) -> &[i32] { &self.row_heights }
    pub fn row_count(&self) -> usize { self.row_count }

    pub fn draw(&mut self, code: &str, width: i32, height: i32) -> Result<()> {
        self.encode(code)?;
        if self.base_1d.base.is_svg_output() {
            return super::databar_render::draw_multi_row_svg(
                &mut self.base_1d.base, &self.patterns, &self.row_heights, width, height,
            );
        }
        #[cfg(any(feature = "png", feature = "jpeg"))]
        {
            super::databar_render::draw_multi_row_png(
                &mut self.base_1d.base, &self.patterns, &self.row_heights, width, height,
            )
        }
        #[cfg(not(any(feature = "png", feature = "jpeg")))]
        Err(BarcodeError::FormatError("PNG/JPEG features not enabled".into()))
    }

    pub fn encode(&mut self, code: &str) -> Result<Vec<i32>> {
        for c in code.chars() {
            if !c.is_ascii_digit() {
                return Err(BarcodeError::InvalidData(
                    "GS1 DataBar requires numeric digits only".into(),
                ));
            }
        }
        if code.len() > 13 {
            return Err(BarcodeError::InvalidData("input too long (max 13 digits)".into()));
        }

        let padded = pad_left(code, 13);
        let accum = parse_int64(&padded);

        let left_reg = (accum / 4537077) as i32;
        let right_reg = (accum % 4537077) as i32;

        let data_chars = [
            left_reg / 1597,
            left_reg % 1597,
            right_reg / 1597,
            right_reg % 1597,
        ];

        // Data groups
        let mut data_group = [0i32; 4];
        for &idx in &[0, 2] {
            data_group[idx] = 0;
            if data_chars[idx] >= 161 { data_group[idx] = 1; }
            if data_chars[idx] >= 961 { data_group[idx] = 2; }
            if data_chars[idx] >= 2015 { data_group[idx] = 3; }
            if data_chars[idx] >= 2715 { data_group[idx] = 4; }
        }
        for &idx in &[1, 3] {
            data_group[idx] = 5;
            if data_chars[idx] >= 336 { data_group[idx] = 6; }
            if data_chars[idx] >= 1036 { data_group[idx] = 7; }
            if data_chars[idx] >= 1516 { data_group[idx] = 8; }
        }

        // Calculate odd/even values
        let mut v_odd = [0i32; 4];
        let mut v_even = [0i32; 4];
        v_odd[0] = (data_chars[0] - G_SUM_TABLE_14[data_group[0] as usize]) / T_TABLE_14[data_group[0] as usize];
        v_even[0] = (data_chars[0] - G_SUM_TABLE_14[data_group[0] as usize]) % T_TABLE_14[data_group[0] as usize];
        v_odd[1] = (data_chars[1] - G_SUM_TABLE_14[data_group[1] as usize]) % T_TABLE_14[data_group[1] as usize];
        v_even[1] = (data_chars[1] - G_SUM_TABLE_14[data_group[1] as usize]) / T_TABLE_14[data_group[1] as usize];
        v_odd[3] = (data_chars[3] - G_SUM_TABLE_14[data_group[3] as usize]) % T_TABLE_14[data_group[3] as usize];
        v_even[3] = (data_chars[3] - G_SUM_TABLE_14[data_group[3] as usize]) / T_TABLE_14[data_group[3] as usize];
        v_odd[2] = (data_chars[2] - G_SUM_TABLE_14[data_group[2] as usize]) / T_TABLE_14[data_group[2] as usize];
        v_even[2] = (data_chars[2] - G_SUM_TABLE_14[data_group[2] as usize]) % T_TABLE_14[data_group[2] as usize];

        // Width patterns
        let mut data_widths = [[0i32; 4]; 8];
        for i in 0..4 {
            let g = data_group[i] as usize;
            if i == 0 || i == 2 {
                let w = get_widths(v_odd[i], MODULES_ODD_14[g], 4, WIDEST_ODD_14[g], 1);
                data_widths[0][i] = w[0]; data_widths[2][i] = w[1];
                data_widths[4][i] = w[2]; data_widths[6][i] = w[3];
                let w = get_widths(v_even[i], MODULES_EVEN_14[g], 4, WIDEST_EVEN_14[g], 0);
                data_widths[1][i] = w[0]; data_widths[3][i] = w[1];
                data_widths[5][i] = w[2]; data_widths[7][i] = w[3];
            } else {
                let w = get_widths(v_odd[i], MODULES_ODD_14[g], 4, WIDEST_ODD_14[g], 0);
                data_widths[0][i] = w[0]; data_widths[2][i] = w[1];
                data_widths[4][i] = w[2]; data_widths[6][i] = w[3];
                let w = get_widths(v_even[i], MODULES_EVEN_14[g], 4, WIDEST_EVEN_14[g], 1);
                data_widths[1][i] = w[0]; data_widths[3][i] = w[1];
                data_widths[5][i] = w[2]; data_widths[7][i] = w[3];
            }
        }

        // Checksum (modulo 79)
        let mut checksum = 0;
        for i in 0..8 {
            checksum += CHECKSUM_WEIGHT_14[i] * data_widths[i][0];
            checksum += CHECKSUM_WEIGHT_14[i + 8] * data_widths[i][1];
            checksum += CHECKSUM_WEIGHT_14[i + 16] * data_widths[i][2];
            checksum += CHECKSUM_WEIGHT_14[i + 24] * data_widths[i][3];
        }
        checksum %= 79;

        if checksum >= 8 { checksum += 1; }
        if checksum >= 72 { checksum += 1; }
        let c_left = checksum / 9;
        let c_right = checksum % 9;

        // Build total widths (46 elements)
        let mut total_widths = vec![0i32; 46];
        total_widths[0] = 1;
        total_widths[1] = 1;
        total_widths[44] = 1;
        total_widths[45] = 1;

        for i in 0..8 {
            total_widths[i + 2] = data_widths[i][0];
            total_widths[i + 15] = data_widths[7 - i][1];
            total_widths[i + 23] = data_widths[i][3];
            total_widths[i + 36] = data_widths[7 - i][2];
        }

        for i in 0..5 {
            total_widths[i + 10] = FINDER_PATTERN_14[i + 5 * c_left as usize];
            total_widths[i + 31] = FINDER_PATTERN_14[(4 - i) + 5 * c_right as usize];
        }

        let cd = calculate_gtin_check_digit(&padded);
        self.gtin14 = format!("{}{}", padded, cd);

        self.generate_grid(&total_widths);

        Ok(total_widths)
    }

    #[allow(clippy::needless_range_loop)]
    fn generate_grid(&mut self, total_widths: &[i32]) {
        let mut grid = [[false; 100]; 5];
        let mut row_count = 0;
        let mut symbol_width = 0;

        match self.symbol_type {
            SymbolType14::Omnidirectional => {
                let mut writer = 0;
                let mut latch = false;
                for i in 0..46 {
                    for _ in 0..total_widths[i] {
                        if latch {
                            grid[row_count][writer] = true;
                        }
                        writer += 1;
                    }
                    latch = !latch;
                }
                if writer > symbol_width {
                    symbol_width = writer;
                }
                row_count += 1;
            }
            SymbolType14::Stacked => {
                // Upper row (row 0)
                let mut writer = 0;
                let mut latch = false;
                for i in 0..23 {
                    for _ in 0..total_widths[i] {
                        grid[row_count][writer] = latch;
                        writer += 1;
                    }
                    latch = !latch;
                }
                grid[row_count][writer] = true;
                grid[row_count][writer + 1] = false;

                // Lower row (row 2)
                row_count += 2;
                grid[row_count][0] = true;
                grid[row_count][1] = false;
                writer = 0;
                latch = true;
                for i in 23..46 {
                    for _ in 0..total_widths[i] {
                        grid[row_count][writer + 2] = latch;
                        writer += 1;
                    }
                    latch = !latch;
                }

                // Separator (row 1)
                for i in 4..46 {
                    if grid[row_count - 2][i] == grid[row_count][i] {
                        if !grid[row_count - 2][i] {
                            grid[row_count - 1][i] = true;
                        }
                    } else if !grid[row_count - 1][i - 1] {
                        grid[row_count - 1][i] = true;
                    }
                }

                row_count += 1;
                if 50 > symbol_width {
                    symbol_width = 50;
                }
            }
            SymbolType14::StackedOmnidirectional => {
                // Upper row (row 0)
                let mut writer = 0;
                let mut latch = false;
                for i in 0..23 {
                    for _ in 0..total_widths[i] {
                        grid[row_count][writer] = latch;
                        writer += 1;
                    }
                    latch = !latch;
                }
                grid[row_count][writer] = true;
                grid[row_count][writer + 1] = false;

                // Lower row (row 4)
                row_count += 4;
                grid[row_count][0] = true;
                grid[row_count][1] = false;
                writer = 0;
                latch = true;
                for i in 23..46 {
                    for _ in 0..total_widths[i] {
                        grid[row_count][writer + 2] = latch;
                        writer += 1;
                    }
                    latch = !latch;
                }

                // Middle separator (row 2)
                let mut i = 5;
                while i < 46 {
                    grid[row_count - 2][i] = true;
                    i += 2;
                }

                // Upper separator (row 1)
                for i in 4..46 {
                    if !grid[row_count - 4][i] {
                        grid[row_count - 3][i] = true;
                    }
                }
                let mut latch_val = true;
                for i in 17..33 {
                    if !grid[row_count - 4][i] {
                        if latch_val {
                            grid[row_count - 3][i] = true;
                            latch_val = false;
                        } else {
                            grid[row_count - 3][i] = false;
                            latch_val = true;
                        }
                    } else {
                        grid[row_count - 3][i] = false;
                        latch_val = true;
                    }
                }

                // Lower separator (row 3)
                for i in 4..46 {
                    if !grid[row_count][i] {
                        grid[row_count - 1][i] = true;
                    }
                }
                latch_val = true;
                for i in 16..32 {
                    if !grid[row_count][i] {
                        if latch_val {
                            grid[row_count - 1][i] = true;
                            latch_val = false;
                        } else {
                            grid[row_count - 1][i] = false;
                            latch_val = true;
                        }
                    } else {
                        grid[row_count - 1][i] = false;
                        latch_val = true;
                    }
                }

                if 50 > symbol_width {
                    symbol_width = 50;
                }
                row_count += 1;
            }
        }

        // Convert grid to pattern strings
        self.patterns.clear();
        self.row_heights.clear();

        for i in 0..row_count {
            let mut binary = String::new();
            for j in 0..symbol_width {
                binary.push(if grid[i][j] { '1' } else { '0' });
            }

            let mut pattern = bin2pat(&binary);
            if !binary.is_empty() && binary.as_bytes()[0] == b'0' {
                pattern = format!("0{}", pattern);
            }
            self.patterns.push(pattern);
        }

        // Row heights
        match self.symbol_type {
            SymbolType14::Omnidirectional => {
                self.row_heights = vec![-1];
            }
            SymbolType14::Stacked => {
                self.row_heights = vec![5, 1, 7];
            }
            SymbolType14::StackedOmnidirectional => {
                self.row_heights = vec![-1, 1, 1, 1, -1];
            }
        }

        self.row_count = row_count;
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_encode_omnidirectional() {
        let mut g = GS1DataBar14::new("svg", SymbolType14::Omnidirectional);
        let widths = g.encode("0000000000000").unwrap();
        assert_eq!(widths.len(), 46);
        assert!(!g.patterns().is_empty());
    }

    #[test]
    fn test_encode_stacked() {
        let mut g = GS1DataBar14::new("svg", SymbolType14::Stacked);
        let widths = g.encode("12345678").unwrap();
        assert_eq!(widths.len(), 46);
        assert_eq!(g.row_count(), 3);
    }

    #[test]
    fn test_encode_stacked_omni() {
        let mut g = GS1DataBar14::new("svg", SymbolType14::StackedOmnidirectional);
        let widths = g.encode("12345678").unwrap();
        assert_eq!(widths.len(), 46);
        assert_eq!(g.row_count(), 5);
    }

    #[test]
    fn test_gtin14() {
        let mut g = GS1DataBar14::new("svg", SymbolType14::Omnidirectional);
        g.encode("0000000000000").unwrap();
        assert_eq!(g.gtin14().len(), 14);
    }

    #[test]
    fn test_encode_non_numeric() {
        let mut g = GS1DataBar14::new("svg", SymbolType14::Omnidirectional);
        assert!(g.encode("ABC").is_err());
    }
}
