use crate::base_1d::BarcodeBase1D;
use crate::error::{BarcodeError, Result};

use super::code128::CODE128_PATTERNS;

const GS1128_START: &[&str] = &["211412", "211214", "211232"];
const GS1128_STOP: &str = "2331112";
const GS1128_ONLY_B: &str = "`abcdefghijklmnopqrstuvwxyz{|}~";

fn gs1128_code_a() -> String {
    let mut a = String::new();
    for i in 32u8..=95 {
        a.push(i as char);
    }
    for i in 0u8..32 {
        a.push(i as char);
    }
    a
}

fn gs1128_code_b() -> String {
    let mut b = String::new();
    for i in 32u8..=126 {
        b.push(i as char);
    }
    b.push(0x7f as char);
    b
}

fn gs1128_code_c() -> Vec<String> {
    (0..100).map(|i| format!("{:02}", i)).collect()
}

fn gs1128_esc_chars() -> String {
    let mut e = String::new();
    for i in 0u8..32 {
        e.push(i as char);
    }
    e
}

fn gs1128_is_int4(s: &[u8], pos: usize) -> bool {
    if pos + 4 > s.len() {
        return false;
    }
    s[pos..pos + 4].iter().all(|&c| c.is_ascii_digit())
}

fn gs1128_get_abc(code: &[u8]) -> i32 {
    for &c in code {
        if c < 32 {
            return 0; // CODE-A
        }
        if GS1128_ONLY_B.contains(c as char) {
            return 1; // CODE-B
        }
    }
    1 // default B
}

fn gs1128_safe_sub(s: &[u8], pos: usize, length: usize) -> &[u8] {
    if pos >= s.len() {
        return &[];
    }
    let end = (pos + length).min(s.len());
    &s[pos..end]
}

/// GS1-128 barcode encoder.
pub struct GS1128 {
    pub base_1d: BarcodeBase1D,
}

impl GS1128 {
    pub fn new(output_format: &str) -> Self {
        let mut s = Self {
            base_1d: BarcodeBase1D::new(output_format),
        };
        s.base_1d.base.fit_width = true;
        s
    }

    pub fn draw(&mut self, code: &str, width: i32, height: i32) -> Result<()> {
        let pattern = encode_gs1128(code)?;
        let display = if self.base_1d.show_text {
            code.replace("{FNC1}", "").replace("{AI}", "")
        } else {
            String::new()
        };
        if self.base_1d.base.is_svg_output() {
            self.base_1d.draw_svg_bars(&pattern, width, height, &display)
        } else {
            #[cfg(any(feature = "png", feature = "jpeg"))]
            {
                self.base_1d.render_bars_to_png(&pattern, width, height, &display)
            }
            #[cfg(not(any(feature = "png", feature = "jpeg")))]
            {
                Err(BarcodeError::FormatError(
                    "PNG/JPEG support requires 'png' or 'jpeg' feature".into(),
                ))
            }
        }
    }
}

pub fn encode_gs1128(code: &str) -> Result<Vec<i32>> {
    if code.is_empty() {
        return Err(BarcodeError::EmptyCode);
    }

    let bytes = code.as_bytes();
    let code_a = gs1128_code_a();
    let code_b = gs1128_code_b();
    let code_c = gs1128_code_c();
    let esc_chars = gs1128_esc_chars();

    let sw_abc = determine_start_set(bytes, &code_c);

    let mut all_pattern = Vec::new();

    // Start code
    let ptn = GS1128_START[sw_abc as usize];
    for ch in ptn.bytes() {
        all_pattern.push((ch - b'0') as i32);
    }

    // Data encoding
    let mut sw_abc2 = sw_abc;
    let mut i = 0;
    while i < bytes.len() {
        let mut chg_flg = false;

        if sw_abc2 == 0 {
            // CODE-A
            if gs1128_is_int4(bytes, i) {
                sw_abc2 = 2;
                chg_flg = true;
            }
            if i < bytes.len() && GS1128_ONLY_B.contains(bytes[i] as char) {
                sw_abc2 = 1;
                chg_flg = true;
            }
        } else if sw_abc2 == 1 {
            // CODE-B
            if gs1128_is_int4(bytes, i) {
                sw_abc2 = 2;
                chg_flg = true;
            }
            if i < bytes.len() && esc_chars.contains(bytes[i] as char) {
                sw_abc2 = 0;
                chg_flg = true;
            }
        } else {
            // CODE-C
            let c2 = gs1128_safe_sub(bytes, i, 2);
            if c2.len() < 2 {
                sw_abc2 = gs1128_get_abc(&bytes[i..]);
                chg_flg = true;
            } else if c2 != b"{F" && c2 != b"{A" {
                let c2_str = std::str::from_utf8(c2).unwrap_or("");
                let found = code_c.iter().any(|cc| cc == c2_str);
                if !found {
                    sw_abc2 = gs1128_get_abc(&bytes[i..]);
                    chg_flg = true;
                }
            }
        }

        if chg_flg {
            let sw_code = match sw_abc2 {
                0 => 101,
                1 => 100,
                _ => 99,
            };
            let ptn = CODE128_PATTERNS[sw_code as usize];
            for ch in ptn.bytes() {
                all_pattern.push((ch - b'0') as i32);
            }
        }

        // Encode character
        let idx;
        if i + 6 <= bytes.len() && &bytes[i..i + 6] == b"{FNC1}" {
            idx = 102;
            i += 6;
        } else if i + 4 <= bytes.len() && &bytes[i..i + 4] == b"{AI}" {
            i += 4;
            continue;
        } else if sw_abc2 == 0 {
            // CODE-A
            let pos = code_a.find(bytes[i] as char).unwrap_or(0);
            idx = pos as i32;
            i += 1;
        } else if sw_abc2 == 1 {
            // CODE-B
            let pos = code_b.find(bytes[i] as char).unwrap_or(0);
            idx = pos as i32;
            i += 1;
        } else {
            // CODE-C
            let c2 = gs1128_safe_sub(bytes, i, 2);
            if c2.len() < 2 {
                let pos = code_b.find(bytes[i] as char).unwrap_or(0);
                idx = pos as i32;
                i += 1;
            } else {
                let c2_str = std::str::from_utf8(c2).unwrap_or("");
                idx = code_c
                    .iter()
                    .position(|cc| cc == c2_str)
                    .unwrap_or(0) as i32;
                i += 2;
            }
        }

        let ptn = CODE128_PATTERNS[idx as usize];
        for ch in ptn.bytes() {
            all_pattern.push((ch - b'0') as i32);
        }
    }

    // Check digit
    let cd_idx = calc_check_digit(bytes, &code_a, &code_b, &code_c, &esc_chars);
    let ptn = CODE128_PATTERNS[cd_idx as usize];
    for ch in ptn.bytes() {
        all_pattern.push((ch - b'0') as i32);
    }

    // Stop code
    for ch in GS1128_STOP.bytes() {
        all_pattern.push((ch - b'0') as i32);
    }

    Ok(all_pattern)
}

fn determine_start_set(bytes: &[u8], _code_c: &[String]) -> i32 {
    let mut cnt_num = 0usize;
    let mut cnt_f = 0usize;
    let mut cnt_ai = 0usize;

    while cnt_num + cnt_f + cnt_ai < bytes.len().saturating_sub(1) {
        let pos = cnt_num + cnt_f + cnt_ai;
        if pos + 6 <= bytes.len() && &bytes[pos..pos + 6] == b"{FNC1}" {
            cnt_f += 6;
            continue;
        }
        if pos + 4 <= bytes.len() && &bytes[pos..pos + 4] == b"{AI}" {
            cnt_ai += 4;
            continue;
        }
        if pos >= bytes.len() {
            break;
        }
        if bytes[pos] < b'0' || bytes[pos] > b'9' {
            break;
        }
        cnt_num += 1;
    }

    if cnt_num < 3 {
        gs1128_get_abc(bytes)
    } else {
        2 // CODE-C
    }
}

fn calc_check_digit(
    bytes: &[u8],
    code_a: &str,
    code_b: &str,
    code_c: &[String],
    esc_chars: &str,
) -> i32 {
    let mut cnt_num = 0usize;
    let mut cnt_f = 0usize;
    let mut cnt_ai = 0usize;

    while cnt_num + cnt_f + cnt_ai < bytes.len().saturating_sub(1) {
        let pos = cnt_num + cnt_f + cnt_ai;
        if pos + 6 <= bytes.len() && &bytes[pos..pos + 6] == b"{FNC1}" {
            cnt_f += 6;
            continue;
        }
        if pos + 4 <= bytes.len() && &bytes[pos..pos + 4] == b"{AI}" {
            cnt_ai += 4;
            continue;
        }
        if pos >= bytes.len() {
            break;
        }
        if bytes[pos] < b'0' || bytes[pos] > b'9' {
            break;
        }
        cnt_num += 1;
    }

    let mut sw_abc;
    let mut char_sum: i32;
    if cnt_num < 3 {
        sw_abc = gs1128_get_abc(bytes);
        char_sum = if sw_abc == 1 { 104 } else { 103 };
    } else {
        sw_abc = 2;
        char_sum = 105;
    }

    let mut cnt = 0i32;
    let mut i = 0;
    while i < bytes.len() {
        if sw_abc == 0 {
            if gs1128_is_int4(bytes, i) {
                sw_abc = 2;
                cnt += 1;
                char_sum += 99 * cnt;
            }
            if i < bytes.len() && GS1128_ONLY_B.contains(bytes[i] as char) {
                sw_abc = 1;
                cnt += 1;
                char_sum += 100 * cnt;
            }
        } else if sw_abc == 1 {
            if gs1128_is_int4(bytes, i) {
                sw_abc = 2;
                cnt += 1;
                char_sum += 99 * cnt;
            }
            if i < bytes.len() && esc_chars.contains(bytes[i] as char) {
                sw_abc = 0;
                cnt += 1;
                char_sum += 101 * cnt;
            }
        } else {
            let c2 = gs1128_safe_sub(bytes, i, 2);
            if c2.len() < 2 {
                sw_abc = gs1128_get_abc(&bytes[i..]);
                cnt += 1;
                if sw_abc == 1 {
                    char_sum += 100 * cnt;
                } else {
                    char_sum += 101 * cnt;
                }
            } else if c2 != b"{F" && c2 != b"{A" {
                let c2_str = std::str::from_utf8(c2).unwrap_or("");
                let found = code_c.iter().any(|cc| cc == c2_str);
                if !found {
                    sw_abc = gs1128_get_abc(&bytes[i..]);
                    cnt += 1;
                    if sw_abc == 1 {
                        char_sum += 100 * cnt;
                    } else {
                        char_sum += 101 * cnt;
                    }
                }
            }
        }

        let idx: i32;
        if i + 6 <= bytes.len() && &bytes[i..i + 6] == b"{FNC1}" {
            idx = 102;
            i += 6;
        } else if i + 4 <= bytes.len() && &bytes[i..i + 4] == b"{AI}" {
            i += 4;
            continue;
        } else if sw_abc == 0 {
            let c = gs1128_safe_sub(bytes, i, 1);
            if !c.is_empty() {
                let pos = code_a.find(c[0] as char);
                idx = pos.map(|p| p as i32).unwrap_or(-1);
            } else {
                idx = -1;
            }
            i += 1;
        } else if sw_abc == 1 {
            let c = gs1128_safe_sub(bytes, i, 1);
            if !c.is_empty() {
                let pos = code_b.find(c[0] as char);
                idx = pos.map(|p| p as i32).unwrap_or(-1);
            } else {
                idx = -1;
            }
            i += 1;
        } else {
            let c2 = gs1128_safe_sub(bytes, i, 2);
            if c2.len() < 2 {
                let pos = code_b.find(bytes[i] as char);
                idx = pos.map(|p| p as i32).unwrap_or(-1);
                i += 1;
            } else {
                let c2_str = std::str::from_utf8(c2).unwrap_or("");
                idx = code_c
                    .iter()
                    .position(|cc| cc == c2_str)
                    .map(|p| p as i32)
                    .unwrap_or(-1);
                i += 2;
            }
        }

        if idx != -1 {
            cnt += 1;
            char_sum += idx * cnt;
        }
    }

    char_sum % 103
}

/// Calculates Modulus 10 Weight 3-1 check digit.
pub fn calc_check_digit_mod10_w3(digits: &str) -> i32 {
    let mut total = 0;
    let mut weight3 = true;
    for c in digits.bytes().rev() {
        let d = (c - b'0') as i32;
        if weight3 {
            total += d * 3;
        } else {
            total += d;
        }
        weight3 = !weight3;
    }
    let m = total % 10;
    if m == 0 { 0 } else { 10 - m }
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_encode_basic() {
        let pattern = encode_gs1128("{FNC1}0104912345123459").unwrap();
        assert!(!pattern.is_empty());
    }

    #[test]
    fn test_encode_empty() {
        assert!(encode_gs1128("").is_err());
    }

    #[test]
    fn test_check_digit_mod10() {
        assert_eq!(calc_check_digit_mod10_w3("629104150021"), 3);
    }
}
