use crate::base_2d::BarcodeBase2D;
use crate::error::{BarcodeError, Result};

// ---------------------------------------------------------------------------
// DataMatrix code-size constants (public API)
// ---------------------------------------------------------------------------

pub const DX_SZ_RECT_AUTO: i32 = -3;
pub const DX_SZ_AUTO: i32 = -2;
pub const DX_SZ_SHAPE_AUTO: i32 = -1;
pub const DX_SZ_10X10: i32 = 0;
pub const DX_SZ_12X12: i32 = 1;
pub const DX_SZ_14X14: i32 = 2;
pub const DX_SZ_16X16: i32 = 3;
pub const DX_SZ_18X18: i32 = 4;
pub const DX_SZ_20X20: i32 = 5;
pub const DX_SZ_22X22: i32 = 6;
pub const DX_SZ_24X24: i32 = 7;
pub const DX_SZ_26X26: i32 = 8;
pub const DX_SZ_32X32: i32 = 9;
pub const DX_SZ_36X36: i32 = 10;
pub const DX_SZ_40X40: i32 = 11;
pub const DX_SZ_44X44: i32 = 12;
pub const DX_SZ_48X48: i32 = 13;
pub const DX_SZ_52X52: i32 = 14;
pub const DX_SZ_64X64: i32 = 15;
pub const DX_SZ_72X72: i32 = 16;
pub const DX_SZ_80X80: i32 = 17;
pub const DX_SZ_88X88: i32 = 18;
pub const DX_SZ_96X96: i32 = 19;
pub const DX_SZ_104X104: i32 = 20;
pub const DX_SZ_120X120: i32 = 21;
pub const DX_SZ_132X132: i32 = 22;
pub const DX_SZ_144X144: i32 = 23;
pub const DX_SZ_8X18: i32 = 24;
pub const DX_SZ_8X32: i32 = 25;
pub const DX_SZ_12X26: i32 = 26;
pub const DX_SZ_12X36: i32 = 27;
pub const DX_SZ_16X36: i32 = 28;
pub const DX_SZ_16X48: i32 = 29;

// ---------------------------------------------------------------------------
// Encoding scheme constants (public API)
// ---------------------------------------------------------------------------

pub const DX_SCHEME_AUTO_FAST: i32 = -2;
pub const DX_SCHEME_AUTO_BEST: i32 = -1;
pub const DX_SCHEME_ASCII: i32 = 0;
pub const DX_SCHEME_C40: i32 = 1;
pub const DX_SCHEME_TEXT: i32 = 2;
pub const DX_SCHEME_X12: i32 = 3;
pub const DX_SCHEME_EDIFACT: i32 = 4;
pub const DX_SCHEME_BASE256: i32 = 5;
pub const DX_SCHEME_ASCII_GS1: i32 = 6;

// ---------------------------------------------------------------------------
// Internal symbol attribute indices
// ---------------------------------------------------------------------------

const DX_ATTR_SYMBOL_ROWS: usize = 0;
const DX_ATTR_SYMBOL_COLS: usize = 1;
const DX_ATTR_DATA_REGION_ROWS: usize = 2;
const DX_ATTR_DATA_REGION_COLS: usize = 3;
const DX_ATTR_HORIZ_DATA_REGIONS: usize = 4;
const DX_ATTR_VERT_DATA_REGIONS: usize = 5;
const DX_ATTR_MAPPING_MATRIX_ROWS: usize = 6;
const DX_ATTR_MAPPING_MATRIX_COLS: usize = 7;
const DX_ATTR_INTERLEAVED_BLOCKS: usize = 8;
const DX_ATTR_BLOCK_ERROR_WORDS: usize = 9;
const DX_ATTR_BLOCK_MAX_CORRECTABLE: usize = 10;
const DX_ATTR_SYMBOL_DATA_WORDS: usize = 11;
const DX_ATTR_SYMBOL_ERROR_WORDS: usize = 12;
const DX_ATTR_SYMBOL_MAX_CORRECTABLE: usize = 13;

// ---------------------------------------------------------------------------
// Internal mask bits
// ---------------------------------------------------------------------------

const DX_MASK_BIT1: i32 = 0x80;
const DX_MASK_BIT2: i32 = 0x40;
const DX_MASK_BIT3: i32 = 0x20;
const DX_MASK_BIT4: i32 = 0x10;
const DX_MASK_BIT5: i32 = 0x08;
const DX_MASK_BIT6: i32 = 0x04;
const DX_MASK_BIT7: i32 = 0x02;
const DX_MASK_BIT8: i32 = 0x01;

// ---------------------------------------------------------------------------
// Channel status / unlatch constants
// ---------------------------------------------------------------------------

const DX_CHANNEL_VALID: i32 = 0x00;
const DX_CHANNEL_UNSUPPORTED_CHAR: i32 = 0x01;
const DX_CHANNEL_CANNOT_UNLATCH: i32 = 0x02;

const DX_UNLATCH_EXPLICIT: i32 = 0;
const DX_UNLATCH_IMPLICIT: i32 = 1;

// ---------------------------------------------------------------------------
// Module state flags
// ---------------------------------------------------------------------------

const DX_MODULE_ON_BLUE: i32 = 0x04;
const DX_MODULE_ON_RGB: i32 = 0x07;
const DX_MODULE_ASSIGNED: i32 = 0x10;
const DX_MODULE_VISITED: i32 = 0x20;
const DX_MODULE_DATA: i32 = 0x40;

// ---------------------------------------------------------------------------
// Encoding character constants
// ---------------------------------------------------------------------------

const DX_CHAR_ASCII_PAD: i32 = 129;
const DX_CHAR_ASCII_UPPER_SHIFT: i32 = 235;
const DX_CHAR_TRIPLET_SHIFT1: i32 = 0;
const DX_CHAR_TRIPLET_SHIFT2: i32 = 1;
const DX_CHAR_TRIPLET_SHIFT3: i32 = 2;
const DX_CHAR_FNC1: i32 = 232;
const DX_CHAR_TRIPLET_UNLATCH: i32 = 254;
const DX_CHAR_EDIFACT_UNLATCH: i32 = 31;
const DX_CHAR_C40_LATCH: i32 = 230;
const DX_CHAR_TEXT_LATCH: i32 = 239;
const DX_CHAR_X12_LATCH: i32 = 238;
const DX_CHAR_EDIFACT_LATCH: i32 = 240;
const DX_CHAR_BASE256_LATCH: i32 = 231;

const DX_SZ_SQUARE_COUNT: i32 = 24;
const DX_SZ_RECT_COUNT: i32 = 6;
const DX_UNDEFINED: i32 = -1;

// ---------------------------------------------------------------------------
// Symbol attribute tables (30 entries: 24 square + 6 rectangular)
// ---------------------------------------------------------------------------

const DX_SYMBOL_ROWS: [i32; 30] = [
    10, 12, 14, 16, 18, 20, 22, 24, 26,
    32, 36, 40, 44, 48, 52,
    64, 72, 80, 88, 96, 104,
    120, 132, 144,
    8, 8, 12, 12, 16, 16,
];

const DX_SYMBOL_COLS: [i32; 30] = [
    10, 12, 14, 16, 18, 20, 22, 24, 26,
    32, 36, 40, 44, 48, 52,
    64, 72, 80, 88, 96, 104,
    120, 132, 144,
    18, 32, 26, 36, 36, 48,
];

const DX_DATA_REGION_ROWS: [i32; 30] = [
    8, 10, 12, 14, 16, 18, 20, 22, 24,
    14, 16, 18, 20, 22, 24,
    14, 16, 18, 20, 22, 24,
    18, 20, 22,
    6, 6, 10, 10, 14, 14,
];

const DX_DATA_REGION_COLS: [i32; 30] = [
    8, 10, 12, 14, 16, 18, 20, 22, 24,
    14, 16, 18, 20, 22, 24,
    14, 16, 18, 20, 22, 24,
    18, 20, 22,
    16, 14, 24, 16, 16, 22,
];

const DX_HORIZ_DATA_REGIONS: [i32; 30] = [
    1, 1, 1, 1, 1, 1, 1, 1, 1,
    2, 2, 2, 2, 2, 2,
    4, 4, 4, 4, 4, 4,
    6, 6, 6,
    1, 2, 1, 2, 2, 2,
];

const DX_INTERLEAVED_BLOCKS: [i32; 30] = [
    1, 1, 1, 1, 1, 1, 1, 1, 1,
    1, 1, 1, 1, 1, 2,
    2, 4, 4, 4, 4, 6,
    6, 8, 10,
    1, 1, 1, 1, 1, 1,
];

const DX_SYMBOL_DATA_WORDS: [i32; 30] = [
    3, 5, 8, 12, 18, 22, 30, 36, 44,
    62, 86, 114, 144, 174, 204,
    280, 368, 456, 576, 696, 816,
    1050, 1304, 1558,
    5, 10, 16, 22, 32, 49,
];

const DX_BLOCK_ERROR_WORDS: [i32; 30] = [
    5, 7, 10, 12, 14, 18, 20, 24, 28,
    36, 42, 48, 56, 68, 42,
    56, 36, 48, 56, 68, 56,
    68, 62, 62,
    7, 11, 14, 18, 24, 28,
];

const DX_BLOCK_MAX_CORRECTABLE: [i32; 30] = [
    2, 3, 5, 6, 7, 9, 10, 12, 14,
    18, 21, 24, 28, 34, 21,
    28, 18, 24, 28, 34, 28,
    34, 31, 31,
    3, 5, 7, 9, 12, 14,
];

// ---------------------------------------------------------------------------
// GF(256) tables
// ---------------------------------------------------------------------------

const DX_ALOG_VAL: [i32; 256] = [
    1, 2, 4, 8, 16, 32, 64, 128, 45, 90, 180, 69, 138, 57, 114, 228,
    229, 231, 227, 235, 251, 219, 155, 27, 54, 108, 216, 157, 23, 46, 92, 184,
    93, 186, 89, 178, 73, 146, 9, 18, 36, 72, 144, 13, 26, 52, 104, 208,
    141, 55, 110, 220, 149, 7, 14, 28, 56, 112, 224, 237, 247, 195, 171, 123,
    246, 193, 175, 115, 230, 225, 239, 243, 203, 187, 91, 182, 65, 130, 41, 82,
    164, 101, 202, 185, 95, 190, 81, 162, 105, 210, 137, 63, 126, 252, 213, 135,
    35, 70, 140, 53, 106, 212, 133, 39, 78, 156, 21, 42, 84, 168, 125, 250,
    217, 159, 19, 38, 76, 152, 29, 58, 116, 232, 253, 215, 131, 43, 86, 172,
    117, 234, 249, 223, 147, 11, 22, 44, 88, 176, 77, 154, 25, 50, 100, 200,
    189, 87, 174, 113, 226, 233, 255, 211, 139, 59, 118, 236, 245, 199, 163, 107,
    214, 129, 47, 94, 188, 85, 170, 121, 242, 201, 191, 83, 166, 97, 194, 169,
    127, 254, 209, 143, 51, 102, 204, 181, 71, 142, 49, 98, 196, 165, 103, 206,
    177, 79, 158, 17, 34, 68, 136, 61, 122, 244, 197, 167, 99, 198, 161, 111,
    222, 145, 15, 30, 60, 120, 240, 205, 183, 67, 134, 33, 66, 132, 37, 74,
    148, 5, 10, 20, 40, 80, 160, 109, 218, 153, 31, 62, 124, 248, 221, 151,
    3, 6, 12, 24, 48, 96, 192, 173, 119, 238, 241, 207, 179, 75, 150, 1,
];

const DX_LOG_VAL: [i32; 256] = [
    -255, 255, 1, 240, 2, 225, 241, 53, 3, 38, 226, 133, 242, 43, 54, 210,
    4, 195, 39, 114, 227, 106, 134, 28, 243, 140, 44, 23, 55, 118, 211, 234,
    5, 219, 196, 96, 40, 222, 115, 103, 228, 78, 107, 125, 135, 8, 29, 162,
    244, 186, 141, 180, 45, 99, 24, 49, 56, 13, 119, 153, 212, 199, 235, 91,
    6, 76, 220, 217, 197, 11, 97, 184, 41, 36, 223, 253, 116, 138, 104, 193,
    229, 86, 79, 171, 108, 165, 126, 145, 136, 34, 9, 74, 30, 32, 163, 84,
    245, 173, 187, 204, 142, 81, 181, 190, 46, 88, 100, 159, 25, 231, 50, 207,
    57, 147, 14, 67, 120, 128, 154, 248, 213, 167, 200, 63, 236, 110, 92, 176,
    7, 161, 77, 124, 221, 102, 218, 95, 198, 90, 12, 152, 98, 48, 185, 179,
    42, 209, 37, 132, 224, 52, 254, 239, 117, 233, 139, 22, 105, 27, 194, 113,
    230, 206, 87, 158, 80, 189, 172, 203, 109, 175, 166, 62, 127, 247, 146, 66,
    137, 192, 35, 252, 10, 183, 75, 216, 31, 83, 33, 73, 164, 144, 85, 170,
    246, 65, 174, 61, 188, 202, 205, 157, 143, 169, 82, 72, 182, 215, 191, 251,
    47, 178, 89, 151, 101, 94, 160, 123, 26, 112, 232, 21, 51, 238, 208, 131,
    58, 69, 148, 18, 15, 16, 68, 17, 121, 149, 129, 19, 155, 59, 249, 70,
    214, 250, 168, 71, 201, 156, 64, 60, 237, 130, 111, 20, 93, 122, 177, 150,
];

// ---------------------------------------------------------------------------
// GF(256) helper functions
// ---------------------------------------------------------------------------

fn dx_gf_product(a: i32, b: i32) -> i32 {
    if a == 0 || b == 0 {
        return 0;
    }
    DX_ALOG_VAL[((DX_LOG_VAL[a as usize] + DX_LOG_VAL[b as usize]) % 255) as usize]
}

fn dx_gf_sum(a: i32, b: i32) -> i32 {
    a ^ b
}

fn dx_gf_doublify(a: i32, b: i32) -> i32 {
    if a == 0 {
        return 0;
    }
    if b == 0 {
        return a;
    }
    DX_ALOG_VAL[((DX_LOG_VAL[a as usize] + b) % 255) as usize]
}

// ---------------------------------------------------------------------------
// Symbol attribute lookup
// ---------------------------------------------------------------------------

fn dx_get_symbol_attribute(attribute: usize, size_idx: i32) -> i32 {
    if !(0..DX_SZ_SQUARE_COUNT + DX_SZ_RECT_COUNT).contains(&size_idx) {
        return DX_UNDEFINED;
    }
    let idx = size_idx as usize;
    match attribute {
        DX_ATTR_SYMBOL_ROWS => DX_SYMBOL_ROWS[idx],
        DX_ATTR_SYMBOL_COLS => DX_SYMBOL_COLS[idx],
        DX_ATTR_DATA_REGION_ROWS => DX_DATA_REGION_ROWS[idx],
        DX_ATTR_DATA_REGION_COLS => DX_DATA_REGION_COLS[idx],
        DX_ATTR_HORIZ_DATA_REGIONS => DX_HORIZ_DATA_REGIONS[idx],
        DX_ATTR_VERT_DATA_REGIONS => {
            if size_idx < DX_SZ_SQUARE_COUNT {
                DX_HORIZ_DATA_REGIONS[idx]
            } else {
                1
            }
        }
        DX_ATTR_MAPPING_MATRIX_ROWS => {
            DX_DATA_REGION_ROWS[idx]
                * dx_get_symbol_attribute(DX_ATTR_VERT_DATA_REGIONS, size_idx)
        }
        DX_ATTR_MAPPING_MATRIX_COLS => {
            DX_DATA_REGION_COLS[idx] * DX_HORIZ_DATA_REGIONS[idx]
        }
        DX_ATTR_INTERLEAVED_BLOCKS => DX_INTERLEAVED_BLOCKS[idx],
        DX_ATTR_BLOCK_ERROR_WORDS => DX_BLOCK_ERROR_WORDS[idx],
        DX_ATTR_BLOCK_MAX_CORRECTABLE => DX_BLOCK_MAX_CORRECTABLE[idx],
        DX_ATTR_SYMBOL_DATA_WORDS => DX_SYMBOL_DATA_WORDS[idx],
        DX_ATTR_SYMBOL_ERROR_WORDS => {
            DX_BLOCK_ERROR_WORDS[idx] * DX_INTERLEAVED_BLOCKS[idx]
        }
        DX_ATTR_SYMBOL_MAX_CORRECTABLE => {
            DX_BLOCK_MAX_CORRECTABLE[idx] * DX_INTERLEAVED_BLOCKS[idx]
        }
        _ => DX_UNDEFINED,
    }
}

fn dx_get_block_data_size(size_idx: i32, block_idx: i32) -> i32 {
    let sdw = dx_get_symbol_attribute(DX_ATTR_SYMBOL_DATA_WORDS, size_idx);
    let ilb = dx_get_symbol_attribute(DX_ATTR_INTERLEAVED_BLOCKS, size_idx);
    if sdw < 1 || ilb < 1 {
        return DX_UNDEFINED;
    }
    let count = sdw / ilb;
    if size_idx == DX_SZ_144X144 && block_idx < 8 {
        return count + 1;
    }
    count
}

// ---------------------------------------------------------------------------
// Reed-Solomon ECC generation
// ---------------------------------------------------------------------------

fn dx_gen_reed_sol_ecc(code: &mut [i32], size_idx: i32) {
    let sdw = dx_get_symbol_attribute(DX_ATTR_SYMBOL_DATA_WORDS, size_idx);
    let sew = dx_get_symbol_attribute(DX_ATTR_SYMBOL_ERROR_WORDS, size_idx);
    let stw = sdw + sew;
    let bew = dx_get_symbol_attribute(DX_ATTR_BLOCK_ERROR_WORDS, size_idx);
    let step = dx_get_symbol_attribute(DX_ATTR_INTERLEAVED_BLOCKS, size_idx);

    if bew != sew / step {
        return;
    }

    let mut g = [0i32; 69];
    g.fill(0x01);

    for i in 1..=bew {
        for j in (0..i).rev() {
            g[j as usize] = dx_gf_doublify(g[j as usize], i);
            if j > 0 {
                g[j as usize] = dx_gf_sum(g[j as usize], g[(j - 1) as usize]);
            }
        }
    }

    for block in 0..step {
        let mut b = [0i32; 68];

        let mut i = block;
        while i < sdw {
            let val = dx_gf_sum(b[(bew - 1) as usize], code[i as usize]);
            for j in (1..bew).rev() {
                b[j as usize] = dx_gf_sum(b[(j - 1) as usize], dx_gf_product(g[j as usize], val));
            }
            b[0] = dx_gf_product(g[0], val);
            i += step;
        }

        let block_data_words = dx_get_block_data_size(size_idx, block);
        let mut b_index = bew;
        let mut i = block + step * block_data_words;
        while i < stw {
            b_index -= 1;
            code[i as usize] = b[b_index as usize];
            i += step;
        }
    }
}

// ---------------------------------------------------------------------------
// Randomize helpers
// ---------------------------------------------------------------------------

fn dx_randomize_253(codeword_value: i32, codeword_position: i32) -> i32 {
    let pseudo_random = ((149 * codeword_position) % 253) + 1;
    let tmp = codeword_value + pseudo_random;
    let tmp = if tmp > 254 { tmp - 254 } else { tmp };
    tmp & 0xFF
}

fn dx_randomize_255(codeword_value: i32, codeword_position: i32) -> i32 {
    let pseudo_random = ((149 * codeword_position) % 255) + 1;
    let tmp = codeword_value + pseudo_random;
    if tmp <= 255 {
        tmp & 0xFF
    } else {
        (tmp - 256) & 0xFF
    }
}

fn dx_un_randomize_255(value: i32, idx: i32) -> i32 {
    let pseudo_random = ((149 * idx) % 255) + 1;
    let mut tmp = value - pseudo_random;
    if tmp < 0 {
        tmp += 256;
    }
    tmp & 0xFF
}

// ---------------------------------------------------------------------------
// Triplet / Quadruplet helpers
// ---------------------------------------------------------------------------

fn dx_get_triplet_values(cw1: i32, cw2: i32) -> [i32; 3] {
    let compact = (cw1 << 8) | cw2;
    [
        (compact - 1) / 1600,
        ((compact - 1) / 40) % 40,
        (compact - 1) % 40,
    ]
}

fn dx_get_quadruplet_values(cw1: i32, cw2: i32, cw3: i32) -> [i32; 4] {
    [
        cw1 >> 2,
        ((cw1 & 0x03) << 4) | ((cw2 & 0xf0) >> 4),
        ((cw2 & 0x0f) << 2) | ((cw3 & 0xc0) >> 6),
        cw3 & 0x3f,
    ]
}

// ---------------------------------------------------------------------------
// DxMessage
// ---------------------------------------------------------------------------

struct DxMessage {
    array: Vec<i32>,
    code: Vec<i32>,
    pad_count: i32,
}

impl DxMessage {
    fn new(size_idx: i32) -> Self {
        let mr = dx_get_symbol_attribute(DX_ATTR_MAPPING_MATRIX_ROWS, size_idx);
        let mc = dx_get_symbol_attribute(DX_ATTR_MAPPING_MATRIX_COLS, size_idx);
        let cs = dx_get_symbol_attribute(DX_ATTR_SYMBOL_DATA_WORDS, size_idx)
            + dx_get_symbol_attribute(DX_ATTR_SYMBOL_ERROR_WORDS, size_idx);
        Self {
            array: vec![0; (mc * mr) as usize],
            code: vec![0; cs as usize],
            pad_count: 0,
        }
    }

    fn symbol_module_status(&self, size_idx: i32, symbol_row: i32, symbol_col: i32) -> i32 {
        let drr = dx_get_symbol_attribute(DX_ATTR_DATA_REGION_ROWS, size_idx);
        let drc = dx_get_symbol_attribute(DX_ATTR_DATA_REGION_COLS, size_idx);
        let sr = dx_get_symbol_attribute(DX_ATTR_SYMBOL_ROWS, size_idx);
        let mc = dx_get_symbol_attribute(DX_ATTR_MAPPING_MATRIX_COLS, size_idx);

        let srr = sr - symbol_row - 1;
        let mapping_row = srr - 1 - 2 * (srr / (drr + 2));
        let mapping_col = symbol_col - 1 - 2 * (symbol_col / (drc + 2));

        // Solid finder pattern (L-shape: bottom and left edges)
        if symbol_row % (drr + 2) == 0 || symbol_col % (drc + 2) == 0 {
            return DX_MODULE_ON_RGB;
        }

        // Alternating clock pattern (top edge)
        if (symbol_row + 1) % (drr + 2) == 0 {
            return if (symbol_col & 0x01) != 0 {
                0
            } else {
                DX_MODULE_ON_RGB
            };
        }

        // Alternating clock pattern (right edge)
        if (symbol_col + 1) % (drc + 2) == 0 {
            return if (symbol_row & 0x01) != 0 {
                0
            } else {
                DX_MODULE_ON_RGB
            };
        }

        // Data module
        self.array[(mapping_row * mc + mapping_col) as usize] | DX_MODULE_DATA
    }
}

// ---------------------------------------------------------------------------
// DxChannel
// ---------------------------------------------------------------------------

#[derive(Clone)]
struct DxChannel {
    enc_scheme: i32,
    invalid: i32,
    input_index: i32,
    encoded_length: i32,
    current_length: i32,
    first_code_word: i32,
    input: Vec<i32>,
    encoded_words: Vec<i32>,
}

impl DxChannel {
    fn new() -> Self {
        Self {
            enc_scheme: DX_SCHEME_ASCII,
            invalid: DX_CHANNEL_VALID,
            input_index: 0,
            encoded_length: 0,
            current_length: 0,
            first_code_word: 0,
            input: Vec::new(),
            encoded_words: vec![0; 1558],
        }
    }
}

// ---------------------------------------------------------------------------
// DxEncode (main encoding engine)
// ---------------------------------------------------------------------------

struct DxEncode {
    scheme: i32,
    size_idx_req: i32,
    #[allow(dead_code)]
    margin_size: i32,
    #[allow(dead_code)]
    module_size: i32,
    flg_gs1: bool,
    message: Option<DxMessage>,
    raw_data: Option<Vec<Vec<bool>>>,
    size_idx: i32,
    symbol_rows: i32,
    symbol_cols: i32,
    mapping_rows: i32,
    mapping_cols: i32,
}

impl DxEncode {
    fn new() -> Self {
        Self {
            scheme: DX_SCHEME_ASCII,
            size_idx_req: DX_SZ_AUTO,
            margin_size: 10,
            module_size: 5,
            flg_gs1: false,
            message: None,
            raw_data: None,
            size_idx: 0,
            symbol_rows: 0,
            symbol_cols: 0,
            mapping_rows: 0,
            mapping_cols: 0,
        }
    }

    fn encode_data_matrix_raw(&mut self, input_bytes: &[i32]) -> Result<()> {
        let mut buf = vec![0i32; 4096];
        let size_idx = self.size_idx_req;

        let (data_word_count, size_idx) = self.encode_data_codewords(&mut buf, input_bytes, size_idx)?;
        if data_word_count <= 0 {
            return Err(BarcodeError::EncodingError(
                "failed to encode data: data may be empty or contain invalid characters".into(),
            ));
        }
        if size_idx == DX_SZ_AUTO || size_idx == DX_SZ_RECT_AUTO {
            return Err(BarcodeError::EncodingError(
                "data too long for specified symbol size".into(),
            ));
        }

        let padded_size = dx_get_symbol_attribute(DX_ATTR_SYMBOL_DATA_WORDS, size_idx);
        let pad_count = self.add_pad_chars(&mut buf, data_word_count, padded_size);

        self.size_idx = size_idx;
        self.symbol_rows = dx_get_symbol_attribute(DX_ATTR_SYMBOL_ROWS, size_idx);
        self.symbol_cols = dx_get_symbol_attribute(DX_ATTR_SYMBOL_COLS, size_idx);
        self.mapping_rows = dx_get_symbol_attribute(DX_ATTR_MAPPING_MATRIX_ROWS, size_idx);
        self.mapping_cols = dx_get_symbol_attribute(DX_ATTR_MAPPING_MATRIX_COLS, size_idx);

        let mut message = DxMessage::new(size_idx);
        message.pad_count = pad_count;

        message.code[..padded_size as usize].copy_from_slice(&buf[..padded_size as usize]);

        dx_gen_reed_sol_ecc(&mut message.code, size_idx);
        self.module_placement_ecc200(&mut message.array, &message.code, size_idx, DX_MODULE_ON_RGB);

        self.message = Some(message);
        self.print_pattern_raw();

        Ok(())
    }

    // --- Encoding schemes ---

    fn encode_data_codewords(
        &mut self,
        buf: &mut [i32],
        input_bytes: &[i32],
        size_idx: i32,
    ) -> Result<(i32, i32)> {
        let dwc = match self.scheme {
            DX_SCHEME_AUTO_BEST => self.encode_auto_best(buf, input_bytes),
            DX_SCHEME_AUTO_FAST => 0,
            _ => self.encode_single_scheme(buf, input_bytes, self.scheme),
        };

        let size_idx = self.find_correct_symbol_size(dwc, size_idx);
        if size_idx == DX_SZ_SHAPE_AUTO {
            return Ok((0, size_idx));
        }
        Ok((dwc, size_idx))
    }

    fn find_correct_symbol_size(&self, data_words: i32, size_idx_request: i32) -> i32 {
        if data_words <= 0 {
            return DX_SZ_SHAPE_AUTO;
        }

        let mut size_idx = size_idx_request;

        if size_idx_request == DX_SZ_AUTO || size_idx_request == DX_SZ_RECT_AUTO {
            let (idx_beg, idx_end) = if size_idx_request == DX_SZ_AUTO {
                (0, DX_SZ_SQUARE_COUNT)
            } else {
                (DX_SZ_SQUARE_COUNT, DX_SZ_SQUARE_COUNT + DX_SZ_RECT_COUNT)
            };

            size_idx = idx_beg;
            for i in idx_beg..idx_end {
                size_idx = i;
                if dx_get_symbol_attribute(DX_ATTR_SYMBOL_DATA_WORDS, i) >= data_words {
                    break;
                }
            }

            if size_idx == idx_end {
                return DX_SZ_SHAPE_AUTO;
            }
        }

        if data_words > dx_get_symbol_attribute(DX_ATTR_SYMBOL_DATA_WORDS, size_idx) {
            return DX_SZ_SHAPE_AUTO;
        }

        size_idx
    }

    fn encode_single_scheme(&mut self, buf: &mut [i32], codewords: &[i32], scheme: i32) -> i32 {
        let mut channel = DxChannel::new();
        self.init_channel(&mut channel, codewords);

        while channel.input_index < channel.input.len() as i32 {
            let ok = self.encode_next_word(&mut channel, scheme);
            if !ok {
                return 0;
            }
            if channel.invalid != DX_CHANNEL_VALID {
                return 0;
            }
        }

        let size = channel.encoded_length / 12;
        buf[..size as usize].copy_from_slice(&channel.encoded_words[..size as usize]);
        size
    }

    fn encode_auto_best(&mut self, buf: &mut [i32], codewords: &[i32]) -> i32 {
        let mut channels: Vec<Option<DxChannel>> = Vec::with_capacity(6);

        for ts in DX_SCHEME_ASCII..=DX_SCHEME_BASE256 {
            let mut ch = DxChannel::new();
            self.init_channel(&mut ch, codewords);
            let ok = self.encode_next_word(&mut ch, ts);
            // C++ bug: "if (err) return 0;" - returns 0 when encoding succeeds
            if ok {
                return 0;
            }
            channels.push(Some(ch));
        }

        // Encode remaining (unreachable due to above bug, ported for faithfulness)
        while channels[0].as_ref().is_none_or(|c| c.input_index < c.input.len() as i32) {
            if channels[0].is_none() {
                break;
            }
            let ch0 = channels[0].as_ref().unwrap();
            if ch0.input_index >= ch0.input.len() as i32 {
                break;
            }
            let mut best: Vec<Option<DxChannel>> = Vec::with_capacity(6);
            for ts in DX_SCHEME_ASCII..=DX_SCHEME_BASE256 {
                best.push(Some(self.find_best_channel(&mut channels, ts)));
            }
            channels = best;
        }

        let mut winner_idx = DX_SCHEME_ASCII as usize;
        for ts in (DX_SCHEME_ASCII + 1) as usize..=DX_SCHEME_BASE256 as usize {
            if let Some(ch) = &channels[ts] {
                if ch.invalid != DX_CHANNEL_VALID {
                    continue;
                }
                if let Some(w) = &channels[winner_idx] {
                    if ch.encoded_length < w.encoded_length {
                        winner_idx = ts;
                    }
                }
            }
        }

        if let Some(winner) = &channels[winner_idx] {
            let ws = winner.encoded_length / 12;
            buf[..ws as usize].copy_from_slice(&winner.encoded_words[..ws as usize]);
            ws
        } else {
            0
        }
    }

    fn find_best_channel(&mut self, channels: &mut [Option<DxChannel>], target_scheme: i32) -> DxChannel {
        let mut winner: Option<DxChannel> = None;

        for s in DX_SCHEME_ASCII..=DX_SCHEME_BASE256 {
            let channel = match &mut channels[s as usize] {
                Some(c) => c,
                None => continue,
            };
            if channel.invalid != DX_CHANNEL_VALID {
                continue;
            }
            if channel.input_index == channel.input.len() as i32 {
                continue;
            }
            self.encode_next_word(channel, target_scheme);
            if (channel.invalid & DX_CHANNEL_UNSUPPORTED_CHAR) != 0 {
                return channel.clone();
            }
            if (channel.invalid & DX_CHANNEL_CANNOT_UNLATCH) != 0 {
                continue;
            }
            let dominated = match &winner {
                Some(w) => channel.current_length < w.current_length,
                None => true,
            };
            if dominated {
                winner = Some(channel.clone());
            }
        }

        winner.unwrap_or_else(DxChannel::new)
    }

    fn encode_next_word(&mut self, channel: &mut DxChannel, target_scheme: i32) -> bool {
        if channel.enc_scheme != target_scheme {
            self.change_enc_scheme(channel, target_scheme, DX_UNLATCH_EXPLICIT);
            if channel.invalid != DX_CHANNEL_VALID {
                return false;
            }
        }

        if channel.enc_scheme != target_scheme {
            return false;
        }

        match channel.enc_scheme {
            DX_SCHEME_ASCII => self.encode_ascii_codeword(channel),
            DX_SCHEME_C40 | DX_SCHEME_TEXT | DX_SCHEME_X12 => {
                self.encode_triplet_codeword(channel)
            }
            DX_SCHEME_EDIFACT => self.encode_edifact_codeword(channel),
            DX_SCHEME_BASE256 => self.encode_base256_codeword(channel),
            _ => false,
        }
    }

    // --- ASCII encoding ---

    fn encode_ascii_codeword(&mut self, channel: &mut DxChannel) -> bool {
        let input_value = channel.input[channel.input_index as usize];

        // Digit pair compression
        if is_digit_byte(input_value)
            && channel.current_length >= channel.first_code_word + 12
        {
            let prev_index = (channel.current_length - 12) / 12;
            let prev_value = (channel.encoded_words[prev_index as usize] - 1) & 0xFF;
            let prev_prev_value = if prev_index > channel.first_code_word / 12 {
                channel.encoded_words[(prev_index - 1) as usize]
            } else {
                0
            };

            if prev_prev_value != 235 && is_digit_byte(prev_value) {
                channel.encoded_words[prev_index as usize] =
                    (10 * (prev_value - 48) + (input_value - 48) + 130) & 0xFF;
                channel.input_index += 1;
                return true;
            }
        }

        // FNC1 in GS1 mode
        if self.flg_gs1 && input_value == DX_CHAR_FNC1 {
            self.push_input_word(channel, DX_CHAR_FNC1);
            self.increment_progress(channel, 12);
            channel.input_index += 1;
            return true;
        }

        // Upper shift for values >= 128
        let mut iv = input_value;
        if iv >= 128 {
            self.push_input_word(channel, DX_CHAR_ASCII_UPPER_SHIFT);
            self.increment_progress(channel, 12);
            iv -= 128;
        }

        self.push_input_word(channel, (iv + 1) & 0xFF);
        self.increment_progress(channel, 12);
        channel.input_index += 1;
        true
    }

    // --- Triplet encoding (C40/Text/X12) ---

    fn encode_triplet_codeword(&mut self, channel: &mut DxChannel) -> bool {
        if channel.enc_scheme != DX_SCHEME_C40
            && channel.enc_scheme != DX_SCHEME_TEXT
            && channel.enc_scheme != DX_SCHEME_X12
        {
            return false;
        }

        if channel.current_length > channel.encoded_length {
            return false;
        }

        if channel.current_length == channel.encoded_length {
            if channel.current_length % 12 != 0 {
                return false;
            }

            let mut ptr_index = channel.input_index;
            let mut triplet_count: i32 = 0;
            let mut buffer = [0i32; 6];

            loop {
                while triplet_count < 3 && ptr_index < channel.input.len() as i32 {
                    let input_word = channel.input[ptr_index as usize];
                    ptr_index += 1;
                    let output_words =
                        self.get_c40_text_x12_words(input_word, channel.enc_scheme);
                    if output_words.is_empty() {
                        channel.invalid = DX_CHANNEL_UNSUPPORTED_CHAR;
                        return false;
                    }
                    for &w in &output_words {
                        buffer[triplet_count as usize] = w & 0xFF;
                        triplet_count += 1;
                    }
                }

                let triplet = [buffer[0], buffer[1], buffer[2]];

                if triplet_count >= 3 {
                    self.push_triplet(channel, triplet);
                    buffer[0] = buffer[3];
                    buffer[1] = buffer[4];
                    buffer[2] = buffer[5];
                    triplet_count -= 3;
                }

                if ptr_index == channel.input.len() as i32 {
                    while channel.current_length < channel.encoded_length {
                        self.increment_progress(channel, 8);
                        channel.input_index += 1;
                    }

                    if channel.current_length == channel.encoded_length + 8 {
                        channel.current_length = channel.encoded_length;
                        channel.input_index -= 1;
                    }

                    let input_count = channel.input.len() as i32 - channel.input_index;
                    let ok = self.process_end_of_symbol_triplet(
                        channel,
                        triplet,
                        triplet_count,
                        input_count,
                    );
                    if !ok {
                        return false;
                    }
                    break;
                }

                if triplet_count == 0 {
                    break;
                }
            }
        }

        if channel.current_length < channel.encoded_length {
            self.increment_progress(channel, 8);
            channel.input_index += 1;
        }

        true
    }

    fn get_c40_text_x12_words(&self, input_word: i32, enc_scheme: i32) -> Vec<i32> {
        let mut result: Vec<i32> = Vec::new();
        let mut iw = input_word;

        if iw > 127 {
            if enc_scheme == DX_SCHEME_X12 {
                return result;
            }
            result.push(DX_CHAR_TRIPLET_SHIFT2);
            result.push(30);
            iw -= 128;
        }

        if enc_scheme == DX_SCHEME_X12 {
            match iw {
                13 => result.push(0),
                42 => result.push(1),
                62 => result.push(2),
                32 => result.push(3),
                45..=57 => result.push(iw - 44),
                65..=90 => result.push(iw - 51),
                _ => {}
            }
        } else {
            match iw {
                0..=31 => {
                    result.push(DX_CHAR_TRIPLET_SHIFT1);
                    result.push(iw);
                }
                32 => {
                    result.push(3);
                }
                33..=47 => {
                    result.push(DX_CHAR_TRIPLET_SHIFT2);
                    result.push(iw - 33);
                }
                48..=57 => {
                    result.push(iw - 44);
                }
                58..=64 => {
                    result.push(DX_CHAR_TRIPLET_SHIFT2);
                    result.push(iw - 43);
                }
                65..=90 if enc_scheme == DX_SCHEME_C40 => {
                    result.push(iw - 51);
                }
                65..=90 if enc_scheme == DX_SCHEME_TEXT => {
                    result.push(DX_CHAR_TRIPLET_SHIFT3);
                    result.push(iw - 64);
                }
                91..=95 => {
                    result.push(DX_CHAR_TRIPLET_SHIFT2);
                    result.push(iw - 69);
                }
                96 if enc_scheme == DX_SCHEME_TEXT => {
                    result.push(DX_CHAR_TRIPLET_SHIFT3);
                    result.push(0);
                }
                97..=122 if enc_scheme == DX_SCHEME_TEXT => {
                    result.push(iw - 83);
                }
                97..=127 => {
                    result.push(DX_CHAR_TRIPLET_SHIFT3);
                    result.push(iw - 96);
                }
                _ => {}
            }
        }

        result
    }

    fn process_end_of_symbol_triplet(
        &mut self,
        channel: &mut DxChannel,
        triplet: [i32; 3],
        triplet_count: i32,
        input_count: i32,
    ) -> bool {
        if channel.current_length % 12 != 0 {
            return false;
        }

        let input_adjust = triplet_count - input_count;
        let current_byte = channel.current_length / 12;
        let needed = if input_count == 3 {
            current_byte + 2
        } else {
            current_byte + input_count
        };
        let size_idx = self.find_correct_symbol_size(needed, self.size_idx_req);
        if size_idx == DX_SZ_SHAPE_AUTO {
            return false;
        }

        let remaining = dx_get_symbol_attribute(DX_ATTR_SYMBOL_DATA_WORDS, size_idx) - current_byte;

        if input_count == 1 && remaining == 1 {
            self.change_enc_scheme(channel, DX_SCHEME_ASCII, DX_UNLATCH_IMPLICIT);
            let ok = self.encode_next_word(channel, DX_SCHEME_ASCII);
            if !ok {
                return false;
            }
            if channel.invalid != DX_CHANNEL_VALID
                || channel.input_index != channel.input.len() as i32
            {
                return false;
            }
        } else if remaining == 2 {
            if triplet_count == 3 {
                self.push_triplet(channel, triplet);
                self.increment_progress(channel, 24);
                channel.enc_scheme = DX_SCHEME_ASCII;
                channel.input_index += 3;
                channel.input_index -= input_adjust;
            } else if triplet_count == 2 {
                let t = [triplet[0], triplet[1], 0];
                self.push_triplet(channel, t);
                self.increment_progress(channel, 24);
                channel.enc_scheme = DX_SCHEME_ASCII;
                channel.input_index += 2;
                channel.input_index -= input_adjust;
            } else if triplet_count == 1 {
                self.change_enc_scheme(channel, DX_SCHEME_ASCII, DX_UNLATCH_EXPLICIT);
                let ok = self.encode_next_word(channel, DX_SCHEME_ASCII);
                if !ok {
                    return false;
                }
                if channel.invalid != DX_CHANNEL_VALID {
                    return false;
                }
            }
        } else {
            let current_byte = channel.current_length / 12;
            let remaining =
                dx_get_symbol_attribute(DX_ATTR_SYMBOL_DATA_WORDS, size_idx) - current_byte;
            if remaining > 0 {
                self.change_enc_scheme(channel, DX_SCHEME_ASCII, DX_UNLATCH_EXPLICIT);
                while channel.input_index < channel.input.len() as i32 {
                    let ok = self.encode_next_word(channel, DX_SCHEME_ASCII);
                    if !ok {
                        return false;
                    }
                    if channel.invalid != DX_CHANNEL_VALID {
                        return false;
                    }
                }
            }
        }

        channel.input_index == channel.input.len() as i32
    }

    fn push_triplet(&mut self, channel: &mut DxChannel, triplet: [i32; 3]) {
        let tv = 1600 * triplet[0] + 40 * triplet[1] + triplet[2] + 1;
        self.push_input_word(channel, (tv >> 8) & 0xFF);
        self.push_input_word(channel, tv & 0xFF);
    }

    // --- Edifact encoding ---

    fn encode_edifact_codeword(&mut self, channel: &mut DxChannel) -> bool {
        if channel.enc_scheme != DX_SCHEME_EDIFACT {
            return false;
        }

        let input_value = channel.input[channel.input_index as usize];
        if !(32..=94).contains(&input_value) {
            channel.invalid = DX_CHANNEL_UNSUPPORTED_CHAR;
            return false;
        }

        self.push_input_word(channel, input_value & 0x3f);
        self.increment_progress(channel, 9);
        channel.input_index += 1;

        self.check_end_of_symbol_edifact(channel);
        true
    }

    fn check_end_of_symbol_edifact(&mut self, channel: &mut DxChannel) {
        let edifact_values = channel.input.len() as i32 - channel.input_index;
        if edifact_values > 4 {
            return;
        }

        let current_byte = channel.current_length / 12;
        let size_idx = self.find_correct_symbol_size(current_byte, DX_SZ_AUTO);
        let symbol_codewords =
            dx_get_symbol_attribute(DX_ATTR_SYMBOL_DATA_WORDS, size_idx) - current_byte;

        if channel.current_length % 12 == 0
            && (symbol_codewords == 1 || symbol_codewords == 2)
        {
            let ascii_codewords = edifact_values;
            if ascii_codewords <= symbol_codewords {
                self.change_enc_scheme(channel, DX_SCHEME_ASCII, DX_UNLATCH_IMPLICIT);
                for _ in 0..edifact_values {
                    let ok = self.encode_next_word(channel, DX_SCHEME_ASCII);
                    if !ok {
                        return;
                    }
                    if channel.invalid != DX_CHANNEL_VALID {
                        return;
                    }
                }
            }
        } else if edifact_values == 0 {
            self.change_enc_scheme(channel, DX_SCHEME_ASCII, DX_UNLATCH_EXPLICIT);
        }
    }

    // --- Base256 encoding ---

    fn encode_base256_codeword(&mut self, channel: &mut DxChannel) -> bool {
        if channel.enc_scheme != DX_SCHEME_BASE256 {
            return false;
        }

        let first_byte_ptr_index = channel.first_code_word / 12;
        let header_byte0 = dx_un_randomize_255(
            channel.encoded_words[first_byte_ptr_index as usize],
            channel.first_code_word / 12 + 1,
        );

        let mut new_data_length = if header_byte0 <= 249 {
            header_byte0
        } else {
            let mut ndl = 250 * (header_byte0 - 249);
            ndl += dx_un_randomize_255(
                channel.encoded_words[(first_byte_ptr_index + 1) as usize],
                channel.first_code_word / 12 + 2,
            );
            ndl
        };

        new_data_length += 1;

        let (header_byte_count, hb) = if new_data_length <= 249 {
            (1, [new_data_length & 0xFF, 0])
        } else {
            (
                2,
                [
                    (new_data_length / 250 + 249) & 0xFF,
                    (new_data_length % 250) & 0xFF,
                ],
            )
        };

        if new_data_length <= 0 || new_data_length > 1555 {
            return false;
        }

        if new_data_length == 250 {
            let mut i = channel.current_length / 12 - 1;
            while i > channel.first_code_word / 12 {
                let val_tmp = dx_un_randomize_255(
                    channel.encoded_words[i as usize],
                    i + 1,
                );
                channel.encoded_words[(i + 1) as usize] = dx_randomize_255(val_tmp, i + 2);
                i -= 1;
            }
            self.increment_progress(channel, 12);
            channel.encoded_length += 12;
        }

        for i in 0..header_byte_count {
            channel.encoded_words[(first_byte_ptr_index + i) as usize] = dx_randomize_255(
                hb[i as usize],
                channel.first_code_word / 12 + i + 1,
            );
        }

        self.push_input_word(
            channel,
            dx_randomize_255(
                channel.input[channel.input_index as usize],
                channel.current_length / 12 + 1,
            ),
        );
        self.increment_progress(channel, 12);
        channel.input_index += 1;
        true
    }

    // --- Channel utilities ---

    fn push_input_word(&self, channel: &mut DxChannel, codeword: i32) {
        if channel.encoded_length / 12 > 3 * 1558 {
            return;
        }

        match channel.enc_scheme {
            DX_SCHEME_ASCII | DX_SCHEME_C40 | DX_SCHEME_TEXT | DX_SCHEME_X12
            | DX_SCHEME_BASE256 => {
                channel.encoded_words[(channel.current_length / 12) as usize] = codeword & 0xFF;
                channel.encoded_length += 12;
            }
            DX_SCHEME_EDIFACT => {
                let pos = channel.current_length % 4;
                let start_byte = ((channel.current_length + 9) / 12) - pos;

                let mut q = dx_get_quadruplet_values(
                    channel.encoded_words[start_byte as usize],
                    channel.encoded_words[(start_byte + 1) as usize],
                    channel.encoded_words[(start_byte + 2) as usize],
                );
                q[pos as usize] = codeword & 0xFF;
                for i in (pos + 1)..4 {
                    q[i as usize] = 0;
                }

                if pos >= 2 {
                    channel.encoded_words[(start_byte + 2) as usize] =
                        ((q[2] & 0x03) << 6) | q[3];
                }
                if pos >= 1 {
                    channel.encoded_words[(start_byte + 1) as usize] =
                        ((q[1] & 0x0f) << 4) | (q[2] >> 2);
                }
                channel.encoded_words[start_byte as usize] = (q[0] << 2) | (q[1] >> 4);

                channel.encoded_length += 9;
            }
            _ => {}
        }
    }

    fn increment_progress(&self, channel: &mut DxChannel, encoded_units: i32) {
        if channel.enc_scheme == DX_SCHEME_C40 || channel.enc_scheme == DX_SCHEME_TEXT {
            let pos = (channel.current_length % 6) / 2;
            let start_byte = (channel.current_length / 12) - (pos >> 1);
            let triplet = dx_get_triplet_values(
                channel.encoded_words[start_byte as usize],
                channel.encoded_words[(start_byte + 1) as usize],
            );
            if triplet[pos as usize] <= 2 {
                channel.current_length += 8;
            }
        }

        channel.current_length += encoded_units;
    }

    fn change_enc_scheme(
        &mut self,
        channel: &mut DxChannel,
        target_scheme: i32,
        unlatch_type: i32,
    ) {
        if channel.enc_scheme == target_scheme {
            return;
        }

        match channel.enc_scheme {
            DX_SCHEME_ASCII => {
                if channel.current_length % 12 != 0 && channel.current_length % 20 != 0 {
                    return;
                }
            }
            DX_SCHEME_C40 | DX_SCHEME_TEXT | DX_SCHEME_X12 => {
                if channel.current_length % 12 != 0 && channel.current_length % 20 != 0 {
                    channel.invalid = DX_CHANNEL_CANNOT_UNLATCH;
                    return;
                }
                if channel.current_length != channel.encoded_length {
                    channel.invalid = DX_CHANNEL_CANNOT_UNLATCH;
                    return;
                }
                if unlatch_type == DX_UNLATCH_EXPLICIT {
                    self.push_input_word(channel, DX_CHAR_TRIPLET_UNLATCH);
                    self.increment_progress(channel, 12);
                }
            }
            DX_SCHEME_EDIFACT => {
                if channel.current_length % 3 != 0 {
                    return;
                }
                if unlatch_type == DX_UNLATCH_EXPLICIT {
                    self.push_input_word(channel, DX_CHAR_EDIFACT_UNLATCH);
                    self.increment_progress(channel, 9);
                }
                let advance = (channel.current_length % 4) * 3;
                channel.current_length += advance;
                channel.encoded_length += advance;
            }
            // Base256: nothing on unlatch
            _ => {}
        }

        channel.enc_scheme = DX_SCHEME_ASCII;

        match target_scheme {
            DX_SCHEME_ASCII => { /* no latch needed */ }
            DX_SCHEME_C40 => {
                self.push_input_word(channel, DX_CHAR_C40_LATCH);
                self.increment_progress(channel, 12);
            }
            DX_SCHEME_TEXT => {
                self.push_input_word(channel, DX_CHAR_TEXT_LATCH);
                self.increment_progress(channel, 12);
            }
            DX_SCHEME_X12 => {
                self.push_input_word(channel, DX_CHAR_X12_LATCH);
                self.increment_progress(channel, 12);
            }
            DX_SCHEME_EDIFACT => {
                self.push_input_word(channel, DX_CHAR_EDIFACT_LATCH);
                self.increment_progress(channel, 12);
            }
            DX_SCHEME_BASE256 => {
                self.push_input_word(channel, DX_CHAR_BASE256_LATCH);
                self.increment_progress(channel, 12);
                self.push_input_word(channel, dx_randomize_255(0, 2));
                self.increment_progress(channel, 12);
            }
            _ => {}
        }

        channel.enc_scheme = target_scheme;
        channel.first_code_word = channel.current_length - 12;
    }

    fn init_channel(&self, channel: &mut DxChannel, codewords: &[i32]) {
        channel.enc_scheme = DX_SCHEME_ASCII;
        channel.invalid = DX_CHANNEL_VALID;
        channel.input_index = 0;
        channel.input = codewords.to_vec();
    }

    // --- Padding ---

    fn add_pad_chars(&self, buf: &mut [i32], data_word_count: i32, padded_size: i32) -> i32 {
        let mut pad_count = 0;
        let mut dwc = data_word_count;
        if dwc < padded_size {
            pad_count += 1;
            buf[dwc as usize] = DX_CHAR_ASCII_PAD;
            dwc += 1;
        }

        while dwc < padded_size {
            pad_count += 1;
            buf[dwc as usize] = dx_randomize_253(DX_CHAR_ASCII_PAD, dwc + 1);
            dwc += 1;
        }

        pad_count
    }

    // --- Module placement ECC200 ---

    fn module_placement_ecc200(
        &self,
        modules: &mut [i32],
        codewords: &[i32],
        size_idx: i32,
        module_on_color: i32,
    ) {
        let mr = dx_get_symbol_attribute(DX_ATTR_MAPPING_MATRIX_ROWS, size_idx);
        let mc = dx_get_symbol_attribute(DX_ATTR_MAPPING_MATRIX_COLS, size_idx);

        let mut ch = 0;
        let mut row = 4;
        let mut col: i32 = 0;

        loop {
            if row == mr && col == 0 {
                Self::pattern_shape_special1(modules, mr, mc, codewords, ch, module_on_color);
                ch += 1;
            } else if row == mr - 2 && col == 0 && mc % 4 != 0 {
                Self::pattern_shape_special2(modules, mr, mc, codewords, ch, module_on_color);
                ch += 1;
            } else if row == mr - 2 && col == 0 && mc % 8 == 4 {
                Self::pattern_shape_special3(modules, mr, mc, codewords, ch, module_on_color);
                ch += 1;
            } else if row == mr + 4 && col == 2 && mc % 8 == 0 {
                Self::pattern_shape_special4(modules, mr, mc, codewords, ch, module_on_color);
                ch += 1;
            }

            // Upward diagonal
            loop {
                if row < mr
                    && col >= 0
                    && (modules[(row * mc + col) as usize] & DX_MODULE_VISITED) == 0
                {
                    Self::pattern_shape_standard(
                        modules,
                        mr,
                        mc,
                        row,
                        col,
                        codewords,
                        ch,
                        module_on_color,
                    );
                    ch += 1;
                }
                row -= 2;
                col += 2;
                if !(row >= 0 && col < mc) {
                    break;
                }
            }
            row += 1;
            col += 3;

            // Downward diagonal
            loop {
                if row >= 0
                    && col < mc
                    && (modules[(row * mc + col) as usize] & DX_MODULE_VISITED) == 0
                {
                    Self::pattern_shape_standard(
                        modules,
                        mr,
                        mc,
                        row,
                        col,
                        codewords,
                        ch,
                        module_on_color,
                    );
                    ch += 1;
                }
                row += 2;
                col -= 2;
                if !(row < mr && col >= 0) {
                    break;
                }
            }
            row += 3;
            col += 1;

            if !(row < mr || col < mc) {
                break;
            }
        }

        // Fill unvisited corner
        if (modules[(mr * mc - 1) as usize] & DX_MODULE_VISITED) == 0 {
            modules[(mr * mc - 1) as usize] |= module_on_color;
            modules[(mr * mc - mc - 2) as usize] |= module_on_color;
        }
    }

    #[allow(clippy::too_many_arguments)]
    fn pattern_shape_standard(
        modules: &mut [i32],
        mr: i32,
        mc: i32,
        row: i32,
        col: i32,
        cw: &[i32],
        ci: usize,
        color: i32,
    ) {
        Self::place_module(modules, mr, mc, row - 2, col - 2, cw, ci, DX_MASK_BIT1, color);
        Self::place_module(modules, mr, mc, row - 2, col - 1, cw, ci, DX_MASK_BIT2, color);
        Self::place_module(modules, mr, mc, row - 1, col - 2, cw, ci, DX_MASK_BIT3, color);
        Self::place_module(modules, mr, mc, row - 1, col - 1, cw, ci, DX_MASK_BIT4, color);
        Self::place_module(modules, mr, mc, row - 1, col, cw, ci, DX_MASK_BIT5, color);
        Self::place_module(modules, mr, mc, row, col - 2, cw, ci, DX_MASK_BIT6, color);
        Self::place_module(modules, mr, mc, row, col - 1, cw, ci, DX_MASK_BIT7, color);
        Self::place_module(modules, mr, mc, row, col, cw, ci, DX_MASK_BIT8, color);
    }

    fn pattern_shape_special1(
        modules: &mut [i32],
        mr: i32,
        mc: i32,
        cw: &[i32],
        ci: usize,
        color: i32,
    ) {
        Self::place_module(modules, mr, mc, mr - 1, 0, cw, ci, DX_MASK_BIT1, color);
        Self::place_module(modules, mr, mc, mr - 1, 1, cw, ci, DX_MASK_BIT2, color);
        Self::place_module(modules, mr, mc, mr - 1, 2, cw, ci, DX_MASK_BIT3, color);
        Self::place_module(modules, mr, mc, 0, mc - 2, cw, ci, DX_MASK_BIT4, color);
        Self::place_module(modules, mr, mc, 0, mc - 1, cw, ci, DX_MASK_BIT5, color);
        Self::place_module(modules, mr, mc, 1, mc - 1, cw, ci, DX_MASK_BIT6, color);
        Self::place_module(modules, mr, mc, 2, mc - 1, cw, ci, DX_MASK_BIT7, color);
        Self::place_module(modules, mr, mc, 3, mc - 1, cw, ci, DX_MASK_BIT8, color);
    }

    fn pattern_shape_special2(
        modules: &mut [i32],
        mr: i32,
        mc: i32,
        cw: &[i32],
        ci: usize,
        color: i32,
    ) {
        Self::place_module(modules, mr, mc, mr - 3, 0, cw, ci, DX_MASK_BIT1, color);
        Self::place_module(modules, mr, mc, mr - 2, 0, cw, ci, DX_MASK_BIT2, color);
        Self::place_module(modules, mr, mc, mr - 1, 0, cw, ci, DX_MASK_BIT3, color);
        Self::place_module(modules, mr, mc, 0, mc - 4, cw, ci, DX_MASK_BIT4, color);
        Self::place_module(modules, mr, mc, 0, mc - 3, cw, ci, DX_MASK_BIT5, color);
        Self::place_module(modules, mr, mc, 0, mc - 2, cw, ci, DX_MASK_BIT6, color);
        Self::place_module(modules, mr, mc, 0, mc - 1, cw, ci, DX_MASK_BIT7, color);
        Self::place_module(modules, mr, mc, 1, mc - 1, cw, ci, DX_MASK_BIT8, color);
    }

    fn pattern_shape_special3(
        modules: &mut [i32],
        mr: i32,
        mc: i32,
        cw: &[i32],
        ci: usize,
        color: i32,
    ) {
        Self::place_module(modules, mr, mc, mr - 3, 0, cw, ci, DX_MASK_BIT1, color);
        Self::place_module(modules, mr, mc, mr - 2, 0, cw, ci, DX_MASK_BIT2, color);
        Self::place_module(modules, mr, mc, mr - 1, 0, cw, ci, DX_MASK_BIT3, color);
        Self::place_module(modules, mr, mc, 0, mc - 2, cw, ci, DX_MASK_BIT4, color);
        Self::place_module(modules, mr, mc, 0, mc - 1, cw, ci, DX_MASK_BIT5, color);
        Self::place_module(modules, mr, mc, 1, mc - 1, cw, ci, DX_MASK_BIT6, color);
        Self::place_module(modules, mr, mc, 2, mc - 1, cw, ci, DX_MASK_BIT7, color);
        Self::place_module(modules, mr, mc, 3, mc - 1, cw, ci, DX_MASK_BIT8, color);
    }

    fn pattern_shape_special4(
        modules: &mut [i32],
        mr: i32,
        mc: i32,
        cw: &[i32],
        ci: usize,
        color: i32,
    ) {
        Self::place_module(modules, mr, mc, mr - 1, 0, cw, ci, DX_MASK_BIT1, color);
        Self::place_module(modules, mr, mc, mr - 1, mc - 1, cw, ci, DX_MASK_BIT2, color);
        Self::place_module(modules, mr, mc, 0, mc - 3, cw, ci, DX_MASK_BIT3, color);
        Self::place_module(modules, mr, mc, 0, mc - 2, cw, ci, DX_MASK_BIT4, color);
        Self::place_module(modules, mr, mc, 0, mc - 1, cw, ci, DX_MASK_BIT5, color);
        Self::place_module(modules, mr, mc, 1, mc - 3, cw, ci, DX_MASK_BIT6, color);
        Self::place_module(modules, mr, mc, 1, mc - 2, cw, ci, DX_MASK_BIT7, color);
        Self::place_module(modules, mr, mc, 1, mc - 1, cw, ci, DX_MASK_BIT8, color);
    }

    #[allow(clippy::too_many_arguments)]
    fn place_module(
        modules: &mut [i32],
        mr: i32,
        mc: i32,
        mut row: i32,
        mut col: i32,
        codeword: &[i32],
        ci: usize,
        mask: i32,
        color: i32,
    ) {
        if row < 0 {
            row += mr;
            col += 4 - ((mr + 4) % 8);
        }
        if col < 0 {
            col += mc;
            row += 4 - ((mc + 4) % 8);
        }

        let index = (row * mc + col) as usize;
        if (modules[index] & DX_MODULE_ASSIGNED) != 0 {
            if (modules[index] & color) != 0 {
                modules[index] |= mask;
            } else {
                modules[index] &= !mask & 0xFF;
            }
        } else {
            if (codeword[ci] & mask) != 0 {
                modules[index] |= color;
            }
            modules[index] |= DX_MODULE_ASSIGNED;
        }

        modules[index] |= DX_MODULE_VISITED;
    }

    // --- Pattern output ---

    fn print_pattern_raw(&mut self) {
        let sc = self.symbol_cols;
        let sr = self.symbol_rows;
        let si = self.size_idx;

        let message = self.message.as_ref().unwrap();

        let mut raw_data: Vec<Vec<bool>> = Vec::with_capacity(sc as usize);
        for _ in 0..sc {
            raw_data.push(vec![false; sr as usize]);
        }
        for symbol_row in 0..sr {
            for symbol_col in 0..sc {
                let status = message.symbol_module_status(si, symbol_row, symbol_col);
                raw_data[symbol_col as usize][(sr - symbol_row - 1) as usize] =
                    (status & DX_MODULE_ON_BLUE) != 0;
            }
        }
        self.raw_data = Some(raw_data);
    }
}

fn is_digit_byte(v: i32) -> bool {
    (48..=57).contains(&v)
}

// ===========================================================================
// Public DataMatrix struct
// ===========================================================================

/// DataMatrix ECC200 barcode encoder.
pub struct DataMatrix {
    pub base_2d: BarcodeBase2D,
    code_size: i32,
    encode_scheme: i32,
}

impl DataMatrix {
    /// Creates a new DataMatrix encoder with the given output format
    /// (`"png"`, `"jpeg"`, or `"svg"`).
    pub fn new(output_format: &str) -> Self {
        Self {
            base_2d: BarcodeBase2D::new(output_format),
            code_size: DX_SZ_AUTO,
            encode_scheme: DX_SCHEME_ASCII,
        }
    }

    /// Sets the DataMatrix symbol size (use `DX_SZ_*` constants).
    pub fn set_code_size(&mut self, size: i32) {
        self.code_size = size;
    }

    /// Returns the current code size setting.
    pub fn code_size(&self) -> i32 {
        self.code_size
    }

    /// Sets the encoding scheme (use `DX_SCHEME_*` constants).
    pub fn set_encode_scheme(&mut self, scheme: i32) {
        self.encode_scheme = scheme;
    }

    /// Returns the current encoding scheme.
    pub fn encode_scheme(&self) -> i32 {
        self.encode_scheme
    }

    /// Renders the DataMatrix barcode at the given pixel size.
    pub fn draw(&mut self, code: &str, size: i32) -> Result<()> {
        let data = string_to_bytes(code);
        self.draw_bytes(&data, size, size)
    }

    /// Returns the raw DataMatrix pattern as `Vec<Vec<bool>>` indexed `[col][row]`.
    pub fn get_pattern(&mut self, code: &str) -> Result<Vec<Vec<bool>>> {
        let data = string_to_bytes(code);
        self.cal_data_matrix_bytes(&data)
    }

    fn draw_bytes(&mut self, data: &[i32], width: i32, height: i32) -> Result<()> {
        let patt = self.cal_data_matrix_bytes(data)?;

        let pattern_cols = patt.len();
        if pattern_cols == 0 {
            return Err(BarcodeError::EncodingError("empty pattern".into()));
        }
        let pattern_rows = patt[0].len();
        if pattern_rows == 0 {
            return Err(BarcodeError::EncodingError("empty pattern".into()));
        }

        if self.base_2d.base.is_svg_output() {
            self.draw_svg_dm(&patt, width, height)
        } else {
            #[cfg(any(feature = "png", feature = "jpeg"))]
            {
                self.draw_png_dm(&patt, width, height)
            }
            #[cfg(not(any(feature = "png", feature = "jpeg")))]
            Err(BarcodeError::FormatError(
                "PNG/JPEG support requires 'png' or 'jpeg' feature".into(),
            ))
        }
    }

    fn cal_data_matrix_bytes(&self, byte_data: &[i32]) -> Result<Vec<Vec<bool>>> {
        let mut enc = DxEncode::new();
        enc.module_size = 1;
        enc.margin_size = 0;
        enc.size_idx_req = self.code_size;
        enc.scheme = self.encode_scheme;

        enc.encode_data_matrix_raw(byte_data)?;
        enc.raw_data.ok_or_else(|| {
            BarcodeError::EncodingError("encoding produced no pattern data".into())
        })
    }

    // --- SVG rendering ---

    fn draw_svg_dm(
        &mut self,
        patt: &[Vec<bool>],
        width: i32,
        height: i32,
    ) -> Result<()> {
        let pc = patt.len();
        let pr = patt[0].len();

        let ms = if self.base_2d.base.fit_width {
            (width as f64 / pc as f64).min(height as f64 / pr as f64)
        } else {
            let v = (width / pc as i32).min(height / pr as i32) as f64;
            if v < 1.0 { 1.0 } else { v }
        };

        let aw = (ms * pc as f64).ceil() as i32;
        let ah = (ms * pr as f64).ceil() as i32;

        let back = self.base_2d.base.back_color;
        let fore = self.base_2d.base.fore_color;
        let adj = self.base_2d.base.px_adj_black as f64;

        self.base_2d.base.svg_begin(aw, ah);
        self.base_2d
            .base
            .svg_rect(0.0, 0.0, aw as f64, ah as f64, back);

        #[allow(clippy::needless_range_loop)]
        for r in 0..pr {
            for (c, col_data) in patt.iter().enumerate().take(pc) {
                if col_data[r] {
                    let dw = (ms + adj + 0.5).max(0.0);
                    let dh = (ms + adj + 0.5).max(0.0);
                    self.base_2d
                        .base
                        .svg_rect(c as f64 * ms, r as f64 * ms, dw, dh, fore);
                }
            }
        }

        self.base_2d.base.svg_end();
        Ok(())
    }

    // --- PNG rendering ---

    #[cfg(any(feature = "png", feature = "jpeg"))]
    fn draw_png_dm(
        &mut self,
        patt: &[Vec<bool>],
        width: i32,
        height: i32,
    ) -> Result<()> {
        let pc = patt.len();
        let pr = patt[0].len();

        let ms = if self.base_2d.base.fit_width {
            (width as f64 / pc as f64).min(height as f64 / pr as f64)
        } else {
            let v = (width / pc as i32).min(height / pr as i32) as f64;
            if v < 1.0 { 1.0 } else { v }
        };

        let aw = (ms * pc as f64).ceil() as i32;
        let ah = (ms * pr as f64).ceil() as i32;

        let bc = self.base_2d.base.back_color;
        let mut img = image::RgbaImage::from_pixel(
            aw as u32,
            ah as u32,
            image::Rgba([bc.r, bc.g, bc.b, bc.a]),
        );

        let fc = self.base_2d.base.fore_color;
        let fore_pixel = image::Rgba([fc.r, fc.g, fc.b, fc.a]);

        #[allow(clippy::needless_range_loop)]
        for r in 0..pr {
            for (c, col_data) in patt.iter().enumerate().take(pc) {
                if col_data[r] {
                    let x0 = (c as f64 * ms) as i32;
                    let y0 = (r as f64 * ms) as i32;
                    let x1 = ((c + 1) as f64 * ms) as i32;
                    let y1 = ((r + 1) as f64 * ms) as i32;
                    crate::base_1d::fill_rect(&mut img, x0, y0, x1, y1, fore_pixel);
                }
            }
        }

        // Trial mode watermark
        if crate::product_info::is_trial_mode() {
            #[cfg(feature = "font")]
            crate::base_1d::draw_sample_overlay_png(&mut img, 0, 0, aw, ah);
        }

        self.base_2d.base.encode_image(&img)
    }
}

fn string_to_bytes(s: &str) -> Vec<i32> {
    s.bytes().map(|b| b as i32).collect()
}

// ===========================================================================
// Tests
// ===========================================================================

#[cfg(test)]
mod tests {
    use super::*;

    // --- GF(256) helper tests ---

    #[test]
    fn test_gf_product_zero() {
        assert_eq!(dx_gf_product(0, 5), 0);
        assert_eq!(dx_gf_product(5, 0), 0);
        assert_eq!(dx_gf_product(0, 0), 0);
    }

    #[test]
    fn test_gf_product_nonzero() {
        assert_eq!(dx_gf_product(1, 1), 1);
        assert_eq!(dx_gf_product(2, 2), 4);
        let p = dx_gf_product(100, 200);
        assert!(p > 0 && p < 256);
    }

    #[test]
    fn test_gf_sum() {
        assert_eq!(dx_gf_sum(0, 0), 0);
        assert_eq!(dx_gf_sum(0xFF, 0xFF), 0);
        assert_eq!(dx_gf_sum(0xAA, 0x55), 0xFF);
    }

    #[test]
    fn test_gf_doublify() {
        assert_eq!(dx_gf_doublify(0, 5), 0);
        assert_eq!(dx_gf_doublify(5, 0), 5);
    }

    // --- Symbol attribute tests ---

    #[test]
    fn test_symbol_attribute_rows_cols() {
        assert_eq!(dx_get_symbol_attribute(DX_ATTR_SYMBOL_ROWS, DX_SZ_10X10), 10);
        assert_eq!(dx_get_symbol_attribute(DX_ATTR_SYMBOL_COLS, DX_SZ_10X10), 10);
        assert_eq!(dx_get_symbol_attribute(DX_ATTR_SYMBOL_ROWS, DX_SZ_144X144), 144);
        assert_eq!(dx_get_symbol_attribute(DX_ATTR_SYMBOL_COLS, DX_SZ_144X144), 144);
    }

    #[test]
    fn test_symbol_attribute_rectangular() {
        assert_eq!(dx_get_symbol_attribute(DX_ATTR_SYMBOL_ROWS, DX_SZ_8X18), 8);
        assert_eq!(dx_get_symbol_attribute(DX_ATTR_SYMBOL_COLS, DX_SZ_8X18), 18);
    }

    #[test]
    fn test_symbol_attribute_invalid() {
        assert_eq!(dx_get_symbol_attribute(DX_ATTR_SYMBOL_ROWS, -1), DX_UNDEFINED);
        assert_eq!(dx_get_symbol_attribute(DX_ATTR_SYMBOL_ROWS, 30), DX_UNDEFINED);
    }

    #[test]
    fn test_mapping_matrix_rows_cols() {
        let mr = dx_get_symbol_attribute(DX_ATTR_MAPPING_MATRIX_ROWS, DX_SZ_10X10);
        let mc = dx_get_symbol_attribute(DX_ATTR_MAPPING_MATRIX_COLS, DX_SZ_10X10);
        assert_eq!(mr, 8);
        assert_eq!(mc, 8);
    }

    #[test]
    fn test_block_data_size() {
        let ds = dx_get_block_data_size(DX_SZ_10X10, 0);
        assert_eq!(ds, 3);
    }

    #[test]
    fn test_block_data_size_144() {
        // 144x144 has special handling for first 8 blocks
        let ds0 = dx_get_block_data_size(DX_SZ_144X144, 0);
        let ds8 = dx_get_block_data_size(DX_SZ_144X144, 8);
        assert_eq!(ds0, ds8 + 1);
    }

    // --- Randomize tests ---

    #[test]
    fn test_randomize_253() {
        let r = dx_randomize_253(129, 1);
        assert!((0..=255).contains(&r));
    }

    #[test]
    fn test_randomize_255_roundtrip() {
        for pos in 1..10 {
            let original = 42;
            let randomized = dx_randomize_255(original, pos);
            let unrandomized = dx_un_randomize_255(randomized, pos);
            assert_eq!(unrandomized, original);
        }
    }

    // --- Triplet / quadruplet tests ---

    #[test]
    fn test_triplet_values() {
        let tv = dx_get_triplet_values(0, 2);
        assert_eq!(tv[0], 0);
        assert_eq!(tv[1], 0);
        assert_eq!(tv[2], 1);
    }

    #[test]
    fn test_quadruplet_values() {
        let qv = dx_get_quadruplet_values(0xFF, 0xFF, 0xFF);
        assert_eq!(qv[0], 63);
        assert_eq!(qv[1], 63);
        assert_eq!(qv[2], 63);
        assert_eq!(qv[3], 63);
    }

    // --- Reed-Solomon ECC tests ---

    #[test]
    fn test_reed_solomon_ecc_size0() {
        let sdw = dx_get_symbol_attribute(DX_ATTR_SYMBOL_DATA_WORDS, DX_SZ_10X10);
        let sew = dx_get_symbol_attribute(DX_ATTR_SYMBOL_ERROR_WORDS, DX_SZ_10X10);
        let mut code = vec![0i32; (sdw + sew) as usize];
        code[0] = 142; // 'A' + 1
        code[1] = 143; // 'B' + 1
        code[2] = 144; // 'C' + 1
        dx_gen_reed_sol_ecc(&mut code, DX_SZ_10X10);
        // Error words should be populated
        let ecc_portion = &code[sdw as usize..];
        assert!(ecc_portion.iter().any(|&v| v != 0));
    }

    // --- Pattern generation tests ---

    #[test]
    fn test_encode_basic_pattern() {
        let mut dm = DataMatrix::new("svg");
        let patt = dm.get_pattern("ABC").unwrap();
        assert!(!patt.is_empty());
        assert!(!patt[0].is_empty());
    }

    #[test]
    fn test_encode_single_char() {
        let mut dm = DataMatrix::new("svg");
        let patt = dm.get_pattern("A").unwrap();
        assert!(!patt.is_empty());
    }

    #[test]
    fn test_encode_digits() {
        let mut dm = DataMatrix::new("svg");
        let patt = dm.get_pattern("1234567890").unwrap();
        assert!(!patt.is_empty());
    }

    #[test]
    fn test_encode_with_fixed_size() {
        let mut dm = DataMatrix::new("svg");
        dm.set_code_size(DX_SZ_20X20);
        let patt = dm.get_pattern("Hello").unwrap();
        // 20x20 symbol
        assert_eq!(patt.len(), 20);
        assert_eq!(patt[0].len(), 20);
    }

    #[test]
    fn test_encode_rectangular() {
        let mut dm = DataMatrix::new("svg");
        dm.set_code_size(DX_SZ_8X18);
        let patt = dm.get_pattern("Hi").unwrap();
        assert_eq!(patt.len(), 18);
        assert_eq!(patt[0].len(), 8);
    }

    #[test]
    fn test_draw_svg() {
        let mut dm = DataMatrix::new("svg");
        dm.draw("Test123", 200).unwrap();
        let svg = dm.base_2d.base.get_svg().unwrap();
        assert!(svg.contains("<svg"));
        assert!(svg.contains("</svg>"));
        assert!(svg.contains("<rect"));
    }

    #[cfg(any(feature = "png", feature = "jpeg"))]
    #[test]
    fn test_draw_png() {
        let mut dm = DataMatrix::new("png");
        dm.draw("Test123", 200).unwrap();
        let mem = dm.base_2d.base.get_image_memory();
        assert!(!mem.is_empty());
        // PNG magic bytes
        assert_eq!(&mem[0..4], &[0x89, 0x50, 0x4E, 0x47]);
    }

    #[test]
    fn test_draw_svg_fit_width() {
        let mut dm = DataMatrix::new("svg");
        dm.base_2d.base.set_fit_width(true);
        dm.draw("FitTest", 200).unwrap();
        let svg = dm.base_2d.base.get_svg().unwrap();
        assert!(svg.contains("<svg"));
    }

    #[test]
    fn test_encode_empty_fails() {
        let mut dm = DataMatrix::new("svg");
        assert!(dm.draw("", 200).is_err());
    }

    #[test]
    fn test_code_size_accessors() {
        let mut dm = DataMatrix::new("svg");
        assert_eq!(dm.code_size(), DX_SZ_AUTO);
        dm.set_code_size(DX_SZ_32X32);
        assert_eq!(dm.code_size(), DX_SZ_32X32);
    }

    #[test]
    fn test_encode_scheme_accessors() {
        let mut dm = DataMatrix::new("svg");
        assert_eq!(dm.encode_scheme(), DX_SCHEME_ASCII);
        dm.set_encode_scheme(DX_SCHEME_C40);
        assert_eq!(dm.encode_scheme(), DX_SCHEME_C40);
    }

    #[test]
    fn test_c40_encoding() {
        let mut dm = DataMatrix::new("svg");
        dm.set_encode_scheme(DX_SCHEME_C40);
        let patt = dm.get_pattern("ABCDEF").unwrap();
        assert!(!patt.is_empty());
    }

    #[test]
    fn test_text_encoding() {
        let mut dm = DataMatrix::new("svg");
        dm.set_encode_scheme(DX_SCHEME_TEXT);
        let patt = dm.get_pattern("abcdef").unwrap();
        assert!(!patt.is_empty());
    }

    #[test]
    fn test_base256_encoding() {
        let mut dm = DataMatrix::new("svg");
        dm.set_encode_scheme(DX_SCHEME_BASE256);
        let patt = dm.get_pattern("Hello").unwrap();
        assert!(!patt.is_empty());
    }

    #[test]
    fn test_edifact_encoding() {
        let mut dm = DataMatrix::new("svg");
        dm.set_encode_scheme(DX_SCHEME_EDIFACT);
        let patt = dm.get_pattern("ABCDEF").unwrap();
        assert!(!patt.is_empty());
    }

    #[test]
    fn test_x12_encoding() {
        let mut dm = DataMatrix::new("svg");
        dm.set_encode_scheme(DX_SCHEME_X12);
        let patt = dm.get_pattern("ABC123").unwrap();
        assert!(!patt.is_empty());
    }

    #[test]
    fn test_auto_size_selection() {
        // Short data -> small symbol
        let mut dm = DataMatrix::new("svg");
        let patt_small = dm.get_pattern("A").unwrap();

        let mut dm2 = DataMatrix::new("svg");
        let patt_big = dm2.get_pattern("ABCDEFGHIJKLMNOPQRSTUVWXYZ").unwrap();

        assert!(patt_big.len() > patt_small.len());
    }

    #[test]
    fn test_digit_pair_compression() {
        // Digit pairs should compress in ASCII mode
        let mut dm = DataMatrix::new("svg");
        let patt = dm.get_pattern("12").unwrap();
        assert!(!patt.is_empty());
        // 10x10 is the smallest
        assert_eq!(patt.len(), 10);
    }

    #[test]
    fn test_pattern_consistency() {
        // Same input should produce same pattern
        let mut dm1 = DataMatrix::new("svg");
        let patt1 = dm1.get_pattern("Consistency").unwrap();

        let mut dm2 = DataMatrix::new("svg");
        let patt2 = dm2.get_pattern("Consistency").unwrap();

        assert_eq!(patt1.len(), patt2.len());
        for c in 0..patt1.len() {
            assert_eq!(patt1[c].len(), patt2[c].len());
            for r in 0..patt1[c].len() {
                assert_eq!(patt1[c][r], patt2[c][r]);
            }
        }
    }

    #[test]
    fn test_finder_pattern_present() {
        let mut dm = DataMatrix::new("svg");
        let patt = dm.get_pattern("X").unwrap();
        let cols = patt.len();
        let rows = patt[0].len();

        // Bottom row (row index = rows-1) should be solid (finder pattern L-shape)
        let mut all_on = true;
        for c in 0..cols {
            if !patt[c][rows - 1] {
                all_on = false;
                break;
            }
        }
        assert!(all_on, "bottom edge of finder pattern should be solid");

        // Left column (col 0) should be solid
        let mut left_on = true;
        for r in 0..rows {
            if !patt[0][r] {
                left_on = false;
                break;
            }
        }
        assert!(left_on, "left edge of finder pattern should be solid");
    }
}
