//! Multi-row rendering for DataBar barcodes.
//! Provides SVG and PNG rendering of run-length encoded pattern rows.

use crate::base::BarcodeBase;
use crate::error::Result;
use crate::product_info::is_trial_mode;

/// Renders a multi-row pattern-based barcode as SVG.
/// patterns: run-length encoded strings for each row (digit = width, leading "0" means starts with space)
/// row_heights: height in X units for each row (-1 = variable/fills remaining space)
pub fn draw_multi_row_svg(
    b: &mut BarcodeBase,
    patterns: &[String],
    row_heights: &[i32],
    width: i32,
    height: i32,
) -> Result<()> {
    if patterns.is_empty() {
        return Err(crate::error::BarcodeError::EncodingError(
            "no patterns to render".into(),
        ));
    }

    let row_px_heights = calculate_row_heights(row_heights, height);

    b.svg_begin(width, height);
    b.svg_rect(0.0, 0.0, width as f64, height as f64, b.back_color);

    let mut y = 0;
    for (ri, pat) in patterns.iter().enumerate() {
        if ri >= row_px_heights.len() {
            break;
        }
        let rh = row_px_heights[ri];
        draw_pattern_row_svg(b, pat, 0, y, width, rh);
        y += rh;
    }

    if is_trial_mode() {
        b.svg_sample_overlay(0, 0, width, height);
    }
    b.svg_stream.push_str("</svg>\n");
    Ok(())
}

/// Renders a multi-row pattern-based barcode as PNG.
#[cfg(any(feature = "png", feature = "jpeg"))]
pub fn draw_multi_row_png(
    b: &mut BarcodeBase,
    patterns: &[String],
    row_heights: &[i32],
    width: i32,
    height: i32,
) -> Result<()> {
    if patterns.is_empty() {
        return Err(crate::error::BarcodeError::EncodingError(
            "no patterns to render".into(),
        ));
    }

    let row_px_heights = calculate_row_heights(row_heights, height);

    let mut img = image::RgbaImage::from_pixel(
        width as u32,
        height as u32,
        image::Rgba([b.back_color.r, b.back_color.g, b.back_color.b, b.back_color.a]),
    );

    let fore = image::Rgba([
        b.fore_color.r,
        b.fore_color.g,
        b.fore_color.b,
        b.fore_color.a,
    ]);

    let mut y = 0;
    for (ri, pat) in patterns.iter().enumerate() {
        if ri >= row_px_heights.len() {
            break;
        }
        let rh = row_px_heights[ri];
        draw_pattern_row_png(&mut img, pat, 0, y, width, rh, fore);
        y += rh;
    }

    if is_trial_mode() {
        #[cfg(feature = "font")]
        crate::base_1d::draw_sample_overlay_png(&mut img, 0, 0, width, height);
    }

    b.encode_image(&img)
}

fn calculate_row_heights(row_heights: &[i32], height: i32) -> Vec<i32> {
    let mut total_fixed = 0;
    let mut variable_count = 0;
    for &h in row_heights {
        if h > 0 {
            total_fixed += h;
        } else {
            variable_count += 1;
        }
    }

    let mut row_px_heights = vec![0i32; row_heights.len()];
    if variable_count > 0 {
        let variable_height = ((height - total_fixed) / variable_count).max(1);
        for (i, &h) in row_heights.iter().enumerate() {
            row_px_heights[i] = if h > 0 { h } else { variable_height };
        }
    } else {
        let scale = height as f64 / total_fixed as f64;
        for (i, &h) in row_heights.iter().enumerate() {
            row_px_heights[i] = (h as f64 * scale).round() as i32;
        }
    }
    row_px_heights
}

fn draw_pattern_row_svg(
    b: &mut BarcodeBase,
    pattern: &str,
    x: i32,
    y: i32,
    width: i32,
    height: i32,
) {
    if pattern.is_empty() {
        return;
    }

    let bytes = pattern.as_bytes();
    let mut start_idx = 0;
    let mut is_bar = true;
    if bytes[0] == b'0' {
        is_bar = false;
        start_idx = 1;
    }

    let mut total_modules = 0;
    for &b in &bytes[start_idx..] {
        total_modules += (b - b'0') as i32;
    }
    if total_modules <= 0 {
        return;
    }

    let unit_w = width as f64 / total_modules as f64;
    let mut accum = 0.0f64;
    let mut bar = is_bar;

    for &byte in &bytes[start_idx..] {
        let modules = (byte - b'0') as i32;
        let x1 = (x as f64 + accum * unit_w).round();
        accum += modules as f64;
        let x2 = (x as f64 + accum * unit_w).round();
        if bar && x2 > x1 {
            b.svg_rect(x1, y as f64, x2 - x1, height as f64, b.fore_color);
        }
        bar = !bar;
    }
}

#[cfg(any(feature = "png", feature = "jpeg"))]
fn draw_pattern_row_png(
    img: &mut image::RgbaImage,
    pattern: &str,
    x: i32,
    y: i32,
    width: i32,
    height: i32,
    fore_color: image::Rgba<u8>,
) {
    if pattern.is_empty() {
        return;
    }

    let bytes = pattern.as_bytes();
    let mut start_idx = 0;
    let mut is_bar = true;
    if bytes[0] == b'0' {
        is_bar = false;
        start_idx = 1;
    }

    let mut total_modules = 0;
    for &b in &bytes[start_idx..] {
        total_modules += (b - b'0') as i32;
    }
    if total_modules <= 0 {
        return;
    }

    let unit_w = width as f64 / total_modules as f64;
    let mut accum = 0.0f64;
    let mut bar = is_bar;

    for &b in &bytes[start_idx..] {
        let modules = (b - b'0') as i32;
        let x1 = (x as f64 + accum * unit_w).round() as i32;
        accum += modules as f64;
        let x2 = (x as f64 + accum * unit_w).round() as i32;
        if bar && x2 > x1 {
            crate::base_1d::fill_rect(img, x1, y, x2, y + height, fore_color);
        }
        bar = !bar;
    }
}
