use crate::base_1d::{BarcodeBase1D, Encoder1D};
use crate::error::{BarcodeError, Result};

const CODE93_CHARS: &str = "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ-. $/+%<>?!";
const CODE93_PATTERNS: &[&str] = &[
    "131112", "111213", "111312", "111411", "121113", "121212", "121311",
    "111114", "131211", "141111", "211113", "211212", "211311", "221112",
    "221211", "231111", "112113", "112212", "112311", "122112", "132111",
    "111123", "111222", "111321", "121122", "131121", "212112", "212211",
    "211122", "211221", "221121", "222111", "112122", "112221", "122121",
    "123111", "121131", "311112", "311211", "321111", "112131", "113121",
    "211131", "121221", "312111", "311121", "122211",
];
const CODE93_START: &str = "111141";
const CODE93_STOP: &str = "1111411";

/// Code93 barcode encoder.
pub struct Code93 {
    pub base_1d: BarcodeBase1D,
}

impl Code93 {
    pub fn new(output_format: &str) -> Self {
        Self {
            base_1d: BarcodeBase1D::new(output_format),
        }
    }

    pub fn encode(&self, code: &str) -> Result<Vec<i32>> {
        Code93Encoder.encode(code)
    }

    pub fn draw(&mut self, code: &str, width: i32, height: i32) -> Result<()> {
        self.base_1d.draw(code, width, height, &Code93Encoder)
    }
}

struct Code93Encoder;

impl Encoder1D for Code93Encoder {
    fn encode(&self, code: &str) -> Result<Vec<i32>> {
        encode_code93(code)
    }
}

fn code93_modulus47(code: &str, weight: usize) -> Result<usize> {
    let mut total = 0usize;
    let bytes = code.as_bytes();
    for i in 0..bytes.len() {
        let value = CODE93_CHARS
            .find(bytes[i] as char)
            .ok_or_else(|| BarcodeError::InvalidCharacter {
                ch: bytes[i] as char,
                symbology: "CODE93".into(),
            })?;
        let mut pos_weight = bytes.len() - i;
        if pos_weight > weight {
            pos_weight = ((pos_weight - 1) % weight) + 1;
        }
        total += value * pos_weight;
    }
    Ok(total % 47)
}

pub fn encode_code93(code: &str) -> Result<Vec<i32>> {
    if code.is_empty() {
        return Err(BarcodeError::EmptyCode);
    }

    let code = code.to_uppercase();

    for ch in code.chars() {
        if CODE93_CHARS.find(ch).is_none() {
            return Err(BarcodeError::InvalidCharacter {
                ch,
                symbology: "CODE93".into(),
            });
        }
    }

    let check_c = code93_modulus47(&code, 20)?;
    let code_with_c = format!(
        "{}{}",
        code,
        CODE93_CHARS.as_bytes()[check_c] as char
    );

    let check_k = code93_modulus47(&code_with_c, 15)?;

    let mut result = Vec::new();

    // Start pattern
    for ch in CODE93_START.chars() {
        result.push((ch as u8 - b'0') as i32);
    }

    // Data
    for ch in code.chars() {
        let index = CODE93_CHARS.find(ch).unwrap();
        for p in CODE93_PATTERNS[index].chars() {
            result.push((p as u8 - b'0') as i32);
        }
    }

    // Check digit C
    for p in CODE93_PATTERNS[check_c].chars() {
        result.push((p as u8 - b'0') as i32);
    }

    // Check digit K
    for p in CODE93_PATTERNS[check_k].chars() {
        result.push((p as u8 - b'0') as i32);
    }

    // Stop pattern
    for ch in CODE93_STOP.chars() {
        result.push((ch as u8 - b'0') as i32);
    }

    Ok(result)
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_encode_basic() {
        let pattern = encode_code93("TEST93").unwrap();
        assert!(!pattern.is_empty());
    }

    #[test]
    fn test_encode_empty() {
        assert!(encode_code93("").is_err());
    }

    #[test]
    fn test_encode_digits() {
        let pattern = encode_code93("12345").unwrap();
        assert!(!pattern.is_empty());
    }
}
