use crate::base_1d::{BarcodeBase1D, Encoder1D};
use crate::error::{BarcodeError, Result};

const CODE39_CHARS: &str = "1234567890ABCDEFGHIJKLMNOPQRSTUVWXYZ-. *$/+%";
const CODE39_PTN: &[u16] = &[
    0x121, 0x061, 0x160, 0x031, 0x130, 0x070, 0x025, 0x124, 0x064, 0x034,
    0x109, 0x049, 0x148, 0x019, 0x118, 0x058, 0x00d, 0x10c, 0x04c, 0x01c,
    0x103, 0x043, 0x142, 0x013, 0x112, 0x052, 0x007, 0x106, 0x046, 0x016,
    0x181, 0x0c1, 0x1c0, 0x091, 0x190, 0x0d0, 0x085, 0x184, 0x0c4, 0x094,
    0x0a8, 0x0a2, 0x08a, 0x02a,
];

/// Code39 barcode encoder.
pub struct Code39 {
    pub base_1d: BarcodeBase1D,
    pub show_start_stop: bool,
}

impl Code39 {
    pub fn new(output_format: &str) -> Self {
        Self {
            base_1d: BarcodeBase1D::new(output_format),
            show_start_stop: true,
        }
    }

    pub fn encode(&self, code: &str) -> Result<Vec<i32>> {
        let encoder = Code39Encoder {
            show_start_stop: self.show_start_stop,
        };
        encoder.encode(code)
    }

    pub fn draw(&mut self, code: &str, width: i32, height: i32) -> Result<()> {
        let encoder = Code39Encoder {
            show_start_stop: self.show_start_stop,
        };
        self.base_1d.draw(code, width, height, &encoder)
    }
}

struct Code39Encoder {
    show_start_stop: bool,
}

impl Encoder1D for Code39Encoder {
    fn encode(&self, code: &str) -> Result<Vec<i32>> {
        encode_code39(code, self.show_start_stop)
    }
}

pub fn encode_code39(code: &str, _show_start_stop: bool) -> Result<Vec<i32>> {
    if code.is_empty() {
        return Err(BarcodeError::EmptyCode);
    }

    let mut valid_code = String::new();
    for ch in code.chars() {
        let upper: String = ch.to_uppercase().collect();
        if upper == "*" {
            return Err(BarcodeError::InvalidCharacter {
                ch: '*',
                symbology: "CODE39".into(),
            });
        }
        let pos = CODE39_CHARS.find(&upper[..]);
        match pos {
            Some(39) => {
                // 39 is '*'
                return Err(BarcodeError::InvalidCharacter {
                    ch,
                    symbology: "CODE39".into(),
                });
            }
            Some(_) => valid_code.push_str(&upper),
            None => {
                return Err(BarcodeError::InvalidCharacter {
                    ch,
                    symbology: "CODE39".into(),
                });
            }
        }
    }

    let full_code = format!("*{}*", valid_code);

    let mut result = Vec::new();
    for ch in full_code.chars() {
        if !result.is_empty() {
            result.push(1); // inter-character gap
        }
        let char_index = CODE39_CHARS
            .find(ch)
            .ok_or_else(|| BarcodeError::InvalidCharacter {
                ch,
                symbology: "CODE39".into(),
            })?;
        let bit_pattern = CODE39_PTN[char_index];

        // 9-bit pattern: MSB first, 1=wide(3), 0=narrow(1)
        let mask: u16 = 0x100;
        for i in 0..9 {
            if (bit_pattern & (mask >> i)) != 0 {
                result.push(3);
            } else {
                result.push(1);
            }
        }
    }

    Ok(result)
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_encode_basic() {
        let pattern = encode_code39("123", true).unwrap();
        assert!(!pattern.is_empty());
        // Pattern should start with '*' encoding and end with '*' encoding
    }

    #[test]
    fn test_encode_empty() {
        assert!(encode_code39("", true).is_err());
    }

    #[test]
    fn test_encode_invalid_star() {
        assert!(encode_code39("A*B", true).is_err());
    }

    #[test]
    fn test_encode_alpha() {
        let pattern = encode_code39("ABCDEF", true).unwrap();
        assert!(!pattern.is_empty());
    }
}
