use crate::base_1d::{BarcodeBase1D, Encoder1D};
use crate::error::{BarcodeError, Result};
use std::collections::HashMap;

/// Code128 mode constants.
pub const CODE128_AUTO: i32 = 0;
pub const CODE128_CODE_A: i32 = 1;
pub const CODE128_CODE_B: i32 = 2;
pub const CODE128_CODE_C: i32 = 3;

pub const CODE128_PATTERNS: &[&str] = &[
    "212222", "222122", "222221", "121223", "121322", "131222", "122213",
    "122312", "132212", "221213", "221312", "231212", "112232", "122132",
    "122231", "113222", "123122", "123221", "223211", "221132", "221231",
    "213212", "223112", "312131", "311222", "321122", "321221", "312212",
    "322112", "322211", "212123", "212321", "232121", "111323", "131123",
    "131321", "112313", "132113", "132311", "211313", "231113", "231311",
    "112133", "112331", "132131", "113123", "113321", "133121", "313121",
    "211331", "231131", "213113", "213311", "213131", "311123", "311321",
    "331121", "312113", "312311", "332111", "314111", "221411", "431111",
    "111224", "111422", "121124", "121421", "141122", "141221", "112214",
    "112412", "122114", "122411", "142112", "142211", "241211", "221114",
    "413111", "241112", "134111", "111242", "121142", "121241", "114212",
    "124112", "124211", "411212", "421112", "421211", "212141", "214121",
    "412121", "111143", "111341", "131141", "114113", "114311", "411113",
    "411311", "113141", "114131", "311141", "411131",
    "211412", "211214", "211232", "2331112", // 103-106: Start A/B/C, Stop
];

const DP_INF: i32 = 999_999_999;
const DP_SET_A: i32 = 0;
const DP_SET_B: i32 = 1;
const _DP_SET_C: i32 = 2;

/// Code128 barcode encoder with AUTO/A/B/C mode support.
pub struct Code128 {
    pub base_1d: BarcodeBase1D,
    pub code_mode: i32,
}

impl Code128 {
    pub fn new(output_format: &str) -> Self {
        Self {
            base_1d: BarcodeBase1D::new(output_format),
            code_mode: CODE128_AUTO,
        }
    }

    pub fn encode(&self, code: &str) -> Result<Vec<i32>> {
        let encoder = Code128Encoder {
            code_mode: self.code_mode,
        };
        encoder.encode(code)
    }

    pub fn draw(&mut self, code: &str, width: i32, height: i32) -> Result<()> {
        let encoder = Code128Encoder {
            code_mode: self.code_mode,
        };
        self.base_1d.draw(code, width, height, &encoder)
    }
}

struct Code128Encoder {
    code_mode: i32,
}

impl Encoder1D for Code128Encoder {
    fn encode(&self, code: &str) -> Result<Vec<i32>> {
        if code.is_empty() {
            return Err(BarcodeError::EmptyCode);
        }
        if self.code_mode == CODE128_AUTO {
            encode_auto(code)
        } else {
            encode_single_set(code, self.code_mode)
        }
    }
}

fn code128_a_index(c: u8) -> i32 {
    if c <= 95 { c as i32 } else { -1 }
}

fn code128_b_index(c: u8) -> i32 {
    if !(32..=126).contains(&c) { -1 } else { c as i32 - 32 }
}

fn code128_c_index(s: &[u8], i: usize) -> i32 {
    if i + 1 >= s.len() {
        return -1;
    }
    let c1 = s[i];
    let c2 = s[i + 1];
    if !c1.is_ascii_digit() || !c2.is_ascii_digit() {
        return -1;
    }
    (c1 - b'0') as i32 * 10 + (c2 - b'0') as i32
}

struct DpResult {
    cost: i32,
    enc: Vec<i32>,
}

fn solve_dp(
    s: &[u8],
    i: usize,
    iset: i32,
    memo: &mut HashMap<i32, DpResult>,
) -> Result<DpResult> {
    if i >= s.len() {
        return Ok(DpResult {
            cost: 0,
            enc: vec![],
        });
    }
    let key = ((i as i32) << 2) + iset;
    if let Some(r) = memo.get(&key) {
        return Ok(DpResult {
            cost: r.cost,
            enc: r.enc.clone(),
        });
    }

    let mut best_cost = DP_INF;
    let mut best_enc: Vec<i32> = vec![];
    let c = s[i];

    // Stay in current set
    if iset == DP_SET_A {
        let idx = code128_a_index(c);
        if idx >= 0 {
            let nxt = solve_dp(s, i + 1, iset, memo)?;
            let cost = 1 + nxt.cost;
            if cost < best_cost {
                best_cost = cost;
                best_enc = std::iter::once(idx).chain(nxt.enc).collect();
            }
        }
    } else if iset == DP_SET_B {
        let idx = code128_b_index(c);
        if idx >= 0 {
            let nxt = solve_dp(s, i + 1, iset, memo)?;
            let cost = 1 + nxt.cost;
            if cost < best_cost {
                best_cost = cost;
                best_enc = std::iter::once(idx).chain(nxt.enc).collect();
            }
        }
    } else {
        // dpSetC
        let idx = code128_c_index(s, i);
        if idx >= 0 {
            let nxt = solve_dp(s, i + 2, iset, memo)?;
            let cost = 1 + nxt.cost;
            if cost < best_cost {
                best_cost = cost;
                best_enc = std::iter::once(idx).chain(nxt.enc).collect();
            }
        }
    }

    // Switch to different set
    let switch_vals: [[i32; 3]; 3] = [
        [0, 100, 99],  // from A: to A(unused), to B=100, to C=99
        [101, 0, 99],  // from B: to A=101, to B(unused), to C=99
        [101, 100, 0], // from C: to A=101, to B=100, to C(unused)
    ];

    for st in 0..3i32 {
        if st == iset {
            continue;
        }
        let sv = switch_vals[iset as usize][st as usize];

        if st == DP_SET_A {
            let idx = code128_a_index(c);
            if idx >= 0 {
                let nxt = solve_dp(s, i + 1, st, memo)?;
                let cost = 2 + nxt.cost;
                if cost < best_cost {
                    best_cost = cost;
                    best_enc = [sv, idx].iter().copied().chain(nxt.enc).collect();
                }
            }
        } else if st == DP_SET_B {
            let idx = code128_b_index(c);
            if idx >= 0 {
                let nxt = solve_dp(s, i + 1, st, memo)?;
                let cost = 2 + nxt.cost;
                if cost < best_cost {
                    best_cost = cost;
                    best_enc = [sv, idx].iter().copied().chain(nxt.enc).collect();
                }
            }
        } else {
            // dpSetC
            let idx = code128_c_index(s, i);
            if idx >= 0 {
                let nxt = solve_dp(s, i + 2, st, memo)?;
                let cost = 2 + nxt.cost;
                if cost < best_cost {
                    best_cost = cost;
                    best_enc = [sv, idx].iter().copied().chain(nxt.enc).collect();
                }
            }
        }
    }

    if best_cost == DP_INF {
        return Err(BarcodeError::EncodingError(format!(
            "DP encode error at pos={}",
            i
        )));
    }

    let result = DpResult {
        cost: best_cost,
        enc: best_enc.clone(),
    };
    memo.insert(key, result);
    Ok(DpResult {
        cost: best_cost,
        enc: best_enc,
    })
}

fn encode_single_set(s: &str, fixed_set: i32) -> Result<Vec<i32>> {
    let start_val = 102 + fixed_set; // A=103, B=104, C=105
    let mut vals = vec![start_val];
    let bytes = s.as_bytes();

    let mut i = 0;
    while i < bytes.len() {
        let ch = bytes[i];
        match fixed_set {
            CODE128_CODE_C => {
                let cc = code128_c_index(bytes, i);
                if cc < 0 {
                    return Err(BarcodeError::EncodingError(format!(
                        "not 2-digit at pos={}",
                        i
                    )));
                }
                vals.push(cc);
                i += 2;
            }
            CODE128_CODE_A => {
                let idx = code128_a_index(ch);
                if idx < 0 {
                    return Err(BarcodeError::EncodingError(
                        "invalid char for CODE_A".into(),
                    ));
                }
                vals.push(idx);
                i += 1;
            }
            _ => {
                // Code128CodeB
                let idx = code128_b_index(ch);
                if idx < 0 {
                    return Err(BarcodeError::EncodingError(
                        "invalid char for CODE_B".into(),
                    ));
                }
                vals.push(idx);
                i += 1;
            }
        }
    }

    // Checksum (Modulus 103)
    let mut check_sum = vals[0];
    for (j, &v) in vals.iter().enumerate().skip(1) {
        check_sum += v * j as i32;
    }
    check_sum %= 103;
    vals.push(check_sum);
    vals.push(106); // Stop

    expand_code128_pattern(&vals)
}

fn encode_auto(s: &str) -> Result<Vec<i32>> {
    let bytes = s.as_bytes();

    struct Candidate {
        start_val: i32,
        cost: i32,
        enc: Vec<i32>,
    }

    let mut candidates = Vec::new();
    for iset in 0..3i32 {
        let start_val = 103 + iset;
        let mut memo = HashMap::new();
        if let Ok(r) = solve_dp(bytes, 0, iset, &mut memo) {
            candidates.push(Candidate {
                start_val,
                cost: r.cost,
                enc: r.enc,
            });
        }
    }

    if candidates.is_empty() {
        return Err(BarcodeError::EncodingError("no valid start code".into()));
    }

    // Pick lowest cost
    let mut best_idx = 0;
    for (i, c) in candidates.iter().enumerate().skip(1) {
        if c.cost < candidates[best_idx].cost {
            best_idx = i;
        }
    }
    let best = &candidates[best_idx];

    let mut enc_vals = vec![best.start_val];
    enc_vals.extend_from_slice(&best.enc);

    // Checksum
    let mut check_sum = enc_vals[0];
    for (j, &v) in enc_vals.iter().enumerate().skip(1) {
        check_sum += v * j as i32;
    }
    let ck = check_sum % 103;
    enc_vals.push(ck);
    enc_vals.push(106); // Stop

    expand_code128_pattern(&enc_vals)
}

fn expand_code128_pattern(vals: &[i32]) -> Result<Vec<i32>> {
    let mut result = Vec::new();
    for &v in vals {
        if !(0..=106).contains(&v) {
            return Err(BarcodeError::EncodingError(format!(
                "invalid code value {}",
                v
            )));
        }
        let ptn = CODE128_PATTERNS[v as usize];
        for ch in ptn.bytes() {
            result.push((ch - b'0') as i32);
        }
    }
    Ok(result)
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_encode_auto() {
        let encoder = Code128Encoder {
            code_mode: CODE128_AUTO,
        };
        let pattern = encoder.encode("Hello123").unwrap();
        assert!(!pattern.is_empty());
    }

    #[test]
    fn test_encode_code_b() {
        let encoder = Code128Encoder {
            code_mode: CODE128_CODE_B,
        };
        let pattern = encoder.encode("Hello").unwrap();
        assert!(!pattern.is_empty());
    }

    #[test]
    fn test_encode_code_c() {
        let encoder = Code128Encoder {
            code_mode: CODE128_CODE_C,
        };
        let pattern = encoder.encode("1234").unwrap();
        assert!(!pattern.is_empty());
    }

    #[test]
    fn test_encode_empty() {
        let encoder = Code128Encoder {
            code_mode: CODE128_AUTO,
        };
        assert!(encoder.encode("").is_err());
    }

    #[test]
    fn test_encode_numeric_uses_code_c() {
        let encoder = Code128Encoder {
            code_mode: CODE128_AUTO,
        };
        let pattern = encoder.encode("123456").unwrap();
        assert!(!pattern.is_empty());
    }
}
