use crate::base::BarcodeBase;
use crate::error::{BarcodeError, Result};

/// Trait implemented by each 1D barcode encoder.
pub trait Encoder1D {
    fn encode(&self, code: &str) -> Result<Vec<i32>>;
}

/// Base for 1D barcode symbologies.
pub struct BarcodeBase1D {
    pub base: BarcodeBase,
    pub show_text: bool,
    pub text_even_spacing: bool,
    pub(crate) text_font_scale: f64,
    pub(crate) text_horizontal_spacing_scale: f64,
    pub(crate) text_vertical_offset_scale: f64,
    pub(crate) min_line_width: i32,
}

impl BarcodeBase1D {
    pub fn new(output_format: &str) -> Self {
        Self {
            base: BarcodeBase::new(output_format),
            show_text: true,
            text_even_spacing: true,
            text_font_scale: 1.0,
            text_horizontal_spacing_scale: 1.0,
            text_vertical_offset_scale: 1.0,
            min_line_width: 1,
        }
    }

    pub fn set_text_font_scale(&mut self, s: f64) {
        self.text_font_scale = s;
    }

    pub fn text_font_scale(&self) -> f64 {
        self.text_font_scale
    }

    pub fn set_text_vertical_offset_scale(&mut self, s: f64) {
        self.text_vertical_offset_scale = s;
    }

    pub fn text_vertical_offset_scale(&self) -> f64 {
        self.text_vertical_offset_scale
    }

    pub fn set_text_horizontal_spacing_scale(&mut self, s: f64) {
        self.text_horizontal_spacing_scale = s;
    }

    pub fn text_horizontal_spacing_scale(&self) -> f64 {
        self.text_horizontal_spacing_scale
    }

    pub fn set_min_line_width(&mut self, w: i32) {
        self.min_line_width = w;
    }

    pub fn min_line_width(&self) -> i32 {
        self.min_line_width
    }

    /// Renders the 1D barcode using the given encoder.
    pub fn draw(&mut self, code: &str, width: i32, height: i32, encoder: &dyn Encoder1D) -> Result<()> {
        let pattern = encoder.encode(code)?;
        if pattern.is_empty() {
            return Err(BarcodeError::EncodingError("empty pattern".into()));
        }
        if self.base.is_svg_output() {
            self.draw_svg_1d(&pattern, code, width, height)
        } else {
            self.draw_png_1d(&pattern, code, width, height)
        }
    }

    // --- SVG rendering for generic 1D ---

    fn draw_svg_1d(&mut self, pattern: &[i32], code: &str, width: i32, height: i32) -> Result<()> {
        let display = if self.show_text {
            code.to_string()
        } else {
            String::new()
        };
        self.draw_svg_bars(pattern, width, height, &display)
    }

    pub(crate) fn draw_svg_bars(
        &mut self,
        pattern: &[i32],
        width: i32,
        height: i32,
        text: &str,
    ) -> Result<()> {
        let total_units: i32 = pattern.iter().sum();
        if total_units <= 0 {
            return Err(BarcodeError::EncodingError(
                "pattern has zero total units".into(),
            ));
        }

        let font_size = (8.0f64).max(height as f64 * 0.15 * self.text_font_scale) as i32;

        let text_h = if !text.is_empty() {
            (font_size as f64 * 1.4) as i32
        } else {
            0
        };
        let mut bar_h = height - text_h;
        let display_text;
        if bar_h <= 0 {
            bar_h = height;
            display_text = "";
        } else {
            display_text = text;
        }

        self.base.svg_begin(width, height);
        self.base
            .svg_rect(0.0, 0.0, width as f64, height as f64, self.base.back_color);

        let unit_w = width as f64 / total_units as f64;
        let mut accum = 0.0f64;
        let mut is_bar = true;
        for &units in pattern {
            let x1 = (accum * unit_w).round();
            accum += units as f64;
            let x2 = (accum * unit_w).round();
            if is_bar && x2 > x1 {
                self.base
                    .svg_rect(x1, 0.0, x2 - x1, bar_h as f64, self.base.fore_color);
            }
            is_bar = !is_bar;
        }

        if !display_text.is_empty() {
            let text_y = bar_h as f64 + 2.0;
            if self.text_even_spacing {
                self.svg_text_even_distribution(
                    display_text,
                    0.0,
                    text_y,
                    width as f64,
                    font_size,
                    self.text_horizontal_spacing_scale,
                );
            } else {
                let fore = self.base.fore_color;
                self.base
                    .svg_text(width as f64 / 2.0, text_y, display_text, font_size, fore, "middle");
            }
        }

        self.base.svg_end();
        Ok(())
    }

    /// Renders text with even character distribution (SVG).
    pub(crate) fn svg_text_even_distribution(
        &mut self,
        text: &str,
        x: f64,
        y: f64,
        width: f64,
        font_size: i32,
        h_scale: f64,
    ) {
        if text.is_empty() {
            return;
        }
        let num_chars = text.len();

        let margin_ratio = if width < 100.0 {
            0.03
        } else if width > 400.0 {
            0.07
        } else {
            0.05
        };

        let horizontal_margin = width * margin_ratio * h_scale;
        let text_width = width - 2.0 * horizontal_margin;

        let mut char_spacing = text_width;
        if num_chars > 1 {
            char_spacing = text_width / (num_chars - 1) as f64;
        }

        let min_spacing = font_size as f64 / 2.0;
        if char_spacing < min_spacing {
            char_spacing = min_spacing;
        }

        let total_text_width = if num_chars > 1 {
            char_spacing * (num_chars - 1) as f64
        } else {
            0.0
        };
        let start_x = x + (width - total_text_width) / 2.0;

        let fore = self.base.fore_color;
        for (i, ch) in text.chars().enumerate() {
            let char_x = start_x + i as f64 * char_spacing;
            self.base.svg_text(
                char_x,
                y,
                &ch.to_string(),
                font_size,
                fore,
                "middle",
            );
        }
    }

    // --- PNG rendering for generic 1D ---

    #[cfg(any(feature = "png", feature = "jpeg"))]
    fn draw_png_1d(&mut self, pattern: &[i32], code: &str, width: i32, height: i32) -> Result<()> {
        let display = if self.show_text {
            code.to_string()
        } else {
            String::new()
        };
        self.render_bars_to_png(pattern, width, height, &display)
    }

    #[cfg(not(any(feature = "png", feature = "jpeg")))]
    fn draw_png_1d(&mut self, _pattern: &[i32], _code: &str, _width: i32, _height: i32) -> Result<()> {
        Err(BarcodeError::FormatError(
            "PNG/JPEG support requires 'png' or 'jpeg' feature".into(),
        ))
    }

    #[cfg(any(feature = "png", feature = "jpeg"))]
    pub(crate) fn render_bars_to_png(
        &mut self,
        pattern: &[i32],
        width: i32,
        height: i32,
        text: &str,
    ) -> Result<()> {
        let total_units: i32 = pattern.iter().sum();
        if total_units <= 0 {
            return Err(BarcodeError::EncodingError(
                "pattern has zero total units".into(),
            ));
        }

        let font_size = (8.0f64).max(height as f64 * 0.15 * self.text_font_scale) as i32;

        let text_h = if !text.is_empty() {
            (font_size as f64 * 1.4) as i32
        } else {
            0
        };
        let mut bar_h = height - text_h;
        let display_text;
        if bar_h <= 0 {
            bar_h = height;
            display_text = "";
        } else {
            display_text = text;
        }

        let mut img = image::RgbaImage::from_pixel(
            width as u32,
            height as u32,
            image::Rgba([
                self.base.back_color.r,
                self.base.back_color.g,
                self.base.back_color.b,
                self.base.back_color.a,
            ]),
        );

        let unit_w = width as f64 / total_units as f64;
        let mut accum = 0.0f64;
        let mut is_bar = true;
        let fore = image::Rgba([
            self.base.fore_color.r,
            self.base.fore_color.g,
            self.base.fore_color.b,
            self.base.fore_color.a,
        ]);

        for &units in pattern {
            let x1 = (accum * unit_w).round() as i32;
            accum += units as f64;
            let x2 = (accum * unit_w).round() as i32;
            if is_bar && x2 > x1 {
                fill_rect(&mut img, x1, 0, x2, bar_h, fore);
            }
            is_bar = !is_bar;
        }

        // Draw text
        if !display_text.is_empty() {
            #[cfg(feature = "font")]
            {
                let text_y = bar_h + 2;
                self.draw_text_on_image(&mut img, display_text, 0, text_y, width, font_size);
            }
        }

        // Trial mode watermark
        if crate::product_info::is_trial_mode() {
            #[cfg(feature = "font")]
            draw_sample_overlay_png(&mut img, 0, 0, width, height);
        }

        self.base.encode_image(&img)
    }

    #[cfg(all(any(feature = "png", feature = "jpeg"), feature = "font"))]
    fn draw_text_on_image(
        &self,
        img: &mut image::RgbaImage,
        text: &str,
        x: i32,
        y: i32,
        width: i32,
        font_size: i32,
    ) {
        let font = crate::font::get_default_font();
        let font = match font {
            Some(f) => f,
            None => return,
        };

        if self.text_even_spacing {
            crate::font::draw_text_even_png(
                img,
                &font,
                text,
                x as f64,
                y as f64,
                width as f64,
                font_size,
                self.base.fore_color,
                self.text_horizontal_spacing_scale,
            );
        } else {
            crate::font::draw_text_centered_png(
                img,
                &font,
                text,
                x + width / 2,
                y,
                font_size,
                self.base.fore_color,
            );
        }
    }
}

/// Fill a rectangle on an RGBA image.
#[cfg(any(feature = "png", feature = "jpeg"))]
pub(crate) fn fill_rect(img: &mut image::RgbaImage, x1: i32, y1: i32, x2: i32, y2: i32, c: image::Rgba<u8>) {
    let (iw, ih) = (img.width() as i32, img.height() as i32);
    let x1 = x1.max(0).min(iw);
    let x2 = x2.max(0).min(iw);
    let y1 = y1.max(0).min(ih);
    let y2 = y2.max(0).min(ih);
    for py in y1..y2 {
        for px in x1..x2 {
            img.put_pixel(px as u32, py as u32, c);
        }
    }
}

/// Draw SAMPLE watermark overlay on PNG image.
#[cfg(all(any(feature = "png", feature = "jpeg"), feature = "font"))]
pub(crate) fn draw_sample_overlay_png(
    img: &mut image::RgbaImage,
    x: i32,
    y: i32,
    _width: i32,
    height: i32,
) {
    let mut font_size = (height as f64 * 0.12) as i32;
    font_size = font_size.clamp(8, 40);
    let mut margin = (height as f64 * 0.01) as i32;
    if margin < 2 {
        margin = 2;
    }
    let text = crate::product_info::get_trial_text();
    let red = crate::color::Rgba::RED;

    let font = match crate::font::get_default_font() {
        Some(f) => f,
        None => return,
    };

    crate::font::draw_text_at_png(
        img,
        &font,
        text,
        x + margin,
        y + margin,
        font_size,
        red,
    );
}
