use crate::color::Rgba;
use crate::error::{BarcodeError, Result};
use crate::product_info::{get_trial_text, is_trial_mode};

/// Output format constants.
pub const FORMAT_PNG: &str = "png";
pub const FORMAT_JPEG: &str = "jpeg";
pub const FORMAT_SVG: &str = "svg";

/// Common state for all barcode types.
pub struct BarcodeBase {
    pub(crate) fore_color: Rgba,
    pub(crate) back_color: Rgba,
    pub(crate) format: String,
    pub(crate) fit_width: bool,
    pub(crate) px_adj_black: i32,
    pub(crate) px_adj_white: i32,
    pub(crate) svg_stream: String,
    pub(crate) svg_width: i32,
    pub(crate) svg_height: i32,
    pub(crate) image_buffer: Vec<u8>,
}

impl BarcodeBase {
    /// Initialises BarcodeBase with defaults.
    pub fn new(output_format: &str) -> Self {
        Self {
            fore_color: Rgba::BLACK,
            back_color: Rgba::WHITE,
            format: output_format.to_string(),
            fit_width: false,
            px_adj_black: 0,
            px_adj_white: 0,
            svg_stream: String::new(),
            svg_width: 0,
            svg_height: 0,
            image_buffer: Vec::new(),
        }
    }

    // --- Color accessors ---

    pub fn set_foreground_color(&mut self, r: u8, g: u8, b: u8, a: u8) {
        self.fore_color = Rgba::new(r, g, b, a);
    }

    pub fn set_background_color(&mut self, r: u8, g: u8, b: u8, a: u8) {
        self.back_color = Rgba::new(r, g, b, a);
    }

    pub fn fore_color(&self) -> Rgba {
        self.fore_color
    }

    pub fn back_color(&self) -> Rgba {
        self.back_color
    }

    // --- Pixel adjustment ---

    pub fn set_px_adjust_black(&mut self, adj: i32) {
        self.px_adj_black = adj;
    }

    pub fn set_px_adjust_white(&mut self, adj: i32) {
        self.px_adj_white = adj;
    }

    // --- Fit width ---

    pub fn set_fit_width(&mut self, fit: bool) {
        self.fit_width = fit;
    }

    pub fn fit_width(&self) -> bool {
        self.fit_width
    }

    // --- Format ---

    pub fn set_output_format(&mut self, fmt: &str) {
        self.format = fmt.to_string();
    }

    pub fn output_format(&self) -> &str {
        &self.format
    }

    pub fn is_svg_output(&self) -> bool {
        self.format == FORMAT_SVG
    }

    // --- Output ---

    /// Returns the rendered image bytes (PNG/JPEG).
    pub fn get_image_memory(&self) -> &[u8] {
        &self.image_buffer
    }

    /// Returns a data-URI string for PNG/JPEG output.
    pub fn get_image_base64(&self) -> Result<String> {
        if self.is_svg_output() {
            return Err(BarcodeError::FormatError(
                "GetImageBase64() is not available in SVG mode; use get_svg()".into(),
            ));
        }
        if self.image_buffer.is_empty() {
            return Ok(String::new());
        }
        use base64::Engine;
        let encoded = base64::engine::general_purpose::STANDARD.encode(&self.image_buffer);
        let mime = if self.format == FORMAT_JPEG {
            "image/jpeg"
        } else {
            "image/png"
        };
        Ok(format!("data:{};base64,{}", mime, encoded))
    }

    /// Returns the SVG string.
    pub fn get_svg(&self) -> Result<String> {
        if !self.is_svg_output() {
            return Err(BarcodeError::FormatError(
                "get_svg() requires SVG output mode".into(),
            ));
        }
        Ok(self.svg_stream.clone())
    }

    // --- SVG helpers ---

    pub(crate) fn svg_begin(&mut self, width: i32, height: i32) {
        self.svg_width = width;
        self.svg_height = height;
        self.svg_stream.clear();
        self.svg_stream
            .push_str("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n");
        self.svg_stream.push_str(&format!(
            "<svg xmlns=\"http://www.w3.org/2000/svg\" version=\"1.1\" \
             width=\"{}\" height=\"{}\" viewBox=\"0 0 {} {}\">\n",
            width, height, width, height
        ));
    }

    pub(crate) fn svg_end(&mut self) {
        if is_trial_mode() {
            self.svg_sample_overlay(0, 0, self.svg_width, self.svg_height);
        }
        self.svg_stream.push_str("</svg>\n");
    }

    pub(crate) fn svg_sample_overlay(&mut self, x: i32, y: i32, _width: i32, height: i32) {
        let mut font_size = (height as f64 * 0.12) as i32;
        font_size = font_size.clamp(8, 40);
        let mut margin = (height as f64 * 0.01) as i32;
        if margin < 2 {
            margin = 2;
        }
        let red = Rgba::RED;
        let text = get_trial_text();
        self.svg_stream.push_str(&format!(
            "  <text x=\"{}\" y=\"{}\" font-family=\"Arial, sans-serif\" font-size=\"{}\" \
             font-weight=\"bold\" font-style=\"italic\" fill=\"{}\" \
             dominant-baseline=\"hanging\">{}</text>\n",
            x + margin,
            y + margin,
            font_size,
            red.to_rgb_string(),
            text
        ));
    }

    pub(crate) fn svg_rect(&mut self, x: f64, y: f64, w: f64, h: f64, c: Rgba) {
        let opacity = if c.a < 255 {
            format!("\" fill-opacity=\"{:.2}", c.a as f64 / 255.0)
        } else {
            String::new()
        };
        self.svg_stream.push_str(&format!(
            "  <rect x=\"{:.2}\" y=\"{:.2}\" width=\"{:.2}\" height=\"{:.2}\" \
             fill=\"{}{}\"/>\n",
            x,
            y,
            w,
            h,
            c.to_rgb_string(),
            opacity
        ));
    }

    pub(crate) fn svg_text(
        &mut self,
        x: f64,
        y: f64,
        text: &str,
        font_size: i32,
        c: Rgba,
        anchor: &str,
    ) {
        let anchor = if anchor.is_empty() { "middle" } else { anchor };
        self.svg_stream.push_str(&format!(
            "  <text x=\"{:.2}\" y=\"{:.2}\" font-family=\"Arial, sans-serif\" \
             font-size=\"{}\" fill=\"{}\" text-anchor=\"{}\" \
             dominant-baseline=\"hanging\">{}</text>\n",
            x,
            y,
            font_size,
            c.to_rgb_string(),
            anchor,
            text
        ));
    }

    /// Encode image to PNG or JPEG and store in image_buffer.
    #[cfg(any(feature = "png", feature = "jpeg"))]
    pub(crate) fn encode_image(&mut self, img: &image::RgbaImage) -> Result<()> {
        use image::ImageEncoder;
        use std::io::Cursor;

        let mut buf = Vec::new();
        let mut cursor = Cursor::new(&mut buf);

        match self.format.as_str() {
            FORMAT_JPEG => {
                // JPEG doesn't support RGBA, convert to RGB
                let rgb_img: image::RgbImage = image::DynamicImage::ImageRgba8(img.clone()).to_rgb8();
                let encoder = image::codecs::jpeg::JpegEncoder::new_with_quality(&mut cursor, 95);
                encoder
                    .write_image(&rgb_img, rgb_img.width(), rgb_img.height(), image::ExtendedColorType::Rgb8)
                    .map_err(|e| BarcodeError::ImageError(e.to_string()))?;
            }
            _ => {
                let encoder = image::codecs::png::PngEncoder::new(&mut cursor);
                encoder
                    .write_image(img, img.width(), img.height(), image::ExtendedColorType::Rgba8)
                    .map_err(|e| BarcodeError::ImageError(e.to_string()))?;
            }
        }

        self.image_buffer = buf;
        Ok(())
    }
}
