//! BarcodeData — intermediate representation for external renderers.
//!
//! Provides a renderer-agnostic data structure that holds all information
//! needed to draw a barcode, decoupling encoding from rendering.

use crate::color::Rgba;
use crate::error::Result;

/// Identifies the rendering category for a barcode.
#[derive(Debug, Clone, Copy, PartialEq, Eq)]
pub enum BarcodeDataType {
    /// Generic 1D bar/space pattern.
    Type1D,
    /// JAN/UPC with extended guard bars.
    TypeJANUPC,
    /// Postal 4-state bars (YubinCustomer).
    Type4State,
    /// 2D matrix (QR, DataMatrix, PDF417).
    Type2DMatrix,
    /// Multi-row pattern (DataBar stacked variants).
    TypeMultiRow,
}

/// Holds all data needed by external renderers.
///
/// Depending on `data_type`, different fields are populated:
/// - `Type1D`: pattern, display_code, show_text, text_*
/// - `TypeJANUPC`: pattern, display_code, jan_upc_type, extended_guard, text_*
/// - `Type4State`: bars_4state
/// - `Type2DMatrix`: matrix, col_major
/// - `TypeMultiRow`: multi_row_patterns, multi_row_heights
#[derive(Debug, Clone)]
pub struct BarcodeData {
    pub data_type: BarcodeDataType,

    // Common
    pub fore_color: Rgba,
    pub back_color: Rgba,

    // 1D properties
    pub pattern: Vec<i32>,
    pub display_code: String,
    pub show_text: bool,
    pub text_even_spacing: bool,
    pub text_font_scale: f64,
    pub text_h_spacing_scale: f64,
    pub text_v_offset_scale: f64,

    // JAN/UPC specific
    pub jan_upc_type: String,
    pub extended_guard: bool,

    // 4-state
    pub bars_4state: Vec<i32>,

    // 2D matrix
    pub matrix: Vec<Vec<bool>>,
    /// true = QR/DataMatrix (matrix[col][row]), false = PDF417 (matrix[row][col])
    pub col_major: bool,

    // Multi-row DataBar
    pub multi_row_patterns: Vec<String>,
    pub multi_row_heights: Vec<i32>,
}

impl BarcodeData {
    fn new(data_type: BarcodeDataType, fore_color: Rgba, back_color: Rgba) -> Self {
        Self {
            data_type,
            fore_color,
            back_color,
            pattern: Vec::new(),
            display_code: String::new(),
            show_text: false,
            text_even_spacing: false,
            text_font_scale: 1.0,
            text_h_spacing_scale: 1.0,
            text_v_offset_scale: 1.0,
            jan_upc_type: String::new(),
            extended_guard: false,
            bars_4state: Vec::new(),
            matrix: Vec::new(),
            col_major: false,
            multi_row_patterns: Vec::new(),
            multi_row_heights: Vec::new(),
        }
    }
}

/// Trait for encoding a barcode into renderer-agnostic BarcodeData.
pub trait EncodeBarcodeData {
    fn encode_data(&mut self, code: &str) -> Result<BarcodeData>;
}

// --- Helper for generic 1D types ---

fn encode_1d_data(
    b: &crate::base_1d::BarcodeBase1D,
    pattern: Vec<i32>,
    code: &str,
) -> BarcodeData {
    let display = if b.show_text {
        code.to_string()
    } else {
        String::new()
    };
    let mut data = BarcodeData::new(BarcodeDataType::Type1D, b.base.fore_color, b.base.back_color);
    data.pattern = pattern;
    data.display_code = display;
    data.show_text = b.show_text;
    data.text_even_spacing = b.text_even_spacing;
    data.text_font_scale = b.text_font_scale();
    data.text_h_spacing_scale = b.text_horizontal_spacing_scale();
    data.text_v_offset_scale = b.text_vertical_offset_scale();
    data
}

fn jan_upc_data(
    b: &crate::base_1d::BarcodeBase1D,
    pattern: Vec<i32>,
    display_code: String,
    jan_upc_type: &str,
    extended_guard: bool,
) -> BarcodeData {
    let mut data = BarcodeData::new(BarcodeDataType::TypeJANUPC, b.base.fore_color, b.base.back_color);
    data.pattern = pattern;
    data.display_code = display_code;
    data.show_text = b.show_text;
    data.text_even_spacing = b.text_even_spacing;
    data.text_font_scale = b.text_font_scale();
    data.text_h_spacing_scale = b.text_horizontal_spacing_scale();
    data.text_v_offset_scale = b.text_vertical_offset_scale();
    data.jan_upc_type = jan_upc_type.to_string();
    data.extended_guard = extended_guard;
    data
}

// --- Generic 1D implementations ---

impl EncodeBarcodeData for crate::encoders::code39::Code39 {
    fn encode_data(&mut self, code: &str) -> Result<BarcodeData> {
        let pattern = self.encode(code)?;
        Ok(encode_1d_data(&self.base_1d, pattern, code))
    }
}

impl EncodeBarcodeData for crate::encoders::code93::Code93 {
    fn encode_data(&mut self, code: &str) -> Result<BarcodeData> {
        let pattern = self.encode(code)?;
        Ok(encode_1d_data(&self.base_1d, pattern, code))
    }
}

impl EncodeBarcodeData for crate::encoders::code128::Code128 {
    fn encode_data(&mut self, code: &str) -> Result<BarcodeData> {
        let pattern = self.encode(code)?;
        Ok(encode_1d_data(&self.base_1d, pattern, code))
    }
}

impl EncodeBarcodeData for crate::encoders::nw7::NW7 {
    fn encode_data(&mut self, code: &str) -> Result<BarcodeData> {
        let pattern = self.encode(code)?;
        Ok(encode_1d_data(&self.base_1d, pattern, code))
    }
}

impl EncodeBarcodeData for crate::encoders::itf::ITF {
    fn encode_data(&mut self, code: &str) -> Result<BarcodeData> {
        let pattern = self.encode(code)?;
        Ok(encode_1d_data(&self.base_1d, pattern, code))
    }
}

impl EncodeBarcodeData for crate::encoders::matrix2of5::Matrix2of5 {
    fn encode_data(&mut self, code: &str) -> Result<BarcodeData> {
        let pattern = self.encode(code)?;
        Ok(encode_1d_data(&self.base_1d, pattern, code))
    }
}

impl EncodeBarcodeData for crate::encoders::nec2of5::NEC2of5 {
    fn encode_data(&mut self, code: &str) -> Result<BarcodeData> {
        let pattern = self.encode(code)?;
        Ok(encode_1d_data(&self.base_1d, pattern, code))
    }
}

// --- GS1-128 (generic 1D with FNC1/AI text cleanup) ---

impl EncodeBarcodeData for crate::encoders::gs1_128::GS1128 {
    fn encode_data(&mut self, code: &str) -> Result<BarcodeData> {
        let pattern = crate::encoders::gs1_128::encode_gs1128(code)?;
        let display = if self.base_1d.show_text {
            code.replace("{FNC1}", "").replace("{AI}", "")
        } else {
            String::new()
        };
        let mut data = BarcodeData::new(
            BarcodeDataType::Type1D,
            self.base_1d.base.fore_color,
            self.base_1d.base.back_color,
        );
        data.pattern = pattern;
        data.display_code = display;
        data.show_text = self.base_1d.show_text;
        data.text_even_spacing = self.base_1d.text_even_spacing;
        data.text_font_scale = self.base_1d.text_font_scale();
        data.text_h_spacing_scale = self.base_1d.text_horizontal_spacing_scale();
        data.text_v_offset_scale = self.base_1d.text_vertical_offset_scale();
        Ok(data)
    }
}

// --- GS1 DataBar Limited (single-row 1D, no text) ---

impl EncodeBarcodeData for crate::encoders::gs1_databar_limited::GS1DataBarLimited {
    fn encode_data(&mut self, code: &str) -> Result<BarcodeData> {
        let pattern = self.encode(code)?;
        let mut data = BarcodeData::new(
            BarcodeDataType::Type1D,
            self.base_1d.base.fore_color,
            self.base_1d.base.back_color,
        );
        data.pattern = pattern;
        data.show_text = false;
        data.text_even_spacing = self.base_1d.text_even_spacing;
        data.text_font_scale = self.base_1d.text_font_scale();
        data.text_h_spacing_scale = self.base_1d.text_horizontal_spacing_scale();
        data.text_v_offset_scale = self.base_1d.text_vertical_offset_scale();
        Ok(data)
    }
}

// --- JAN/UPC types with extended guard bars ---

impl EncodeBarcodeData for crate::encoders::jan8::JAN8 {
    fn encode_data(&mut self, code: &str) -> Result<BarcodeData> {
        let pattern = crate::encoders::jan8::encode_jan8(code)?;
        let mut s = code.to_string();
        if s.len() == 7 {
            s.push_str(&crate::encoders::jan8::calculate_check_digit_jan8(&s));
        }
        Ok(jan_upc_data(&self.base_1d, pattern, s, "JAN8", self.extended_guard))
    }
}

impl EncodeBarcodeData for crate::encoders::jan13::JAN13 {
    fn encode_data(&mut self, code: &str) -> Result<BarcodeData> {
        let pattern = crate::encoders::jan13::encode_jan13(code)?;
        let mut s = code.to_string();
        if s.len() == 12 {
            s.push_str(&crate::encoders::jan13::calculate_check_digit_jan13(&s));
        }
        Ok(jan_upc_data(&self.base_1d, pattern, s, "JAN13", self.extended_guard))
    }
}

impl EncodeBarcodeData for crate::encoders::upc_a::UPCA {
    fn encode_data(&mut self, code: &str) -> Result<BarcodeData> {
        let pattern = crate::encoders::upc_a::encode_upca(code)?;
        let mut s = code.to_string();
        if s.len() == 11 {
            s.push_str(&crate::encoders::upc_a::calculate_check_digit_upca(&s));
        }
        Ok(jan_upc_data(&self.base_1d, pattern, s, "UPCA", self.extended_guard))
    }
}

impl EncodeBarcodeData for crate::encoders::upc_e::UPCE {
    fn encode_data(&mut self, code: &str) -> Result<BarcodeData> {
        let pattern = crate::encoders::upc_e::encode_upce(code)?;
        let mut display_code = code.to_string();
        if display_code.len() == 6 {
            display_code = format!("0{}", display_code);
            display_code.push_str(&crate::encoders::upc_e::calculate_check_digit_upce(&display_code));
        } else if display_code.len() == 7 {
            display_code.push_str(&crate::encoders::upc_e::calculate_check_digit_upce(&display_code));
        }
        Ok(jan_upc_data(&self.base_1d, pattern, display_code, "UPCE", self.extended_guard))
    }
}

// --- Multi-row DataBar variants ---

impl EncodeBarcodeData for crate::encoders::gs1_databar_14::GS1DataBar14 {
    fn encode_data(&mut self, code: &str) -> Result<BarcodeData> {
        self.encode(code)?;
        let mut data = BarcodeData::new(
            BarcodeDataType::TypeMultiRow,
            self.base_1d.base.fore_color,
            self.base_1d.base.back_color,
        );
        data.multi_row_patterns = self.patterns().to_vec();
        data.multi_row_heights = self.row_heights().to_vec();
        Ok(data)
    }
}

impl EncodeBarcodeData for crate::encoders::gs1_databar_expanded::GS1DataBarExpanded {
    fn encode_data(&mut self, code: &str) -> Result<BarcodeData> {
        self.encode(code)?;
        let mut data = BarcodeData::new(
            BarcodeDataType::TypeMultiRow,
            self.base_1d.base.fore_color,
            self.base_1d.base.back_color,
        );
        data.multi_row_patterns = self.patterns().to_vec();
        data.multi_row_heights = self.row_heights().to_vec();
        Ok(data)
    }
}

// --- 4-State postal ---

impl EncodeBarcodeData for crate::encoders::yubin_customer::YubinCustomer {
    fn encode_data(&mut self, code: &str) -> Result<BarcodeData> {
        self.encode(code)?;
        let mut data = BarcodeData::new(
            BarcodeDataType::Type4State,
            self.base_1d.base.fore_color,
            self.base_1d.base.back_color,
        );
        data.bars_4state = self.bars().to_vec();
        Ok(data)
    }
}

// --- 2D Matrix types ---

impl EncodeBarcodeData for crate::encoders::qr::QR {
    fn encode_data(&mut self, code: &str) -> Result<BarcodeData> {
        let patt = self.get_pattern(code)?;
        let mut data = BarcodeData::new(
            BarcodeDataType::Type2DMatrix,
            self.base_2d.base.fore_color,
            self.base_2d.base.back_color,
        );
        data.matrix = patt;
        data.col_major = true;
        Ok(data)
    }
}

impl EncodeBarcodeData for crate::encoders::datamatrix::DataMatrix {
    fn encode_data(&mut self, code: &str) -> Result<BarcodeData> {
        let patt = self.get_pattern(code)?;
        let mut data = BarcodeData::new(
            BarcodeDataType::Type2DMatrix,
            self.base_2d.base.fore_color,
            self.base_2d.base.back_color,
        );
        data.matrix = patt;
        data.col_major = true;
        Ok(data)
    }
}

impl EncodeBarcodeData for crate::encoders::pdf417::PDF417 {
    fn encode_data(&mut self, code: &str) -> Result<BarcodeData> {
        let patt = self.get_pattern(code)?;
        let mut data = BarcodeData::new(
            BarcodeDataType::Type2DMatrix,
            self.base_2d.base.fore_color,
            self.base_2d.base.back_color,
        );
        data.matrix = patt;
        data.col_major = false; // PDF417: matrix[row][col]
        Ok(data)
    }
}

#[cfg(test)]
mod tests {
    use super::*;
    use crate::base::FORMAT_SVG;

    #[test]
    fn test_encode_data_code39() {
        let mut bc = crate::Code39::new(FORMAT_SVG);
        let data = bc.encode_data("ABC123").unwrap();
        assert_eq!(data.data_type, BarcodeDataType::Type1D);
        assert!(!data.pattern.is_empty());
        assert_eq!(data.display_code, "ABC123");
        assert!(data.show_text);
    }

    #[test]
    fn test_encode_data_jan13() {
        let mut bc = crate::JAN13::new(FORMAT_SVG);
        let data = bc.encode_data("490123456789").unwrap();
        assert_eq!(data.data_type, BarcodeDataType::TypeJANUPC);
        assert_eq!(data.jan_upc_type, "JAN13");
        assert_eq!(data.display_code.len(), 13);
        assert!(data.extended_guard);
    }

    #[test]
    fn test_encode_data_qr() {
        let mut bc = crate::QR::new(FORMAT_SVG);
        let data = bc.encode_data("Hello World").unwrap();
        assert_eq!(data.data_type, BarcodeDataType::Type2DMatrix);
        assert!(data.col_major);
        assert!(!data.matrix.is_empty());
    }

    #[test]
    fn test_encode_data_pdf417() {
        let mut bc = crate::PDF417::new(FORMAT_SVG);
        let data = bc.encode_data("Hello World").unwrap();
        assert_eq!(data.data_type, BarcodeDataType::Type2DMatrix);
        assert!(!data.col_major); // row-major
        assert!(!data.matrix.is_empty());
    }

    #[test]
    fn test_encode_data_yubin() {
        let mut bc = crate::YubinCustomer::new(FORMAT_SVG);
        let data = bc.encode_data("1000001").unwrap();
        assert_eq!(data.data_type, BarcodeDataType::Type4State);
        assert!(!data.bars_4state.is_empty());
    }

    #[test]
    fn test_encode_data_databar14() {
        use crate::encoders::gs1_databar_14::SymbolType14;
        let mut bc = crate::GS1DataBar14::new(FORMAT_SVG, SymbolType14::Stacked);
        let data = bc.encode_data("0000000012345").unwrap();
        assert_eq!(data.data_type, BarcodeDataType::TypeMultiRow);
        assert!(!data.multi_row_patterns.is_empty());
        assert!(!data.multi_row_heights.is_empty());
    }

    #[test]
    fn test_encode_data_all_18_types() {
        use crate::encoders::gs1_databar_14::SymbolType14;
        use crate::encoders::gs1_databar_expanded::ExpandedSymbolType;

        // Test all 18 types produce valid BarcodeData
        let mut count = 0;

        let mut c39 = crate::Code39::new(FORMAT_SVG);
        assert!(c39.encode_data("ABC").is_ok()); count += 1;

        let mut c93 = crate::Code93::new(FORMAT_SVG);
        assert!(c93.encode_data("ABC").is_ok()); count += 1;

        let mut c128 = crate::Code128::new(FORMAT_SVG);
        assert!(c128.encode_data("Hello").is_ok()); count += 1;

        let mut gs1 = crate::GS1128::new(FORMAT_SVG);
        assert!(gs1.encode_data("{FNC1}0104912345678904").is_ok()); count += 1;

        let mut nw7 = crate::NW7::new(FORMAT_SVG);
        assert!(nw7.encode_data("A123A").is_ok()); count += 1;

        let mut itf = crate::ITF::new(FORMAT_SVG);
        assert!(itf.encode_data("1234567890").is_ok()); count += 1;

        let mut m2 = crate::Matrix2of5::new(FORMAT_SVG);
        assert!(m2.encode_data("1234567890").is_ok()); count += 1;

        let mut n2 = crate::NEC2of5::new(FORMAT_SVG);
        assert!(n2.encode_data("1234567890").is_ok()); count += 1;

        let mut j8 = crate::JAN8::new(FORMAT_SVG);
        assert!(j8.encode_data("1234567").is_ok()); count += 1;

        let mut j13 = crate::JAN13::new(FORMAT_SVG);
        assert!(j13.encode_data("490123456789").is_ok()); count += 1;

        let mut ua = crate::UPCA::new(FORMAT_SVG);
        assert!(ua.encode_data("01234567890").is_ok()); count += 1;

        let mut ue = crate::UPCE::new(FORMAT_SVG);
        assert!(ue.encode_data("0123456").is_ok()); count += 1;

        let mut db14 = crate::GS1DataBar14::new(FORMAT_SVG, SymbolType14::Omnidirectional);
        assert!(db14.encode_data("0000000012345").is_ok()); count += 1;

        let mut dbl = crate::GS1DataBarLimited::new(FORMAT_SVG);
        assert!(dbl.encode_data("0000000012345").is_ok()); count += 1;

        let mut dbe = crate::GS1DataBarExpanded::new(FORMAT_SVG, ExpandedSymbolType::Unstacked, 2);
        assert!(dbe.encode_data("{FNC1}0104912345678904").is_ok()); count += 1;

        let mut yb = crate::YubinCustomer::new(FORMAT_SVG);
        assert!(yb.encode_data("1000001").is_ok()); count += 1;

        let mut qr = crate::QR::new(FORMAT_SVG);
        assert!(qr.encode_data("Hello").is_ok()); count += 1;

        let mut dm = crate::DataMatrix::new(FORMAT_SVG);
        assert!(dm.encode_data("Hello").is_ok()); count += 1;

        let mut p417 = crate::PDF417::new(FORMAT_SVG);
        assert!(p417.encode_data("Hello").is_ok()); count += 1;

        assert_eq!(count, 19); // 18 types + DataBar14 counted once above = Code39..PDF417 = 19
    }
}
