// Barcode.Rust - All-in-One
//
// Full 18-type barcode REST API with 4 output modes using axum.
//
//   cargo run
//   -> http://localhost:5721

use axum::{
    extract::{Form, Query},
    http::{header, StatusCode},
    response::{Html, IntoResponse, Json},
    routing::{get, post},
    Router,
};
use serde::{Deserialize, Serialize};

use barcode_pao::{
    Code128, Code39, Code93, DataMatrix, ExpandedSymbolType, GS1128, GS1DataBar14,
    GS1DataBarExpanded, GS1DataBarLimited, ITF, JAN13, JAN8, Matrix2of5, NEC2of5, NW7, PDF417,
    QR, SymbolType14, UPCA, UPCE, YubinCustomer, FORMAT_PNG, FORMAT_SVG,
};

// ---------------------------------------------------------------------------
// Barcode type metadata (injected into HTML template)
// ---------------------------------------------------------------------------

const BARCODE_TYPES_JSON: &str = r#"[
{"id":"QR","label":"QR Code","group":"2D Barcode","dim":"2d","default":"https://www.pao.ac/","w":200,"h":200},
{"id":"DataMatrix","label":"DataMatrix","group":"2D Barcode","dim":"2d","default":"Hello Barcode.Rust","w":200,"h":200},
{"id":"PDF417","label":"PDF417","group":"2D Barcode","dim":"2d","default":"Hello Barcode.Rust","w":400,"h":120},
{"id":"Code39","label":"Code 39","group":"1D Barcode","dim":"1d","default":"ABC-1234","w":400,"h":100},
{"id":"Code93","label":"Code 93","group":"1D Barcode","dim":"1d","default":"ABC-1234","w":400,"h":100},
{"id":"Code128","label":"Code 128","group":"1D Barcode","dim":"1d","default":"Hello-2026","w":400,"h":100},
{"id":"NW7","label":"NW-7 (Codabar)","group":"1D Barcode","dim":"1d","default":"A12345B","w":400,"h":100},
{"id":"ITF","label":"ITF","group":"1D Barcode","dim":"1d","default":"1234567890","w":400,"h":100},
{"id":"Matrix2of5","label":"Matrix 2 of 5","group":"1D Barcode","dim":"1d","default":"1234567890","w":400,"h":100},
{"id":"NEC2of5","label":"NEC 2 of 5","group":"1D Barcode","dim":"1d","default":"1234567890","w":400,"h":100},
{"id":"GS1-128","label":"GS1-128","group":"1D Barcode","dim":"1d","default":"{01}04912345123459{10}ABC123","w":500,"h":100},
{"id":"JAN13","label":"JAN-13 (EAN-13)","group":"JAN / UPC","dim":"1d","default":"4901234567894","w":300,"h":100},
{"id":"JAN8","label":"JAN-8 (EAN-8)","group":"JAN / UPC","dim":"1d","default":"49123456","w":250,"h":100},
{"id":"UPC-A","label":"UPC-A","group":"JAN / UPC","dim":"1d","default":"01234567890","w":300,"h":100},
{"id":"UPC-E","label":"UPC-E","group":"JAN / UPC","dim":"1d","default":"0123456","w":250,"h":100},
{"id":"GS1DataBar14","label":"GS1 DataBar","group":"GS1 DataBar","dim":"1d","default":"0001234567890","w":400,"h":100},
{"id":"GS1DataBarLimited","label":"GS1 DataBar Limited","group":"GS1 DataBar","dim":"1d","default":"0001234567890","w":400,"h":100},
{"id":"GS1DataBarExpanded","label":"GS1 DataBar Expanded","group":"GS1 DataBar","dim":"1d","default":"{01}90012345678908{3103}000123","w":500,"h":100},
{"id":"YubinCustomer","label":"Yubin Customer","group":"Special","dim":"postal","default":"10408-0506001","w":400,"h":40}
]"#;

// ---------------------------------------------------------------------------
// Form parameters
// ---------------------------------------------------------------------------

#[derive(Debug, Deserialize)]
struct DrawParams {
    #[serde(rename = "type")]
    type_id: Option<String>,
    #[serde(default)]
    code: String,
    #[serde(default = "default_width")]
    width: i32,
    #[serde(default = "default_height")]
    height: i32,
    #[serde(default)]
    x: i32,
    #[serde(default)]
    y: i32,
    #[serde(default = "default_show_text")]
    show_text: String,
    #[serde(default = "default_even_spacing")]
    even_spacing: String,
    #[serde(default)]
    fore_color: Option<String>,
    #[serde(default)]
    back_color: Option<String>,
    #[serde(default)]
    transparent_bg: Option<String>,
    #[serde(default)]
    #[allow(dead_code)]
    font_size: Option<i32>,
    #[serde(default)]
    extended_guard: Option<String>,
    #[serde(default)]
    disp_start_stop: Option<String>,
    // QR
    #[serde(default)]
    qr_error_level: Option<i32>,
    #[serde(default)]
    qr_version: Option<i32>,
    // DataMatrix
    #[serde(default)]
    datamatrix_size: Option<i32>,
    // PDF417
    #[serde(default)]
    pdf417_error_level: Option<i32>,
    #[serde(default)]
    pdf417_cols: Option<i32>,
    #[serde(default)]
    pdf417_rows: Option<i32>,
    #[serde(default)]
    pdf417_aspect_ratio: Option<f64>,
    #[serde(default)]
    pdf417_size_kind: Option<String>,
    // GS1 DataBar 14
    #[serde(default)]
    databar14_type: Option<String>,
    // GS1 DataBar Expanded
    #[serde(default)]
    databar_expanded_type: Option<String>,
    #[serde(default)]
    databar_expanded_cols: Option<i32>,
}

fn default_width() -> i32 { 300 }
fn default_height() -> i32 { 100 }
fn default_show_text() -> String { "1".to_string() }
fn default_even_spacing() -> String { "1".to_string() }

#[derive(Debug, Deserialize)]
struct PdfQuery {
    #[serde(rename = "type")]
    type_id: Option<String>,
    #[serde(default)]
    code: Option<String>,
    #[serde(default)]
    width: Option<i32>,
    #[serde(default)]
    height: Option<i32>,
}

// ---------------------------------------------------------------------------
// JSON response
// ---------------------------------------------------------------------------

#[derive(Serialize)]
struct DrawResponse {
    ok: bool,
    #[serde(skip_serializing_if = "Option::is_none")]
    base64: Option<String>,
    #[serde(skip_serializing_if = "Option::is_none")]
    svg: Option<String>,
    #[serde(skip_serializing_if = "Option::is_none")]
    error: Option<String>,
}

impl DrawResponse {
    fn ok_base64(data: String) -> Self {
        Self { ok: true, base64: Some(data), svg: None, error: None }
    }
    fn ok_svg(data: String) -> Self {
        Self { ok: true, base64: None, svg: Some(data), error: None }
    }
    fn err(msg: String) -> Self {
        Self { ok: false, base64: None, svg: None, error: Some(msg) }
    }
}

// ---------------------------------------------------------------------------
// Helpers
// ---------------------------------------------------------------------------

fn parse_hex_color(hex: &str) -> (u8, u8, u8) {
    let hex = hex.trim_start_matches('#');
    if hex.len() >= 6 {
        let r = u8::from_str_radix(&hex[0..2], 16).unwrap_or(0);
        let g = u8::from_str_radix(&hex[2..4], 16).unwrap_or(0);
        let b = u8::from_str_radix(&hex[4..6], 16).unwrap_or(0);
        (r, g, b)
    } else {
        (0, 0, 0)
    }
}

fn apply_colors_1d(bc: &mut barcode_pao::BarcodeBase1D, params: &DrawParams) {
    if let Some(ref fc) = params.fore_color {
        if !fc.is_empty() {
            let (r, g, b) = parse_hex_color(fc);
            if r != 0 || g != 0 || b != 0 {
                bc.base.set_foreground_color(r, g, b, 255);
            }
        }
    }
    if params.transparent_bg.as_deref() == Some("1") {
        bc.base.set_background_color(0, 0, 0, 0);
    } else if let Some(ref bg) = params.back_color {
        if !bg.is_empty() {
            let (r, g, b) = parse_hex_color(bg);
            if r != 255 || g != 255 || b != 255 {
                bc.base.set_background_color(r, g, b, 255);
            }
        }
    }
}

fn apply_colors_2d(bc: &mut barcode_pao::BarcodeBase2D, params: &DrawParams) {
    if let Some(ref fc) = params.fore_color {
        if !fc.is_empty() {
            let (r, g, b) = parse_hex_color(fc);
            if r != 0 || g != 0 || b != 0 {
                bc.base.set_foreground_color(r, g, b, 255);
            }
        }
    }
    if params.transparent_bg.as_deref() == Some("1") {
        bc.base.set_background_color(0, 0, 0, 0);
    } else if let Some(ref bg) = params.back_color {
        if !bg.is_empty() {
            let (r, g, b) = parse_hex_color(bg);
            if r != 255 || g != 255 || b != 255 {
                bc.base.set_background_color(r, g, b, 255);
            }
        }
    }
}

// ---------------------------------------------------------------------------
// Barcode drawing dispatch
// ---------------------------------------------------------------------------

enum DrawOutput {
    Base64(String),
    Svg(String),
}

fn create_and_draw(params: &DrawParams, output_format: &str) -> Result<DrawOutput, String> {
    let type_id = params.type_id.as_deref().unwrap_or("QR");
    let code = &params.code;
    let w = params.width;
    let h = params.height;
    let show_text = params.show_text == "1";
    let even_spacing = params.even_spacing == "1";
    let is_svg = output_format == FORMAT_SVG;
    let show_start_stop = params.disp_start_stop.as_deref() != Some("0");
    let extended_guard = params.extended_guard.as_deref() != Some("0");

    match type_id {
        // 2D barcodes
        "QR" => {
            let mut bc = QR::new(output_format);
            bc.error_correction = params.qr_error_level.unwrap_or(1);
            bc.version = params.qr_version.unwrap_or(0);
            apply_colors_2d(&mut bc.base_2d, params);
            bc.draw(code, w).map_err(|e| e.to_string())?;
            extract_2d_output(&bc.base_2d.base, is_svg)
        }
        "DataMatrix" => {
            let mut bc = DataMatrix::new(output_format);
            if let Some(sz) = params.datamatrix_size {
                if sz >= 0 { bc.set_code_size(sz); }
            }
            apply_colors_2d(&mut bc.base_2d, params);
            bc.draw(code, w).map_err(|e| e.to_string())?;
            extract_2d_output(&bc.base_2d.base, is_svg)
        }
        "PDF417" => {
            let mut bc = PDF417::new(output_format);
            if let Some(lvl) = params.pdf417_error_level {
                if lvl >= 0 { bc.set_error_correction_level(lvl); }
            }
            if let Some(cols) = params.pdf417_cols {
                if cols > 0 { bc.set_columns(cols); }
            }
            if let Some(rows) = params.pdf417_rows {
                if rows > 0 { bc.set_rows(rows); }
            }
            if let Some(ratio) = params.pdf417_aspect_ratio {
                if ratio > 0.0 { bc.set_aspect_ratio(ratio); }
            }
            if let Some(ref kind) = params.pdf417_size_kind {
                let mode = match kind.as_str() {
                    "COL" => 1,
                    "ROW" => 2,
                    "BOTH" => 3,
                    _ => 0,
                };
                bc.set_size_mode(mode);
            }
            apply_colors_2d(&mut bc.base_2d, params);
            bc.draw(code, w, h).map_err(|e| e.to_string())?;
            extract_2d_output(&bc.base_2d.base, is_svg)
        }

        // Standard 1D barcodes
        "Code39" => {
            let mut bc = Code39::new(output_format);
            bc.base_1d.show_text = show_text;
            bc.base_1d.text_even_spacing = even_spacing;
            bc.show_start_stop = show_start_stop;
            apply_colors_1d(&mut bc.base_1d, params);
            bc.draw(code, w, h).map_err(|e| e.to_string())?;
            extract_1d_output(&bc.base_1d.base, is_svg)
        }
        "Code93" => {
            let mut bc = Code93::new(output_format);
            bc.base_1d.show_text = show_text;
            bc.base_1d.text_even_spacing = even_spacing;
            apply_colors_1d(&mut bc.base_1d, params);
            bc.draw(code, w, h).map_err(|e| e.to_string())?;
            extract_1d_output(&bc.base_1d.base, is_svg)
        }
        "Code128" => {
            let mut bc = Code128::new(output_format);
            bc.base_1d.show_text = show_text;
            bc.base_1d.text_even_spacing = even_spacing;
            apply_colors_1d(&mut bc.base_1d, params);
            bc.draw(code, w, h).map_err(|e| e.to_string())?;
            extract_1d_output(&bc.base_1d.base, is_svg)
        }
        "NW7" => {
            let mut bc = NW7::new(output_format);
            bc.base_1d.show_text = show_text;
            bc.base_1d.text_even_spacing = even_spacing;
            bc.show_start_stop = show_start_stop;
            apply_colors_1d(&mut bc.base_1d, params);
            bc.draw(code, w, h).map_err(|e| e.to_string())?;
            extract_1d_output(&bc.base_1d.base, is_svg)
        }
        "ITF" => {
            let mut bc = ITF::new(output_format);
            bc.base_1d.show_text = show_text;
            bc.base_1d.text_even_spacing = even_spacing;
            apply_colors_1d(&mut bc.base_1d, params);
            bc.draw(code, w, h).map_err(|e| e.to_string())?;
            extract_1d_output(&bc.base_1d.base, is_svg)
        }
        "Matrix2of5" => {
            let mut bc = Matrix2of5::new(output_format);
            bc.base_1d.show_text = show_text;
            bc.base_1d.text_even_spacing = even_spacing;
            apply_colors_1d(&mut bc.base_1d, params);
            bc.draw(code, w, h).map_err(|e| e.to_string())?;
            extract_1d_output(&bc.base_1d.base, is_svg)
        }
        "NEC2of5" => {
            let mut bc = NEC2of5::new(output_format);
            bc.base_1d.show_text = show_text;
            bc.base_1d.text_even_spacing = even_spacing;
            apply_colors_1d(&mut bc.base_1d, params);
            bc.draw(code, w, h).map_err(|e| e.to_string())?;
            extract_1d_output(&bc.base_1d.base, is_svg)
        }
        "GS1-128" => {
            let mut bc = GS1128::new(output_format);
            bc.base_1d.show_text = show_text;
            bc.base_1d.text_even_spacing = even_spacing;
            apply_colors_1d(&mut bc.base_1d, params);
            bc.draw(code, w, h).map_err(|e| e.to_string())?;
            extract_1d_output(&bc.base_1d.base, is_svg)
        }

        // JAN / UPC
        "JAN13" => {
            let mut bc = JAN13::new(output_format);
            bc.base_1d.show_text = show_text;
            bc.base_1d.text_even_spacing = even_spacing;
            bc.extended_guard = extended_guard;
            apply_colors_1d(&mut bc.base_1d, params);
            bc.draw(code, w, h).map_err(|e| e.to_string())?;
            extract_1d_output(&bc.base_1d.base, is_svg)
        }
        "JAN8" => {
            let mut bc = JAN8::new(output_format);
            bc.base_1d.show_text = show_text;
            bc.base_1d.text_even_spacing = even_spacing;
            bc.extended_guard = extended_guard;
            apply_colors_1d(&mut bc.base_1d, params);
            bc.draw(code, w, h).map_err(|e| e.to_string())?;
            extract_1d_output(&bc.base_1d.base, is_svg)
        }
        "UPC-A" => {
            let mut bc = UPCA::new(output_format);
            bc.base_1d.show_text = show_text;
            bc.base_1d.text_even_spacing = even_spacing;
            bc.extended_guard = extended_guard;
            apply_colors_1d(&mut bc.base_1d, params);
            bc.draw(code, w, h).map_err(|e| e.to_string())?;
            extract_1d_output(&bc.base_1d.base, is_svg)
        }
        "UPC-E" => {
            let mut bc = UPCE::new(output_format);
            bc.base_1d.show_text = show_text;
            bc.base_1d.text_even_spacing = even_spacing;
            bc.extended_guard = extended_guard;
            apply_colors_1d(&mut bc.base_1d, params);
            bc.draw(code, w, h).map_err(|e| e.to_string())?;
            extract_1d_output(&bc.base_1d.base, is_svg)
        }

        // GS1 DataBar
        "GS1DataBar14" => {
            let sym = match params.databar14_type.as_deref() {
                Some("Stacked") => SymbolType14::Stacked,
                Some("StackedOmni") => SymbolType14::StackedOmnidirectional,
                _ => SymbolType14::Omnidirectional,
            };
            let mut bc = GS1DataBar14::new(output_format, sym);
            bc.base_1d.show_text = show_text;
            bc.base_1d.text_even_spacing = even_spacing;
            apply_colors_1d(&mut bc.base_1d, params);
            bc.draw(code, w, h).map_err(|e| e.to_string())?;
            extract_1d_output(&bc.base_1d.base, is_svg)
        }
        "GS1DataBarLimited" => {
            let mut bc = GS1DataBarLimited::new(output_format);
            bc.base_1d.show_text = show_text;
            bc.base_1d.text_even_spacing = even_spacing;
            apply_colors_1d(&mut bc.base_1d, params);
            bc.draw(code, w, h).map_err(|e| e.to_string())?;
            extract_1d_output(&bc.base_1d.base, is_svg)
        }
        "GS1DataBarExpanded" => {
            let sym = match params.databar_expanded_type.as_deref() {
                Some("Stacked") => ExpandedSymbolType::Stacked,
                _ => ExpandedSymbolType::Unstacked,
            };
            let cols = params.databar_expanded_cols.unwrap_or(2);
            let mut bc = GS1DataBarExpanded::new(output_format, sym, cols);
            bc.base_1d.show_text = show_text;
            bc.base_1d.text_even_spacing = even_spacing;
            apply_colors_1d(&mut bc.base_1d, params);
            bc.draw(code, w, h).map_err(|e| e.to_string())?;
            extract_1d_output(&bc.base_1d.base, is_svg)
        }

        // Special
        "YubinCustomer" => {
            let mut bc = YubinCustomer::new(output_format);
            apply_colors_1d(&mut bc.base_1d, params);
            bc.draw(code, h).map_err(|e| e.to_string())?;
            extract_1d_output(&bc.base_1d.base, is_svg)
        }

        _ => Err(format!("Unknown barcode type: {}", type_id)),
    }
}

fn extract_1d_output(base: &barcode_pao::BarcodeBase, is_svg: bool) -> Result<DrawOutput, String> {
    if is_svg {
        Ok(DrawOutput::Svg(base.get_svg().map_err(|e| e.to_string())?))
    } else {
        Ok(DrawOutput::Base64(base.get_image_base64().map_err(|e| e.to_string())?))
    }
}

fn extract_2d_output(base: &barcode_pao::BarcodeBase, is_svg: bool) -> Result<DrawOutput, String> {
    if is_svg {
        Ok(DrawOutput::Svg(base.get_svg().map_err(|e| e.to_string())?))
    } else {
        Ok(DrawOutput::Base64(base.get_image_base64().map_err(|e| e.to_string())?))
    }
}

// ---------------------------------------------------------------------------
// Canvas scene
// ---------------------------------------------------------------------------

fn draw_canvas_scene(params: &DrawParams) -> Result<String, String> {
    use barcode_pao::barcode_data::EncodeBarcodeData;
    use barcode_pao::renderers::tiny_skia::Renderer;
    use base64::Engine;
    use tiny_skia::*;

    let type_id = params.type_id.as_deref().unwrap_or("QR");
    let code = &params.code;
    let bw = params.width as f64;
    let bh = params.height as f64;
    let pos_x = params.x as f64;
    let pos_y = params.y as f64;

    let cw = 780u32;
    let ch = 680u32;
    let mut pixmap = Pixmap::new(cw, ch).ok_or("Failed to create pixmap")?;

    // --- Sunset gradient sky ---
    for y in 0..ch {
        let t = y as f32 / ch as f32;
        let r = (180.0 - 120.0 * t) as u8;
        let g = (100.0 + 60.0 * t) as u8;
        let b = (60.0 + 140.0 * t) as u8;
        let mut paint = Paint::default();
        paint.set_color_rgba8(r, g, b, 255);
        if let Some(rect) = Rect::from_xywh(0.0, y as f32, cw as f32, 1.0) {
            pixmap.fill_rect(rect, &paint, Transform::identity(), None);
        }
    }

    // --- Sun ---
    {
        let mut paint = Paint::default();
        paint.set_color_rgba8(255, 200, 50, 180);
        if let Some(path) = {
            let mut pb = PathBuilder::new();
            pb.push_circle(cw as f32 - 120.0, 80.0, 60.0);
            pb.finish()
        } {
            pixmap.fill_path(&path, &paint, FillRule::Winding, Transform::identity(), None);
        }
        paint.set_color_rgba8(255, 220, 100, 100);
        if let Some(path) = {
            let mut pb = PathBuilder::new();
            pb.push_circle(cw as f32 - 120.0, 80.0, 90.0);
            pb.finish()
        } {
            pixmap.fill_path(&path, &paint, FillRule::Winding, Transform::identity(), None);
        }
    }

    // --- Green grass ---
    let grass_y = ch as f32 * 0.72;
    {
        let mut paint = Paint::default();
        paint.set_color_rgba8(80, 160, 60, 255);
        if let Some(rect) = Rect::from_xywh(0.0, grass_y, cw as f32, ch as f32 - grass_y) {
            pixmap.fill_rect(rect, &paint, Transform::identity(), None);
        }
    }

    // --- Flowers ---
    let flower_positions: [(f32, f32, f32, u8, u8, u8); 15] = [
        (50.0, 0.14, 4.0, 255, 100, 100), (130.0, 0.06, 3.0, 255, 200, 50),
        (210.0, 0.18, 5.0, 200, 100, 255), (290.0, 0.08, 3.0, 255, 150, 200),
        (370.0, 0.22, 4.0, 255, 100, 100), (440.0, 0.12, 3.0, 255, 200, 50),
        (520.0, 0.16, 5.0, 200, 100, 255), (600.0, 0.04, 4.0, 255, 150, 200),
        (680.0, 0.20, 3.0, 255, 100, 100), (100.0, 0.10, 4.0, 200, 100, 255),
        (250.0, 0.24, 3.0, 255, 200, 50), (400.0, 0.02, 5.0, 255, 150, 200),
        (550.0, 0.15, 3.0, 255, 100, 100), (700.0, 0.09, 4.0, 200, 100, 255),
        (160.0, 0.21, 3.0, 255, 200, 50),
    ];
    let grass_h = ch as f32 - grass_y;
    for (fx, frac, sz, cr, cg, cb) in &flower_positions {
        let fy = grass_y + frac * grass_h;
        let mut paint = Paint::default();
        paint.set_color_rgba8(*cr, *cg, *cb, 200);
        if let Some(path) = {
            let mut pb = PathBuilder::new();
            pb.push_circle(*fx, fy, *sz);
            pb.finish()
        } {
            pixmap.fill_path(&path, &paint, FillRule::Winding, Transform::identity(), None);
        }
    }

    // --- Walking elephant ---
    draw_elephant(&mut pixmap, 100.0, grass_y - 40.0, 1.0);

    // --- Flying elephant (small) ---
    draw_elephant(&mut pixmap, cw as f32 - 200.0, 160.0, 0.6);

    // --- White card + barcode ---
    let bx = if pos_x == 0.0 && pos_y == 0.0 {
        cw as f64 / 2.0 - bw / 2.0
    } else {
        pos_x
    };
    let by = if pos_x == 0.0 && pos_y == 0.0 {
        grass_y as f64 - bh - 30.0
    } else {
        pos_y
    };

    {
        let mut paint = Paint::default();
        paint.set_color_rgba8(255, 255, 255, 220);
        if let Some(rect) = Rect::from_xywh((bx - 10.0) as f32, (by - 10.0) as f32, (bw + 20.0) as f32, (bh + 20.0) as f32) {
            pixmap.fill_rect(rect, &paint, Transform::identity(), None);
        }
    }

    // Draw barcode via renderer
    let renderer = Renderer::new();
    let mut bc: Box<dyn EncodeBarcodeData> = create_barcode_for_canvas(type_id, params);
    renderer.draw_barcode(&mut pixmap, bc.as_mut(), code, bx, by, bw, bh)
        .map_err(|e| e.to_string())?;

    // --- Title label ---
    {
        let mut paint = Paint::default();
        paint.set_color_rgba8(255, 255, 255, 220);
        if let Some(rect) = Rect::from_xywh(8.0, 8.0, 260.0, 28.0) {
            pixmap.fill_rect(rect, &paint, Transform::identity(), None);
        }
    }

    // Encode to PNG base64
    let png_data = pixmap.encode_png().map_err(|e| e.to_string())?;
    let b64 = base64::engine::general_purpose::STANDARD.encode(&png_data);
    Ok(format!("data:image/png;base64,{}", b64))
}

fn create_barcode_for_canvas(type_id: &str, params: &DrawParams) -> Box<dyn barcode_pao::barcode_data::EncodeBarcodeData> {
    let show_text = params.show_text == "1";
    match type_id {
        "QR" => Box::new(QR::new(FORMAT_PNG)),
        "DataMatrix" => Box::new(DataMatrix::new(FORMAT_PNG)),
        "PDF417" => Box::new(PDF417::new(FORMAT_PNG)),
        "Code39" => { let mut bc = Code39::new(FORMAT_PNG); bc.base_1d.show_text = show_text; Box::new(bc) }
        "Code93" => { let mut bc = Code93::new(FORMAT_PNG); bc.base_1d.show_text = show_text; Box::new(bc) }
        "Code128" => { let mut bc = Code128::new(FORMAT_PNG); bc.base_1d.show_text = show_text; Box::new(bc) }
        "NW7" => { let mut bc = NW7::new(FORMAT_PNG); bc.base_1d.show_text = show_text; Box::new(bc) }
        "ITF" => { let mut bc = ITF::new(FORMAT_PNG); bc.base_1d.show_text = show_text; Box::new(bc) }
        "Matrix2of5" => { let mut bc = Matrix2of5::new(FORMAT_PNG); bc.base_1d.show_text = show_text; Box::new(bc) }
        "NEC2of5" => { let mut bc = NEC2of5::new(FORMAT_PNG); bc.base_1d.show_text = show_text; Box::new(bc) }
        "GS1-128" => { let mut bc = GS1128::new(FORMAT_PNG); bc.base_1d.show_text = show_text; Box::new(bc) }
        "JAN13" => { let mut bc = JAN13::new(FORMAT_PNG); bc.base_1d.show_text = show_text; Box::new(bc) }
        "JAN8" => { let mut bc = JAN8::new(FORMAT_PNG); bc.base_1d.show_text = show_text; Box::new(bc) }
        "UPC-A" => { let mut bc = UPCA::new(FORMAT_PNG); bc.base_1d.show_text = show_text; Box::new(bc) }
        "UPC-E" => { let mut bc = UPCE::new(FORMAT_PNG); bc.base_1d.show_text = show_text; Box::new(bc) }
        "GS1DataBar14" => Box::new(GS1DataBar14::new(FORMAT_PNG, SymbolType14::Omnidirectional)),
        "GS1DataBarLimited" => Box::new(GS1DataBarLimited::new(FORMAT_PNG)),
        "GS1DataBarExpanded" => Box::new(GS1DataBarExpanded::new(FORMAT_PNG, ExpandedSymbolType::Unstacked, 2)),
        "YubinCustomer" => Box::new(YubinCustomer::new(FORMAT_PNG)),
        _ => Box::new(QR::new(FORMAT_PNG)),
    }
}

fn draw_elephant(pixmap: &mut tiny_skia::Pixmap, ex: f32, ey: f32, s: f32) {
    use tiny_skia::*;

    // Body ellipse
    let mut paint = Paint::default();
    paint.set_color_rgba8(140, 155, 175, 230);
    if let Some(path) = {
        let mut pb = PathBuilder::new();
        pb.push_oval(Rect::from_xywh(ex - 50.0 * s, ey - 30.0 * s, 100.0 * s, 60.0 * s).unwrap());
        pb.finish()
    } {
        pixmap.fill_path(&path, &paint, FillRule::Winding, Transform::identity(), None);
    }

    // Head
    if let Some(path) = {
        let mut pb = PathBuilder::new();
        pb.push_oval(Rect::from_xywh(ex + 20.0 * s, ey - 42.0 * s, 50.0 * s, 44.0 * s).unwrap());
        pb.finish()
    } {
        pixmap.fill_path(&path, &paint, FillRule::Winding, Transform::identity(), None);
    }

    // Ear
    paint.set_color_rgba8(120, 135, 160, 200);
    if let Some(path) = {
        let mut pb = PathBuilder::new();
        pb.push_oval(Rect::from_xywh(ex + 42.0 * s, ey - 50.0 * s, 36.0 * s, 40.0 * s).unwrap());
        pb.finish()
    } {
        pixmap.fill_path(&path, &paint, FillRule::Winding, Transform::identity(), None);
    }

    // Eye
    paint.set_color_rgba8(255, 255, 255, 255);
    if let Some(path) = {
        let mut pb = PathBuilder::new();
        pb.push_circle(ex + 45.0 * s, ey - 28.0 * s, 4.0 * s);
        pb.finish()
    } {
        pixmap.fill_path(&path, &paint, FillRule::Winding, Transform::identity(), None);
    }
    paint.set_color_rgba8(20, 20, 40, 255);
    if let Some(path) = {
        let mut pb = PathBuilder::new();
        pb.push_circle(ex + 46.0 * s, ey - 27.0 * s, 2.0 * s);
        pb.finish()
    } {
        pixmap.fill_path(&path, &paint, FillRule::Winding, Transform::identity(), None);
    }

    // Legs
    paint.set_color_rgba8(130, 145, 170, 220);
    for lx in &[-25.0_f32, -5.0, 15.0, 35.0] {
        if let Some(rect) = Rect::from_xywh(ex + lx * s, ey + 25.0 * s, 14.0 * s, 28.0 * s) {
            pixmap.fill_rect(rect, &paint, Transform::identity(), None);
        }
    }
}

// ---------------------------------------------------------------------------
// PDF report
// ---------------------------------------------------------------------------

fn mm(v: f64) -> printpdf::Mm {
    printpdf::Mm(v as f32)
}

fn generate_pdf_report(type_id: &str, code: &str, width: i32, height: i32) -> Result<Vec<u8>, String> {
    use barcode_pao::renderers::printpdf::Renderer;
    use printpdf::*;
    use printpdf::path::{PaintMode, WindingOrder};

    let page_height: f64 = 297.0;
    let (doc, page1, layer1) = PdfDocument::new("Barcode Report", mm(210.0), mm(page_height), "Layer 1");
    let layer = doc.get_page(page1).get_layer(layer1);
    let renderer = Renderer::new();

    // Header background
    layer.set_fill_color(printpdf::Color::Rgb(Rgb::new(30.0 / 255.0, 64.0 / 255.0, 175.0 / 255.0, None)));
    let hdr = vec![
        (Point::new(mm(0.0), mm(page_height)), false),
        (Point::new(mm(210.0), mm(page_height)), false),
        (Point::new(mm(210.0), mm(page_height - 22.0)), false),
        (Point::new(mm(0.0), mm(page_height - 22.0)), false),
    ];
    layer.add_polygon(Polygon { rings: vec![hdr], mode: PaintMode::Fill, winding_order: WindingOrder::NonZero });

    // Header text
    let font_bold = doc.add_builtin_font(BuiltinFont::HelveticaBold).map_err(|e| e.to_string())?;
    let font_reg = doc.add_builtin_font(BuiltinFont::Helvetica).map_err(|e| e.to_string())?;

    layer.set_fill_color(printpdf::Color::Rgb(Rgb::new(1.0, 1.0, 1.0, None)));
    layer.use_text("BARCODE REPORT", 18.0, mm(15.0), mm(page_height - 14.0), &font_bold);

    let today = chrono::Local::now().format("%Y-%m-%d").to_string();
    layer.use_text(&format!("Date: {}  |  Type: {}", today, type_id), 8.0, mm(15.0), mm(page_height - 20.0), &font_reg);

    // Main barcode
    layer.set_fill_color(printpdf::Color::Rgb(Rgb::new(0.0, 0.0, 0.0, None)));
    layer.use_text(&format!("Main Barcode — {}", type_id), 10.0, mm(15.0), mm(page_height - 32.0), &font_bold);

    let label = if code.len() > 50 { &code[..50] } else { code };
    layer.use_text(&format!("Data: {}", label), 7.0, mm(15.0), mm(page_height - 37.0), &font_reg);

    // Background for main barcode
    layer.set_fill_color(printpdf::Color::Rgb(Rgb::new(248.0 / 255.0, 250.0 / 255.0, 252.0 / 255.0, None)));
    let bc_bg = vec![
        (Point::new(mm(12.0), mm(page_height - 40.0)), false),
        (Point::new(mm(198.0), mm(page_height - 40.0)), false),
        (Point::new(mm(198.0), mm(page_height - 80.0)), false),
        (Point::new(mm(12.0), mm(page_height - 80.0)), false),
    ];
    layer.add_polygon(Polygon { rings: vec![bc_bg], mode: PaintMode::Fill, winding_order: WindingOrder::NonZero });

    // Draw main barcode
    let mut bc: Box<dyn barcode_pao::barcode_data::EncodeBarcodeData> = match type_id {
        "QR" => Box::new(QR::new(FORMAT_PNG)),
        "DataMatrix" => Box::new(DataMatrix::new(FORMAT_PNG)),
        "PDF417" => Box::new(PDF417::new(FORMAT_PNG)),
        "Code39" => Box::new(Code39::new(FORMAT_PNG)),
        "Code93" => Box::new(Code93::new(FORMAT_PNG)),
        "Code128" => Box::new(Code128::new(FORMAT_PNG)),
        "NW7" => Box::new(NW7::new(FORMAT_PNG)),
        "ITF" => Box::new(ITF::new(FORMAT_PNG)),
        "Matrix2of5" => Box::new(Matrix2of5::new(FORMAT_PNG)),
        "NEC2of5" => Box::new(NEC2of5::new(FORMAT_PNG)),
        "GS1-128" => Box::new(GS1128::new(FORMAT_PNG)),
        "JAN13" => Box::new(JAN13::new(FORMAT_PNG)),
        "JAN8" => Box::new(JAN8::new(FORMAT_PNG)),
        "UPC-A" => Box::new(UPCA::new(FORMAT_PNG)),
        "UPC-E" => Box::new(UPCE::new(FORMAT_PNG)),
        "GS1DataBar14" => Box::new(GS1DataBar14::new(FORMAT_PNG, SymbolType14::Omnidirectional)),
        "GS1DataBarLimited" => Box::new(GS1DataBarLimited::new(FORMAT_PNG)),
        "GS1DataBarExpanded" => Box::new(GS1DataBarExpanded::new(FORMAT_PNG, ExpandedSymbolType::Unstacked, 2)),
        "YubinCustomer" => Box::new(YubinCustomer::new(FORMAT_PNG)),
        _ => Box::new(QR::new(FORMAT_PNG)),
    };
    let bc_w = (width as f64).min(170.0);
    let bc_h = (height as f64).min(35.0);
    renderer.draw_barcode(&layer, bc.as_mut(), code, 20.0, 43.0, bc_w, bc_h, page_height)
        .map_err(|e| e.to_string())?;

    // Sample barcodes table
    let table_y: f64 = page_height - 90.0;
    layer.set_fill_color(printpdf::Color::Rgb(Rgb::new(30.0 / 255.0, 64.0 / 255.0, 175.0 / 255.0, None)));
    let th = vec![
        (Point::new(mm(12.0), mm(table_y)), false),
        (Point::new(mm(198.0), mm(table_y)), false),
        (Point::new(mm(198.0), mm(table_y - 7.0)), false),
        (Point::new(mm(12.0), mm(table_y - 7.0)), false),
    ];
    layer.add_polygon(Polygon { rings: vec![th], mode: PaintMode::Fill, winding_order: WindingOrder::NonZero });
    layer.set_fill_color(printpdf::Color::Rgb(Rgb::new(1.0, 1.0, 1.0, None)));
    layer.use_text("Type", 7.0, mm(15.0), mm(table_y - 5.0), &font_bold);
    layer.use_text("Data", 7.0, mm(55.0), mm(table_y - 5.0), &font_bold);
    layer.use_text("Barcode", 7.0, mm(120.0), mm(table_y - 5.0), &font_bold);

    // Sample entries
    let samples: [(&str, &str, fn(&str) -> Box<dyn barcode_pao::barcode_data::EncodeBarcodeData>); 4] = [
        ("QR Code", "https://pao.ac", |f| Box::new(QR::new(f))),
        ("Code128", "SAMPLE-2026", |f| Box::new(Code128::new(f))),
        ("JAN-13", "4901234567894", |f| Box::new(JAN13::new(f))),
        ("DataMatrix", "Barcode.Rust", |f| Box::new(DataMatrix::new(f))),
    ];

    for (i, (name, data, ctor)) in samples.iter().enumerate() {
        let ry: f64 = table_y - 7.0 - (i as f64 + 1.0) * 22.0;

        if i % 2 == 0 {
            layer.set_fill_color(printpdf::Color::Rgb(Rgb::new(248.0 / 255.0, 250.0 / 255.0, 252.0 / 255.0, None)));
            let row = vec![
                (Point::new(mm(12.0), mm(ry + 22.0)), false),
                (Point::new(mm(198.0), mm(ry + 22.0)), false),
                (Point::new(mm(198.0), mm(ry)), false),
                (Point::new(mm(12.0), mm(ry)), false),
            ];
            layer.add_polygon(Polygon { rings: vec![row], mode: PaintMode::Fill, winding_order: WindingOrder::NonZero });
        }

        layer.set_fill_color(printpdf::Color::Rgb(Rgb::new(0.0, 0.0, 0.0, None)));
        layer.use_text(*name, 7.0, mm(15.0), mm(ry + 10.0), &font_bold);
        layer.use_text(*data, 6.0, mm(55.0), mm(ry + 10.0), &font_reg);

        let mut sbc = ctor(FORMAT_PNG);
        let _ = renderer.draw_barcode(&layer, sbc.as_mut(), data, 120.0, page_height - ry - 20.0, 60.0, 16.0, page_height);
    }

    // Footer
    layer.set_fill_color(printpdf::Color::Rgb(Rgb::new(148.0 / 255.0, 163.0 / 255.0, 184.0 / 255.0, None)));
    layer.use_text("Generated by Barcode.Rust + PrintpdfRenderer — pao.ac", 6.0, mm(15.0), mm(12.0), &font_reg);

    doc.save_to_bytes().map_err(|e| e.to_string())
}

// ---------------------------------------------------------------------------
// Route handlers
// ---------------------------------------------------------------------------

async fn index_handler() -> Html<String> {
    let template = include_str!("../templates/index.html");
    let html = template.replace("__BARCODE_TYPES_JSON__", BARCODE_TYPES_JSON);
    Html(html)
}

async fn api_types_handler() -> Json<serde_json::Value> {
    let v: serde_json::Value = serde_json::from_str(BARCODE_TYPES_JSON).unwrap();
    Json(v)
}

async fn draw_base64_handler(Form(params): Form<DrawParams>) -> Json<DrawResponse> {
    match create_and_draw(&params, FORMAT_PNG) {
        Ok(DrawOutput::Base64(data)) => Json(DrawResponse::ok_base64(data)),
        Ok(DrawOutput::Svg(_)) => Json(DrawResponse::err("Unexpected SVG output".to_string())),
        Err(e) => Json(DrawResponse::err(e)),
    }
}

async fn draw_svg_handler(Form(params): Form<DrawParams>) -> Json<DrawResponse> {
    match create_and_draw(&params, FORMAT_SVG) {
        Ok(DrawOutput::Svg(data)) => Json(DrawResponse::ok_svg(data)),
        Ok(DrawOutput::Base64(_)) => Json(DrawResponse::err("Unexpected Base64 output".to_string())),
        Err(e) => Json(DrawResponse::err(e)),
    }
}

async fn draw_canvas_handler(Form(params): Form<DrawParams>) -> Json<DrawResponse> {
    match draw_canvas_scene(&params) {
        Ok(b64) => Json(DrawResponse::ok_base64(b64)),
        Err(e) => Json(DrawResponse::err(e)),
    }
}

async fn pdf_handler(Query(query): Query<PdfQuery>) -> impl IntoResponse {
    let type_id = query.type_id.as_deref().unwrap_or("QR");
    let code = query.code.as_deref().unwrap_or("https://www.pao.ac/");
    let width = query.width.unwrap_or(300);
    let height = query.height.unwrap_or(100);

    match generate_pdf_report(type_id, code, width, height) {
        Ok(bytes) => {
            let headers = [
                (header::CONTENT_TYPE, "application/pdf"),
                (header::CONTENT_DISPOSITION, "inline; filename=barcode_report.pdf"),
            ];
            (StatusCode::OK, headers, bytes).into_response()
        }
        Err(e) => (StatusCode::INTERNAL_SERVER_ERROR, e).into_response(),
    }
}

// ---------------------------------------------------------------------------
// main
// ---------------------------------------------------------------------------

#[tokio::main]
async fn main() {
    let app = Router::new()
        .route("/", get(index_handler))
        .route("/api/types", get(api_types_handler))
        .route("/draw-base64", post(draw_base64_handler))
        .route("/draw-svg", post(draw_svg_handler))
        .route("/draw-canvas", post(draw_canvas_handler))
        .route("/pdf", get(pdf_handler));

    let addr = "0.0.0.0:5721";
    println!("Barcode.Rust All-in-One running at http://localhost:5721/");

    let listener = tokio::net::TcpListener::bind(addr).await.unwrap();
    axum::serve(listener, app).await.unwrap();
}
