# Barcode.Ruby - Easy 2 Steps
#
# QR code generation in just 2 steps using Sinatra.
# 4 output modes: PNG, SVG, PDF (Prawn), Canvas (ChunkyPNG)
#
#   bundle exec ruby app.rb
#   -> http://localhost:5740

require "sinatra"
require "json"
require "base64"
require "date"

# Load Pure Ruby barcode library
$LOAD_PATH.unshift(File.join(__dir__, "..", "barcode_pao", "lib"))
require "barcode_pao"

set :port, 5740
set :bind, "0.0.0.0"
set :host_authorization, permitted_hosts: [".pao.ac", "localhost"]

get "/" do
  erb :index
end

post "/draw-base64" do
  content_type :json
  code = params[:code]
  code = "https://www.pao.ac/" if code.nil? || code.empty?
  begin
    qr = BarcodePao::QRCode.new(BarcodePao::FORMAT_PNG)
    qr.draw(code, 300)
    { ok: true, base64: qr.get_image_base64 }.to_json
  rescue => e
    { ok: false, error: e.message }.to_json
  end
end

post "/draw-svg" do
  content_type :json
  code = params[:code]
  code = "https://www.pao.ac/" if code.nil? || code.empty?
  begin
    qr = BarcodePao::QRCode.new(BarcodePao::FORMAT_SVG)
    qr.draw(code, 300)
    { ok: true, svg: qr.get_svg }.to_json
  rescue => e
    { ok: false, error: e.message }.to_json
  end
end

post "/draw-canvas" do
  content_type :json
  code = params[:code]
  code = "https://www.pao.ac/" if code.nil? || code.empty?
  begin
    b64 = draw_canvas_fun(code)
    { ok: true, base64: b64 }.to_json
  rescue => e
    { ok: false, error: e.message }.to_json
  end
end

get "/pdf" do
  code = params[:code]
  code = "https://www.pao.ac/" if code.nil? || code.empty?
  begin
    pdf_bytes = generate_invoice_pdf(code)
    content_type "application/pdf"
    headers "Content-Disposition" => "inline; filename=invoice_sample.pdf"
    pdf_bytes
  rescue => e
    content_type :json
    status 500
    { ok: false, error: e.message }.to_json
  end
end

# Canvas: ChunkyPNG with night sky + elephant + barcodes
def draw_canvas_fun(code)
  require "chunky_png"

  w, h = 900, 520
  canvas = ChunkyPNG::Image.new(w, h, ChunkyPNG::Color::BLACK)

  # Night sky gradient
  h.times do |i|
    t = i.to_f / h
    r = (30 + 20 * t).to_i
    g = (60 + 40 * t).to_i
    b = (120 + 80 * t).to_i
    w.times { |x| canvas[x, i] = ChunkyPNG::Color.rgba(r, g, b, 255) }
  end

  # Stars
  rng = Random.new(42)
  25.times do
    sx = rng.rand(w - 20) + 10
    sy = rng.rand(h - 20) + 10
    sz = rng.rand(2..5)
    alpha = rng.rand(80..200)
    fill_circle(canvas, sx, sy, sz, ChunkyPNG::Color.rgba(255, 255, 200, alpha))
  end

  # Elephant (simplified)
  ex, ey = 150, 290
  s = 1.4
  fill_ellipse(canvas, ex, ey, (55*s).to_i, (35*s).to_i, ChunkyPNG::Color.rgba(180, 190, 210, 220))
  fill_ellipse(canvas, (ex+60*s).to_i, (ey-25*s).to_i, (30*s).to_i, (30*s).to_i, ChunkyPNG::Color.rgba(180, 190, 210, 220))
  fill_ellipse(canvas, (ex+85*s).to_i, (ey-35*s).to_i, (20*s).to_i, (25*s).to_i, ChunkyPNG::Color.rgba(160, 170, 195, 200))
  fill_circle(canvas, (ex+55*s).to_i, (ey-35*s).to_i, (4*s).to_i, ChunkyPNG::Color::WHITE)
  fill_circle(canvas, (ex+57*s).to_i, (ey-33*s).to_i, (2*s).to_i, ChunkyPNG::Color.rgba(30, 30, 50, 255))
  # Legs
  [-30, -5, 25, 50].each do |lx|
    fill_rect(canvas, (lx*s).to_i + ex, (ey+30*s).to_i, 20, (30*s).to_i, ChunkyPNG::Color.rgba(160, 170, 195, 220))
  end

  # White card area
  card_x, card_y = 340, 40
  card_w, card_h = 520, 440
  fill_rect(canvas, card_x, card_y, card_w, card_h, ChunkyPNG::Color.rgba(255, 255, 255, 230))

  # Draw QR code onto card
  qr = BarcodePao::QRCode.new(BarcodePao::FORMAT_PNG)
  qr.draw(code, 220)
  qr_bytes = qr.get_image_memory
  if qr_bytes && !qr_bytes.empty?
    qr_img = ChunkyPNG::Image.from_blob(qr_bytes)
    qr_x = card_x + (card_w - qr_img.width) / 2
    qr_y = card_y + 60
    canvas.compose!(qr_img, qr_x, qr_y)
  end

  # Draw Code128 onto card
  c128 = BarcodePao::Code128.new(BarcodePao::FORMAT_PNG)
  c128.show_text = true
  c128.draw("BARCODE-2026", 380, 60)
  c128_bytes = c128.get_image_memory
  if c128_bytes && !c128_bytes.empty?
    c128_img = ChunkyPNG::Image.from_blob(c128_bytes)
    c128_x = card_x + (card_w - c128_img.width) / 2
    c128_y = card_y + 330
    canvas.compose!(c128_img, c128_x, c128_y)
  end

  # Encode to base64
  blob = canvas.to_blob(:fast_rgba)
  png_blob = canvas.to_blob
  b64 = Base64.strict_encode64(png_blob)
  "data:image/png;base64,#{b64}"
end

# PDF: DELIVERY NOTE invoice using Prawn
def generate_invoice_pdf(code)
  require "prawn"

  pdf = Prawn::Document.new(page_size: "A4", margin: 0)
  pw = 595.28
  ph = 841.89

  today = Date.today.iso8601
  inv_no = "INV-#{Date.today.strftime('%Y-%m%d')}"

  # Header (blue)
  pdf.fill_color "1E40AF"
  pdf.fill_rectangle [0, ph], pw, 82
  pdf.fill_color "7C3AED"
  pdf.fill_rectangle [0, ph - 82], pw, 4

  pdf.fill_color "FFFFFF"
  pdf.font_size 24
  pdf.draw_text "DELIVERY NOTE", at: [40, ph - 48]

  pdf.font_size 9
  pdf.draw_text "Pao@Office", at: [pw - 160, ph - 35]
  pdf.draw_text "No: #{inv_no}", at: [pw - 160, ph - 48]
  pdf.draw_text "Date: #{today}", at: [pw - 160, ph - 61]
  pdf.draw_text "Tokyo, Japan", at: [pw - 160, ph - 74]

  # Recipient
  pdf.fill_color "000000"
  pdf.font_size 11
  pdf.draw_text "Sample Corporation", at: [40, ph - 115]

  # Barcode area background
  y = ph - 155
  pdf.fill_color "F1F5F9"
  pdf.fill_rectangle [35, y], pw - 70, 130

  # Code128 barcode
  pdf.fill_color "475569"
  pdf.font_size 8
  pdf.draw_text "INVOICE BARCODE (Code128)", at: [50, y - 15]

  c128 = BarcodePao::Code128.new(BarcodePao::FORMAT_PNG)
  c128.show_text = true
  c128.draw(inv_no, 220, 40)
  c128_bytes = c128.get_image_memory
  if c128_bytes && !c128_bytes.empty?
    c128_io = StringIO.new(c128_bytes)
    pdf.image c128_io, at: [50, y - 25], width: 220
  end

  # QR Code
  pdf.fill_color "475569"
  pdf.draw_text "QR CODE", at: [310, y - 15]

  qr = BarcodePao::QRCode.new(BarcodePao::FORMAT_PNG)
  qr.draw(code, 100)
  qr_bytes = qr.get_image_memory
  if qr_bytes && !qr_bytes.empty?
    qr_io = StringIO.new(qr_bytes)
    pdf.image qr_io, at: [310, y - 25], width: 100
  end

  pdf.fill_color "94A3B8"
  pdf.font_size 7
  label = code.length > 40 ? code[0..39] : code
  pdf.draw_text "Data: #{label}", at: [310, y - 128]

  # Invoice table
  y = ph - 315
  # Header row
  pdf.fill_color "1E40AF"
  pdf.fill_rectangle [40, y], pw - 80, 22
  pdf.fill_color "FFFFFF"
  pdf.font_size 9
  [["Item", 50], ["Qty", 330], ["Unit Price", 420], ["Amount", 520]].each do |text, x|
    pdf.draw_text text, at: [x, y - 16]
  end

  # Data rows
  items = [
    ["Barcode.Ruby Pure Ruby Edition", "1", "\\33,000", "\\33,000"],
    ["Annual Support (3 years)", "1", "\\9,900", "\\9,900"],
  ]
  pdf.fill_color "000000"
  pdf.font_size 9
  items.each_with_index do |item, i|
    ry = y - 22 * (i + 1)
    if i.even?
      pdf.fill_color "F8FAFC"
      pdf.fill_rectangle [40, ry], pw - 80, 22
      pdf.fill_color "000000"
    end
    pdf.draw_text item[0], at: [50, ry - 16]
    pdf.draw_text item[1], at: [340, ry - 16]
    pdf.draw_text item[2], at: [420, ry - 16]
    pdf.draw_text item[3], at: [520, ry - 16]
  end

  # Total
  ry = y - 22 * (items.length + 1) - 10
  pdf.stroke_color "1E40AF"
  pdf.line_width 2
  pdf.stroke_line [350, ry], [pw - 40, ry]
  pdf.fill_color "1E40AF"
  pdf.font_size 13
  pdf.draw_text "Total:  \\42,900", at: [pw - 170, ry - 18]

  # JAN-13
  ry -= 50
  pdf.fill_color "475569"
  pdf.font_size 8
  pdf.draw_text "PRODUCT BARCODE (JAN-13)", at: [40, ry]

  jan = BarcodePao::JAN13.new(BarcodePao::FORMAT_PNG)
  jan.show_text = true
  jan.draw("490123456789", 180, 45)
  jan_bytes = jan.get_image_memory
  if jan_bytes && !jan_bytes.empty?
    jan_io = StringIO.new(jan_bytes)
    pdf.image jan_io, at: [40, ry - 10], width: 180
  end

  # Footer
  pdf.stroke_color "E2E8F0"
  pdf.line_width 0.5
  pdf.stroke_line [40, 55], [pw - 40, 55]
  pdf.fill_color "94A3B8"
  pdf.font_size 7
  pdf.draw_text "Generated by Barcode.Ruby — barcode_pao + Prawn", at: [40, 40]
  pdf.draw_text "pao.ac", at: [pw - 70, 40]

  pdf.render
end

# ChunkyPNG drawing helpers
def fill_rect(canvas, x, y, w, h, color)
  x1 = [x, 0].max
  y1 = [y, 0].max
  x2 = [x + w - 1, canvas.width - 1].min
  y2 = [y + h - 1, canvas.height - 1].min
  (y1..y2).each do |py|
    (x1..x2).each do |px|
      canvas.compose_pixel(px, py, color)
    end
  end
end

def fill_circle(canvas, cx, cy, r, color)
  (cy - r..cy + r).each do |py|
    next if py < 0 || py >= canvas.height
    (cx - r..cx + r).each do |px|
      next if px < 0 || px >= canvas.width
      if (px - cx)**2 + (py - cy)**2 <= r**2
        canvas.compose_pixel(px, py, color)
      end
    end
  end
end

def fill_ellipse(canvas, cx, cy, rx, ry, color)
  return if rx <= 0 || ry <= 0
  (cy - ry..cy + ry).each do |py|
    next if py < 0 || py >= canvas.height
    (cx - rx..cx + rx).each do |px|
      next if px < 0 || px >= canvas.width
      dx = (px - cx).to_f / rx
      dy = (py - cy).to_f / ry
      if dx * dx + dy * dy <= 1.0
        canvas.compose_pixel(px, py, color)
      end
    end
  end
end

__END__

@@ index
<!DOCTYPE html>
<html lang="ja">
<head>
<meta charset="UTF-8">
<meta name="viewport" content="width=device-width, initial-scale=1.0">
<title>Barcode.Ruby - Easy 2 Steps</title>
<style>
*{margin:0;padding:0;box-sizing:border-box}
body{font-family:'Segoe UI',system-ui,sans-serif;background:#f0f4f8;color:#1e293b}
header{background:linear-gradient(135deg,#CC342D,#8B0000);color:#fff;padding:24px 32px;position:relative;overflow:hidden}
header h1{font-size:22px;font-weight:700;letter-spacing:.5px}
header p{font-size:13px;opacity:.85;margin-top:4px}
.badge{display:inline-block;background:rgba(255,255,255,.2);border-radius:12px;padding:3px 12px;font-size:11px;margin-top:6px}
.container{max-width:960px;margin:0 auto;padding:24px}
.steps{display:flex;gap:16px;margin-bottom:20px}
.step{flex:1;background:#fff;border-radius:10px;padding:16px;box-shadow:0 1px 3px rgba(0,0,0,.08);border-left:4px solid #CC342D}
.step-num{display:inline-flex;align-items:center;justify-content:center;width:28px;height:28px;border-radius:50%;background:#CC342D;color:#fff;font-size:13px;font-weight:700;margin-right:8px}
.step-title{font-weight:600;font-size:14px;color:#CC342D}
.step-desc{font-size:12px;color:#64748b;margin-top:4px}
.main-row{display:flex;gap:20px}
.left-panel{flex:1;min-width:0}
.right-panel{width:360px;flex-shrink:0}
.card{background:#fff;border-radius:10px;padding:20px;box-shadow:0 1px 3px rgba(0,0,0,.08);margin-bottom:16px}
label{display:block;font-size:13px;font-weight:600;color:#475569;margin-bottom:6px}
input[type=text]{width:100%;padding:10px 14px;border:1.5px solid #e2e8f0;border-radius:8px;font-size:14px;transition:border .2s}
input[type=text]:focus{outline:none;border-color:#CC342D;box-shadow:0 0 0 3px rgba(204,52,45,.15)}
.mode-row{display:flex;gap:8px;margin:16px 0 12px}
.mode-btn{flex:1;padding:10px 8px;border:1.5px solid #e2e8f0;border-radius:8px;background:#fff;cursor:pointer;font-size:12px;font-weight:600;text-align:center;transition:all .15s;color:#64748b}
.mode-btn.active{border-color:#CC342D;background:#fef2f2;color:#CC342D}
.mode-btn:hover{border-color:#f87171}
.code-panel{background:#1e293b;border-radius:8px;padding:14px 16px;margin-top:12px}
.code-panel pre{font-family:'Consolas','Courier New',monospace;font-size:12px;color:#e2e8f0;line-height:1.7;white-space:pre-wrap;overflow-x:auto}
.code-panel .kw{color:#CC342D}
.code-panel .fn{color:#38bdf8}
.code-panel .str{color:#86efac}
.code-panel .cm{color:#64748b}
.preview-card{background:#fff;border-radius:10px;box-shadow:0 1px 3px rgba(0,0,0,.08);position:sticky;top:20px}
.preview-header{padding:14px 16px;border-bottom:1px solid #f1f5f9;display:flex;align-items:center;justify-content:space-between}
.preview-header h3{font-size:14px;color:#CC342D}
.preview-body{padding:20px;display:flex;align-items:center;justify-content:center;min-height:360px}
.preview-body img{max-width:100%;image-rendering:pixelated}
.preview-body .svg-container{max-width:100%;overflow:auto}
.preview-body .svg-container svg{max-width:100%;height:auto}
.preview-body iframe{width:100%;height:500px;border:1px solid #e2e8f0;border-radius:6px}
.empty-preview{color:#94a3b8;font-size:13px;text-align:center}
.loading{color:#CC342D;font-size:13px}
footer{text-align:center;padding:20px;font-size:12px;color:#94a3b8}
@media(max-width:768px){.main-row{flex-direction:column}.right-panel{width:100%}.steps{flex-direction:column}}
</style>
</head>
<body>
<header>
  <h1>Barcode.Ruby &#x2014; Easy 2 Steps</h1>
  <p>Pure Ruby barcode library &#x2014; Sinatra REST API</p>
  <span class="badge">Ruby + Sinatra</span>
</header>

<div class="container">
  <div class="steps">
    <div class="step">
      <span class="step-num">1</span>
      <span class="step-title">Create barcode</span>
      <div class="step-desc">qr = BarcodePao::QRCode.new("png")</div>
    </div>
    <div class="step">
      <span class="step-num">2</span>
      <span class="step-title">Draw!</span>
      <div class="step-desc">qr.draw(code, 300)</div>
    </div>
  </div>

  <div class="main-row">
    <div class="left-panel">
      <div class="card">
        <label for="code-input">Data</label>
        <input type="text" id="code-input" value="https://www.pao.ac/" placeholder="Enter data...">

        <div class="mode-row">
          <button class="mode-btn active" data-mode="base64" onclick="setMode('base64')">PNG (Base64)</button>
          <button class="mode-btn" data-mode="svg" onclick="setMode('svg')">SVG</button>
          <button class="mode-btn" data-mode="pdf" onclick="setMode('pdf')">PDF</button>
          <button class="mode-btn" data-mode="canvas" onclick="setMode('canvas')">Canvas</button>
        </div>

        <div class="code-panel" id="code-panel-base64">
<pre><span class="cm"># Step 1: Create barcode</span>
qr = <span class="kw">BarcodePao</span>::<span class="fn">QRCode</span>.<span class="fn">new</span>(<span class="str">"png"</span>)
<span class="cm"># Step 2: Draw!</span>
qr.<span class="fn">draw</span>(code, <span class="str">300</span>)
base64 = qr.<span class="fn">get_image_base64</span></pre>
        </div>
        <div class="code-panel" id="code-panel-svg" style="display:none">
<pre><span class="cm"># Step 1: Create barcode (SVG mode)</span>
qr = <span class="kw">BarcodePao</span>::<span class="fn">QRCode</span>.<span class="fn">new</span>(<span class="str">"svg"</span>)
<span class="cm"># Step 2: Draw!</span>
qr.<span class="fn">draw</span>(code, <span class="str">300</span>)
svg = qr.<span class="fn">get_svg</span></pre>
        </div>
        <div class="code-panel" id="code-panel-pdf" style="display:none">
<pre><span class="cm"># Prawn PDF with barcode image</span>
<span class="kw">require</span> <span class="str">"prawn"</span>

qr = <span class="kw">BarcodePao</span>::<span class="fn">QRCode</span>.<span class="fn">new</span>(<span class="str">"png"</span>)
qr.<span class="fn">draw</span>(code, <span class="str">200</span>)
bytes = qr.<span class="fn">get_image_memory</span>

pdf = <span class="kw">Prawn</span>::<span class="fn">Document</span>.<span class="fn">new</span>
pdf.<span class="fn">image</span> <span class="fn">StringIO</span>.<span class="fn">new</span>(bytes), <span class="str">width: 200</span>
pdf.<span class="fn">render_file</span>(<span class="str">"output.pdf"</span>)</pre>
        </div>
        <div class="code-panel" id="code-panel-canvas" style="display:none">
<pre><span class="cm"># ChunkyPNG canvas + barcode compositing</span>
<span class="kw">require</span> <span class="str">"chunky_png"</span>

canvas = <span class="kw">ChunkyPNG</span>::<span class="fn">Image</span>.<span class="fn">new</span>(<span class="str">900</span>, <span class="str">520</span>)
<span class="cm"># ... draw background ...</span>
qr = <span class="kw">BarcodePao</span>::<span class="fn">QRCode</span>.<span class="fn">new</span>(<span class="str">"png"</span>)
qr.<span class="fn">draw</span>(code, <span class="str">220</span>)
qr_img = <span class="kw">ChunkyPNG</span>::<span class="fn">Image</span>.<span class="fn">from_blob</span>(qr.<span class="fn">get_image_memory</span>)
canvas.<span class="fn">compose!</span>(qr_img, x, y)</pre>
        </div>
      </div>
    </div>

    <div class="right-panel">
      <div class="preview-card">
        <div class="preview-header">
          <h3>Preview</h3>
          <span id="mode-badge" style="font-size:11px;color:#64748b">PNG</span>
        </div>
        <div class="preview-body" id="preview-body">
          <div class="empty-preview">Enter data and it will render automatically</div>
        </div>
      </div>
    </div>
  </div>
</div>

<footer>Barcode.Ruby &#x2014; Pure Ruby barcode library &#x2014; pao.ac</footer>

<script>
let currentMode = 'base64';
let debounceTimer = null;

const codeInput = document.getElementById('code-input');
const previewBody = document.getElementById('preview-body');
const modeBadge = document.getElementById('mode-badge');

const modeLabels = { base64: 'PNG', svg: 'SVG', pdf: 'PDF', canvas: 'Canvas' };

function setMode(mode) {
  currentMode = mode;
  document.querySelectorAll('.mode-btn').forEach(b => b.classList.remove('active'));
  document.querySelector(`[data-mode="${mode}"]`).classList.add('active');
  ['base64','svg','pdf','canvas'].forEach(m => {
    document.getElementById('code-panel-' + m).style.display = m === mode ? '' : 'none';
  });
  modeBadge.textContent = modeLabels[mode];
  generate();
}

function generate() {
  const code = codeInput.value.trim();
  if (!code) {
    previewBody.innerHTML = '<div class="empty-preview">Enter data to generate</div>';
    return;
  }

  if (currentMode === 'pdf') {
    const qs = new URLSearchParams({ code });
    previewBody.innerHTML = `<iframe src="pdf?${qs}"></iframe>`;
    return;
  }

  previewBody.innerHTML = '<div class="loading">Generating...</div>';
  const params = new URLSearchParams();
  params.set('code', code);

  const endpoint = currentMode === 'svg' ? 'draw-svg' : currentMode === 'canvas' ? 'draw-canvas' : 'draw-base64';
  fetch(endpoint, {
    method: 'POST',
    headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
    body: params.toString()
  })
    .then(r => r.json())
    .then(data => {
      if (!data.ok) {
        previewBody.innerHTML = `<div class="empty-preview" style="color:#ef4444">${data.error}</div>`;
        return;
      }
      if (currentMode === 'svg') {
        previewBody.innerHTML = `<div class="svg-container">${data.svg}</div>`;
      } else {
        previewBody.innerHTML = `<img src="${data.base64}" alt="QR Code">`;
      }
    })
    .catch(err => {
      previewBody.innerHTML = `<div class="empty-preview" style="color:#ef4444">${err}</div>`;
    });
}

codeInput.addEventListener('input', () => {
  clearTimeout(debounceTimer);
  debounceTimer = setTimeout(generate, 400);
});

// Initial render
generate();
</script>
</body>
</html>
