# frozen_string_literal: true

module BarcodePao
  YUBIN_PTN_N = ["144", "114", "132", "312", "123",
                 "141", "321", "213", "231", "411"].freeze
  YUBIN_PTN_C = ["324", "342", "234", "432", "243", "423", "441", "111"].freeze
  YUBIN_START = "13"
  YUBIN_STOP  = "31"
  YUBIN_PTN_HI = "414"
  YUBIN_ASC_CHRS = "0123456789-ABCDEFGHIJKLMNOPQRSTUVWXYZ"

  module YubinUtils
    def self.get_pattern(index)
      return "" if index < 0 || index > 36
      if index <= 9
        YUBIN_PTN_N[index]
      elsif index == 10
        YUBIN_PTN_HI
      elsif index <= 20
        YUBIN_PTN_C[0] + YUBIN_PTN_N[index - 11]
      elsif index <= 30
        YUBIN_PTN_C[1] + YUBIN_PTN_N[index - 21]
      else
        YUBIN_PTN_C[2] + YUBIN_PTN_N[index - 31]
      end
    end

    def self.get_check_pattern(chk_d)
      return "" if chk_d < 0 || chk_d > 18
      if chk_d <= 9
        YUBIN_PTN_N[chk_d]
      elsif chk_d == 10
        YUBIN_PTN_HI
      else
        YUBIN_PTN_C[chk_d - 11]
      end
    end

    def self.pattern_to_check_value(pattern)
      10.times { |j| return j if pattern == YUBIN_PTN_N[j] }
      return 10 if pattern == YUBIN_PTN_HI
      8.times { |j| return 11 + j if pattern == YUBIN_PTN_C[j] }
      -1
    end
  end

  class YubinCustomer < BarcodeBase1D
    attr_reader :bars, :check_digit

    def initialize(output_format = FORMAT_PNG)
      super(output_format)
      @bars = nil
      @check_digit = -1
    end

    def encode(code)
      raise "input code is empty" if code.nil? || code.empty?
      upper_code = code.upcase

      bars = []
      chk_str = String.new

      # Start code
      YUBIN_START.each_char { |ch| bars << ch.to_i }

      # Data part (20 character positions)
      code_len = 0
      i = 0
      while i < upper_code.length && code_len < 20
        p_idx = YUBIN_ASC_CHRS.index(upper_code[i])
        unless p_idx
          i += 1
          next
        end

        pattern = YubinUtils.get_pattern(p_idx)

        # Special case: at position 19, if pattern > 3 bars, use CC1 only
        if code_len == 19 && pattern.length > 3
          YUBIN_PTN_C[0].each_char do |ch|
            bars << ch.to_i
            chk_str << ch
          end
          code_len += 1
          break
        end

        pattern.each_char do |ch|
          bars << ch.to_i
          chk_str << ch
        end

        code_len += pattern.length <= 3 ? 1 : 2
        i += 1
      end

      # Pad with CC4 to fill 20 positions
      while code_len < 20
        YUBIN_PTN_C[3].each_char { |ch| bars << ch.to_i }
        chk_str << YUBIN_PTN_C[3]
        code_len += 1
      end

      # Check digit calculation (modulo 19)
      chk_sum = 0
      j = 0
      while j <= chk_str.length - 3
        value = YubinUtils.pattern_to_check_value(chk_str[j, 3])
        chk_sum += value if value >= 0
        j += 3
      end

      chk_d = 19 - (chk_sum % 19)
      chk_d = 0 if chk_d == 19
      @check_digit = chk_d

      # Check digit bars
      chk_pattern = YubinUtils.get_check_pattern(chk_d)
      chk_pattern.each_char { |ch| bars << ch.to_i }

      # Stop code
      YUBIN_STOP.each_char { |ch| bars << ch.to_i }

      @bars = bars
      bars
    end

    def draw(code, height)
      bars = encode(code)
      raise "empty bar pattern" if bars.empty?

      # YubinCustomer uses height as the only dimension parameter
      # width is calculated proportionally
      width = (height * 2.5 * bars.length / 3.0).round
      width = [width, bars.length * 3].max

      if svg_output?
        draw_svg_4state(bars, width, height)
      else
        draw_png_4state(bars, width, height)
      end
    end
  end
end
