# frozen_string_literal: true

module BarcodePao
  UPCE_OE = [0x00, 0x0b, 0x0d, 0x0e, 0x13, 0x19, 0x1c, 0x15, 0x16, 0x1a].freeze
  UPCE_LEFT = [
    [0x27, 0x33, 0x1b, 0x21, 0x1d, 0x39, 0x05, 0x11, 0x09, 0x17].freeze, # G-pattern
    [0x0d, 0x19, 0x13, 0x3d, 0x23, 0x31, 0x2f, 0x3b, 0x37, 0x0b].freeze, # L-pattern
  ].freeze

  def self.expand_to_upca(code)
    return "" unless code.length == 7
    c0 = code[0]; c6 = code[6]
    case
    when c0 == "0" && (c6 == "0" || c6 == "1" || c6 == "2")
      code[0, 3] + c6 + "0000" + code[3, 3]
    when c0 == "0" && c6 == "3"
      code[0, 4] + "00000" + code[4, 2]
    when c0 == "0" && c6 == "4"
      code[0, 5] + "00000" + code[5, 1]
    when c0 == "0" && c6 >= "5" && c6 <= "9"
      code[0, 6] + "0000" + c6
    else
      ""
    end
  end

  def self.calculate_check_digit_upce(src)
    expanded = if src.length == 7
                 expand_to_upca(src)
               else
                 src
               end
    return "" if expanded.empty?

    odds = 0; evens = 0; odd = true
    expanded.length.downto(1) do |i|
      n = expanded[i - 1].to_i
      if odd
        odds += n
      else
        evens += n
      end
      odd = !odd
    end

    s = odds * 3 + evens
    cd = s % 10
    cd = 10 - cd if cd != 0
    cd.to_s
  end

  class UPCE < BarcodeBase1D
    attr_accessor :extended_guard

    def initialize(output_format = FORMAT_PNG)
      super(output_format)
      @extended_guard = true
    end

    def encode(code)
      raise "empty string" if code.nil? || code.empty?

      display_code = code
      if display_code.length == 6
        display_code = "0#{display_code}"
        display_code += BarcodePao.calculate_check_digit_upce(display_code)
      elsif display_code.length == 7
        display_code += BarcodePao.calculate_check_digit_upce(display_code)
      elsif display_code.length != 8
        raise "UPC-E requires 6, 7 or 8 digits"
      end

      display_code.each_char { |c| raise "UPC-E barcode requires numeric digits only" unless c >= "0" && c <= "9" }

      encode_upce(display_code)
    end

    def draw(code, width, height)
      pattern = encode(code)
      display_code = code
      if display_code.length == 6
        display_code = "0#{display_code}"
        display_code += BarcodePao.calculate_check_digit_upce(display_code)
      elsif display_code.length == 7
        display_code += BarcodePao.calculate_check_digit_upce(display_code)
      end

      if svg_output?
        if @show_text && @extended_guard
          draw_svg_upce(pattern, display_code, width, height)
        else
          display = @show_text ? display_code : ""
          draw_svg_bars(pattern, width, height, display)
        end
      else
        draw_png_jan_upc(pattern, display_code, width, height, "UPCE")
      end
    end

    private

    def encode_upce(src)
      raise "EncodeUPCE: code must be 8 digits" unless src.length == 8

      cd = src[7].to_i
      data = src[1, 6]

      pre = cd
      chk_oe = 0x20

      modules = []
      modules.push(1, 0, 1) # Start guard "101"

      6.times do |i|
        d = data[i].to_i
        flg_oe = (UPCE_OE[pre] & (chk_oe >> i)) != 0 ? 1 : 0
        bits = UPCE_LEFT[flg_oe][d]
        mask = 0x40
        7.times do |j|
          modules << ((bits & (mask >> j)) != 0 ? 1 : 0)
        end
      end

      # End guard "010101"
      6.times { |j| modules << (j.even? ? 0 : 1) }

      # Run-length encode
      raise "EncodeUPCE: internal module sequence error" if modules.empty? || modules[0] != 1

      pattern = []
      current = modules[0]
      count = 1
      (1...modules.length).each do |k|
        if modules[k] == current
          count += 1
        else
          pattern << count
          current = modules[k]
          count = 1
        end
      end
      pattern << count
      pattern
    end
  end
end
