# frozen_string_literal: true

module BarcodePao
  module RSSUtils
    # Calculate nCr (binomial coefficient)
    def self.get_combinations(n, r)
      return 0 if r < 0 || r > n
      if n - r > r
        min_denom = r
        max_denom = n - r
      else
        min_denom = n - r
        max_denom = r
      end

      val = 1
      j = 1
      i = n
      while i > max_denom
        val *= i
        if j <= min_denom
          val /= j
          j += 1
        end
        i -= 1
      end
      while j <= min_denom
        val /= j
        j += 1
      end
      val
    end

    # RSS subset width algorithm (ISO 24724)
    def self.get_widths(val, n, elements, max_width, no_narrow)
      widths = Array.new(elements, 0)
      narrow_mask = 0

      (0...elements - 1).each do |bar|
        elm_width = 1
        narrow_mask |= (1 << bar)
        sub_val = 0

        loop do
          sub_val = get_combinations(n - elm_width - 1, elements - bar - 2)

          if no_narrow == 0 && narrow_mask == 0 &&
             n - elm_width - (elements - bar - 1) >= elements - bar - 1
            sub_val -= get_combinations(n - elm_width - (elements - bar),
                                        elements - bar - 2)
          end

          if elements - bar - 1 > 1
            less_val = 0
            mxw_element = n - elm_width - (elements - bar - 2)
            while mxw_element > max_width
              less_val += get_combinations(n - elm_width - mxw_element - 1,
                                           elements - bar - 3)
              mxw_element -= 1
            end
            sub_val -= less_val * (elements - 1 - bar)
          elsif n - elm_width > max_width
            sub_val -= 1
          end

          val -= sub_val
          break if val < 0

          elm_width += 1
          narrow_mask &= ~(1 << bar)
        end

        val += sub_val
        n -= elm_width
        widths[bar] = elm_width
      end

      widths[elements - 1] = n
      widths
    end

    # Calculate GS1 GTIN-14 check digit from 13 digits
    def self.calculate_gtin_check_digit(src13)
      total = 0
      13.times do |i|
        weight = i.odd? ? 1 : 3
        total += src13[i].to_i * weight
      end
      (10 - (total % 10)) % 10
    end

    # Convert binary string to run-length encoded pattern string
    def self.bin2pat(binary)
      return "" if binary.nil? || binary.empty?

      result = String.new
      count = 1
      last_char = binary[0]

      (1...binary.length).each do |i|
        if binary[i] == last_char
          count += 1
        else
          result << (48 + count).chr
          count = 1
          last_char = binary[i]
        end
      end
      result << (48 + count).chr
      result
    end
  end
end
