# frozen_string_literal: true

# SVG/PNG rendering for 4-state barcodes (YubinCustomer).
# Generic 1D SVG rendering is in barcode_base_1d.rb.

module BarcodePao
  class BarcodeBase1D
    # Render 4-state postal barcode as SVG
    def draw_svg_4state(bars, width, height)
      n = bars.length
      raise "empty bar pattern" if n == 0

      pitch = width.to_f / n
      bar_w = [1, (pitch * 0.45).round].max.to_f
      h3 = height / 3.0

      svg_begin(width, height)
      svg_rect(0, 0, width.to_f, height.to_f, @background_color)

      bars.each_with_index do |bt, i|
        bx = (i * pitch + (pitch - bar_w) / 2.0).round.to_f
        case bt
        when 1
          svg_rect(bx, 0, bar_w, height.to_f, @foreground_color)
        when 2
          svg_rect(bx, 0, bar_w, (h3 * 2).round.to_f, @foreground_color)
        when 3
          svg_rect(bx, h3.round.to_f, bar_w, (h3 * 2).round.to_f, @foreground_color)
        when 4
          svg_rect(bx, h3.round.to_f, bar_w, h3.round.to_f, @foreground_color)
        end
      end

      svg_end
    end

    # Render 4-state postal barcode as PNG
    def draw_png_4state(bars, width, height)
      n = bars.length
      raise "empty bar pattern" if n == 0

      img = PNGImage.new(width, height, @background_color)

      pitch = width.to_f / n
      bar_w = [1, (pitch * 0.45).round].max
      h3 = height / 3.0

      bars.each_with_index do |bt, i|
        bx = (i * pitch + (pitch - bar_w) / 2.0).round
        case bt
        when 1
          img.fill_rect(bx, 0, bx + bar_w, height, @foreground_color)
        when 2
          img.fill_rect(bx, 0, bx + bar_w, (h3 * 2).round, @foreground_color)
        when 3
          img.fill_rect(bx, h3.round, bx + bar_w, h3.round + (h3 * 2).round, @foreground_color)
        when 4
          img.fill_rect(bx, h3.round, bx + bar_w, h3.round + h3.round, @foreground_color)
        end
      end

      if BarcodePao.trial_mode?
        img.draw_sample_overlay(0, 0, width, height)
      end

      @image_buffer = img.to_png
    end
  end
end
