# frozen_string_literal: true

require "chunky_png"
require "ttfunk"

module BarcodePao
  # Simple PNG image wrapper using ChunkyPNG.
  class PNGImage
    attr_reader :width, :height

    def initialize(width, height, bg_color = [255, 255, 255, 255])
      @width = width
      @height = height
      @canvas = ChunkyPNG::Image.new(width, height, to_chunky_color(bg_color))
    end

    def fill_rect(x1, y1, x2, y2, color)
      c = to_chunky_color(color)
      x1 = [x1, 0].max
      y1 = [y1, 0].max
      x2 = [x2, @width].min
      y2 = [y2, @height].min
      (y1...y2).each do |py|
        (x1...x2).each do |px|
          @canvas[px, py] = c
        end
      end
    end

    def set_pixel(x, y, color)
      return unless x >= 0 && x < @width && y >= 0 && y < @height
      @canvas[x, y] = to_chunky_color(color)
    end

    def draw_sample_overlay(x, y, width, height)
      font_size = (height * 0.12).to_i
      font_size = 8 if font_size < 8
      font_size = 40 if font_size > 40
      margin = (height * 0.01).to_i
      margin = 2 if margin < 2
      text = BarcodePao.trial_text
      red = [255, 0, 0, 255]
      draw_text_at(text, x + margin, y + margin, font_size, red)
    end

    def draw_text_even(text, x, y, total_width, font_size, fg_color, h_scale)
      return if text.nil? || text.empty?
      num_chars = text.length

      margin_ratio = if total_width < 100
                       0.03
                     elsif total_width > 400
                       0.07
                     else
                       0.05
                     end

      horizontal_margin = total_width * margin_ratio * h_scale
      text_width = total_width - 2 * horizontal_margin

      char_spacing = if num_chars > 1
                       text_width / (num_chars - 1).to_f
                     else
                       text_width
                     end

      min_spacing = font_size / 2.0
      char_spacing = min_spacing if char_spacing < min_spacing

      total_text_width = num_chars > 1 ? char_spacing * (num_chars - 1) : 0.0
      start_x = x + (total_width - total_text_width) / 2.0

      num_chars.times do |i|
        char_x = (start_x + i * char_spacing).round
        draw_char_centered(text[i], char_x, y.round, font_size, fg_color)
      end
    end

    def draw_text_centered(text, cx, y, font_size, fg_color)
      return if text.nil? || text.empty?
      glyphs = text_glyphs(text, font_size)
      total_w = glyphs.sum { |g| g[:advance] }
      start_x = cx - total_w / 2
      draw_glyphs(glyphs, start_x, y, font_size, fg_color)
    end

    def draw_char_centered(ch, cx, y, font_size, fg_color)
      glyphs = text_glyphs(ch.to_s, font_size)
      return if glyphs.empty?
      adv = glyphs[0][:advance]
      start_x = cx - adv / 2
      draw_glyphs(glyphs, start_x, y, font_size, fg_color)
    end

    def to_png
      @canvas.to_blob
    end

    private

    def to_chunky_color(c)
      ChunkyPNG::Color.rgba(c[0], c[1], c[2], c[3] || 255)
    end

    # ---- Font rendering with ttfunk ----

    def font_file
      @font_file ||= TTFunk::File.open(font_path)
    end

    def font_path
      File.join(File.dirname(__FILE__), "..", "..", "Roboto-Regular.ttf")
    end

    def text_glyphs(text, font_size)
      scale = font_size.to_f / font_file.header.units_per_em
      text.chars.map do |ch|
        glyph_id = font_file.cmap.unicode.first[ch.ord]
        glyph_id ||= 0
        advance = 0
        if glyph_id > 0 && glyph_id < font_file.horizontal_metrics.metrics.length
          advance = (font_file.horizontal_metrics.for(glyph_id).advance_width * scale).round
        else
          advance = (font_size * 0.6).round
        end
        { char: ch, glyph_id: glyph_id, advance: advance }
      end
    end

    def draw_glyphs(glyphs, start_x, start_y, font_size, fg_color)
      scale = font_size.to_f / font_file.header.units_per_em
      ascent = (font_file.os2.ascent * scale).round

      cursor_x = start_x
      glyphs.each do |g|
        draw_glyph_pixels(g[:glyph_id], cursor_x, start_y + ascent, font_size, scale, fg_color)
        cursor_x += g[:advance]
      end
    end

    def draw_glyph_pixels(glyph_id, origin_x, baseline_y, font_size, scale, fg_color)
      return if glyph_id == 0

      glyph = font_file.glyph_outlines.for(glyph_id)
      return unless glyph

      # Simple rasterization: sample outline points and fill
      # For production quality we'd use a proper rasterizer, but for barcode text
      # this bitmap approach using scan-line fill is sufficient.
      render_glyph_scanline(glyph, origin_x, baseline_y, scale, fg_color)
    end

    def render_glyph_scanline(glyph, origin_x, baseline_y, scale, fg_color)
      return unless glyph.respond_to?(:contours) && glyph.contours

      # Collect all contour points
      contours = extract_contour_points(glyph, origin_x, baseline_y, scale)
      return if contours.empty?

      # Find bounding box
      all_points = contours.flatten(1)
      return if all_points.empty?

      min_y = all_points.map { |p| p[1] }.min.floor
      max_y = all_points.map { |p| p[1] }.max.ceil

      color = to_chunky_color(fg_color)

      # Scan-line fill for each row
      (min_y..max_y).each do |y|
        next if y < 0 || y >= @height
        intersections = []

        contours.each do |pts|
          next if pts.length < 2
          pts.length.times do |i|
            p1 = pts[i]
            p2 = pts[(i + 1) % pts.length]
            y1, y2 = p1[1], p2[1]
            next if y1 == y2
            if (y1 <= y && y2 > y) || (y2 <= y && y1 > y)
              t = (y - y1).to_f / (y2 - y1)
              ix = p1[0] + t * (p2[0] - p1[0])
              intersections << ix
            end
          end
        end

        intersections.sort!
        i = 0
        while i + 1 < intersections.length
          x1 = intersections[i].ceil
          x2 = intersections[i + 1].floor
          (x1..x2).each do |x|
            @canvas[x, y] = color if x >= 0 && x < @width
          end
          i += 2
        end
      end
    end

    def extract_contour_points(glyph, origin_x, baseline_y, scale)
      contours = []
      return contours unless glyph.respond_to?(:contours) && glyph.contours

      glyph.contours.each do |contour|
        points = []
        # Decompose the contour (on-curve + quadratic off-curve)
        raw = []
        contour.each do |point|
          px = origin_x + (point.x * scale).round
          py = baseline_y - (point.y * scale).round
          raw << { x: px, y: py, on_curve: point.on_curve? }
        end

        next if raw.empty?

        # Interpolate quadratic Bezier curves
        i = 0
        while i < raw.length
          p = raw[i]
          if p[:on_curve]
            points << [p[:x], p[:y]]
            i += 1
          else
            # Off-curve: find implied on-curve points
            prev_p = points.last || [raw[-1][:x], raw[-1][:y]]
            next_i = (i + 1) % raw.length
            next_p = raw[next_i]

            if next_p[:on_curve]
              # Quadratic bezier: prev -> p -> next
              5.times do |t_i|
                t = (t_i + 1) / 5.0
                bx = (1 - t)**2 * prev_p[0] + 2 * (1 - t) * t * p[:x] + t**2 * next_p[:x]
                by = (1 - t)**2 * prev_p[1] + 2 * (1 - t) * t * p[:y] + t**2 * next_p[:y]
                points << [bx.round, by.round]
              end
              i += 1
            else
              # Two off-curve: implied on-curve at midpoint
              mid_x = (p[:x] + next_p[:x]) / 2.0
              mid_y = (p[:y] + next_p[:y]) / 2.0
              5.times do |t_i|
                t = (t_i + 1) / 5.0
                bx = (1 - t)**2 * prev_p[0] + 2 * (1 - t) * t * p[:x] + t**2 * mid_x
                by = (1 - t)**2 * prev_p[1] + 2 * (1 - t) * t * p[:y] + t**2 * mid_y
                points << [bx.round, by.round]
              end
              i += 1
            end
          end
        end

        contours << points unless points.empty?
      end

      contours
    end

    def draw_text_at(text, x, y, font_size, fg_color)
      glyphs = text_glyphs(text, font_size)
      draw_glyphs(glyphs, x, y, font_size, fg_color)
    end
  end

  # Mixin for BarcodeBase1D PNG rendering
  module PNGRenderer1D
    def draw_png_1d(pattern, code, width, height)
      display = @show_text ? code : ""
      render_bars_to_png(pattern, width, height, display)
    end

    def render_bars_to_png(pattern, width, height, text)
      total_units = pattern.sum
      raise "pattern has zero total units" if total_units <= 0

      font_size = [8, (height * 0.15 * @text_font_scale).to_i].max

      text_h = 0
      if text && !text.empty?
        text_h = (font_size * 1.4).to_i
      end
      bar_h = height - text_h
      if bar_h <= 0
        bar_h = height
        text = ""
      end

      img = PNGImage.new(width, height, @background_color)

      unit_w = width.to_f / total_units
      accum = 0.0
      is_bar = true
      pattern.each do |units|
        x1 = (accum * unit_w).round
        accum += units
        x2 = (accum * unit_w).round
        if is_bar && x2 > x1
          img.fill_rect(x1, 0, x2, bar_h, @foreground_color)
        end
        is_bar = !is_bar
      end

      # Draw text
      if text && !text.empty?
        text_y = bar_h + 2
        if @text_even_spacing
          img.draw_text_even(text, 0, text_y, width, font_size, @foreground_color, @text_horizontal_spacing_scale)
        else
          img.draw_text_centered(text, width / 2, text_y, font_size, @foreground_color)
        end
      end

      # Trial mode watermark
      if BarcodePao.trial_mode?
        img.draw_sample_overlay(0, 0, width, height)
      end

      @image_buffer = img.to_png
    end
  end

  # Include PNG rendering in BarcodeBase1D
  class BarcodeBase1D
    include PNGRenderer1D
  end
end
