# frozen_string_literal: true

module BarcodePao
  # QR Code error correction levels
  QR_ECC_L = 0  # Low (~7%)
  QR_ECC_M = 1  # Medium (~15%)
  QR_ECC_Q = 2  # Quartile (~25%)
  QR_ECC_H = 3  # High (~30%)

  # QR Code encode modes
  QR_MODE_NUMERIC      = "N"
  QR_MODE_ALPHANUMERIC = "A"
  QR_MODE_BINARY       = "B"
  QR_MODE_KANJI        = "K"

  class QRCode < BarcodeBase2D
    attr_accessor :error_correction_level, :version, :encode_mode

    def initialize(output_format = FORMAT_PNG)
      super(output_format)
      @error_correction_level = QR_ECC_M
      @version = 0  # 0 = auto
      @encode_mode = QR_MODE_BINARY
    end

    def draw(code, size)
      data = code.bytes
      raise "empty input" if data.empty?

      jam = create_qr_jam
      jam.send(:make_arr, data)
      pattern = jam.instance_variable_get(:@patt)
      raise "QR encoding produced empty pattern" if pattern.nil? || pattern.empty?

      if svg_output?
        draw_svg_2d(pattern, size, size)
      else
        draw_png_2d(pattern, size, size)
      end
    end

    def get_pattern(code)
      data = code.bytes
      raise "empty input" if data.empty?

      jam = create_qr_jam
      jam.send(:make_arr, data)
      jam.instance_variable_get(:@patt)
    end

    private

    def create_qr_jam
      jam = QRJam.new
      jam.send(:set_encode_mode, @encode_mode)
      jam.send(:set_version, @version)
      jam.send(:set_ecc_rate, @error_correction_level)
      jam
    end
  end

  # ---------------------------------------------------------------------------
  # QRJam - internal QR code generation engine
  # ---------------------------------------------------------------------------
  class QRJam
    # Internal text types
    QR_TEXT_AUTO           = 0
    QR_TEXT_NUMERIC        = 1
    QR_TEXT_ALPHA_NUMERIC  = 2
    QR_TEXT_KANJI          = 3
    QR_TEXT_BINARY         = 4

    # Internal ECC rates
    QR_ECC_LOW7     = 0
    QR_ECC_MEDIUM15 = 1
    QR_ECC_QUAL25   = 2
    QR_ECC_HIGHQ30  = 3

    # gDataWords[4][40]
    G_DATA_WORDS = [
      [19, 34, 55, 80, 108, 68, 78, 97, 116, 68,
       81, 92, 107, 115, 87, 98, 107, 120, 113, 107,
       116, 111, 121, 117, 106, 114, 122, 117, 116, 115,
       115, 115, 115, 115, 121, 121, 122, 122, 117, 118],
      [16, 28, 44, 32, 43, 27, 31, 38, 36, 43,
       50, 36, 37, 40, 41, 45, 46, 43, 44, 41,
       42, 46, 47, 45, 47, 46, 45, 45, 45, 47,
       46, 46, 46, 46, 47, 47, 46, 46, 47, 47],
      [13, 22, 17, 24, 15, 19, 14, 18, 16, 19,
       22, 20, 20, 16, 24, 19, 22, 22, 21, 24,
       22, 24, 24, 24, 24, 22, 23, 24, 23, 24,
       24, 24, 24, 24, 24, 24, 24, 24, 24, 24],
      [9, 16, 13, 9, 11, 15, 13, 14, 12, 15,
       12, 14, 11, 12, 12, 15, 14, 14, 13, 15,
       16, 13, 15, 16, 15, 16, 15, 15, 15, 15,
       15, 15, 15, 16, 15, 15, 15, 15, 15, 15]
    ].freeze

    # gExtraDataWords[4][40]
    G_EXTRA_DATA_WORDS = [
      [0, 0, 0, 0, 0, 0, 0, 0, 0, 2,
       0, 2, 0, 1, 1, 1, 5, 1, 4, 5,
       4, 7, 5, 4, 4, 2, 4, 10, 7, 10,
       3, 0, 1, 6, 7, 14, 4, 18, 4, 6],
      [0, 0, 0, 0, 0, 0, 0, 2, 2, 1,
       4, 2, 1, 5, 5, 3, 1, 4, 11, 13,
       0, 0, 14, 14, 13, 4, 3, 23, 7, 10,
       29, 23, 21, 23, 26, 34, 14, 32, 7, 31],
      [0, 0, 0, 0, 2, 0, 4, 2, 4, 2,
       4, 6, 4, 5, 7, 2, 15, 1, 4, 5,
       6, 16, 14, 16, 22, 6, 26, 31, 37, 25,
       1, 35, 19, 7, 14, 10, 10, 14, 22, 34],
      [0, 0, 0, 0, 2, 0, 1, 2, 4, 2,
       8, 4, 4, 5, 7, 13, 17, 19, 16, 10,
       6, 0, 14, 2, 13, 4, 28, 31, 26, 25,
       28, 35, 46, 1, 41, 64, 46, 32, 67, 61]
    ].freeze

    # gECCWords[4][40]
    G_ECC_WORDS = [
      [7, 10, 15, 20, 26, 18, 20, 24, 30, 18,
       20, 24, 26, 30, 22, 24, 28, 30, 28, 28,
       28, 28, 30, 30, 26, 28, 30, 30, 30, 30,
       30, 30, 30, 30, 30, 30, 30, 30, 30, 30],
      [10, 16, 26, 18, 24, 16, 18, 22, 22, 26,
       30, 22, 22, 24, 24, 28, 28, 26, 26, 26,
       26, 28, 28, 28, 28, 28, 28, 28, 28, 28,
       28, 28, 28, 28, 28, 28, 28, 28, 28, 28],
      [13, 22, 18, 26, 18, 24, 18, 22, 20, 24,
       28, 26, 24, 20, 30, 24, 28, 28, 26, 30,
       28, 30, 30, 30, 30, 28, 30, 30, 30, 30,
       30, 30, 30, 30, 30, 30, 30, 30, 30, 30],
      [17, 28, 22, 16, 22, 28, 26, 26, 24, 28,
       24, 28, 22, 24, 24, 30, 28, 28, 26, 28,
       30, 24, 30, 30, 30, 30, 30, 30, 30, 30,
       30, 30, 30, 30, 30, 30, 30, 30, 30, 30]
    ].freeze

    # gRSBlockCount[4][40]
    G_RS_BLOCK_COUNT = [
      [1, 1, 1, 1, 1, 2, 2, 2, 2, 4,
       4, 4, 4, 4, 6, 6, 6, 6, 7, 8,
       8, 9, 9, 10, 12, 12, 12, 13, 14, 15,
       16, 17, 18, 19, 19, 20, 21, 22, 24, 25],
      [1, 1, 1, 2, 2, 4, 4, 4, 5, 5,
       5, 8, 9, 9, 10, 10, 11, 13, 14, 16,
       17, 17, 18, 20, 21, 23, 25, 26, 28, 29,
       31, 33, 35, 37, 38, 40, 43, 45, 47, 49],
      [1, 1, 2, 2, 4, 4, 6, 6, 8, 8,
       8, 10, 12, 16, 12, 17, 16, 18, 21, 20,
       23, 23, 25, 27, 29, 34, 34, 35, 38, 40,
       43, 45, 48, 51, 53, 56, 59, 62, 65, 68],
      [1, 1, 2, 4, 4, 4, 5, 6, 8, 8,
       11, 11, 16, 16, 18, 16, 19, 21, 25, 25,
       25, 34, 30, 32, 35, 37, 40, 42, 45, 48,
       51, 54, 57, 60, 63, 66, 70, 74, 77, 81]
    ].freeze

    # Alignment pattern positions by version (1..40)
    QR_ALIGNMENT_PATTERNS = {
      1 => [6], 2 => [6, 18], 3 => [6, 22], 4 => [6, 26], 5 => [6, 30], 6 => [6, 34],
      7 => [6, 22, 38], 8 => [6, 24, 42], 9 => [6, 26, 46], 10 => [6, 28, 50],
      11 => [6, 30, 54], 12 => [6, 32, 58], 13 => [6, 34, 62],
      14 => [6, 26, 46, 66], 15 => [6, 26, 48, 70], 16 => [6, 26, 50, 74],
      17 => [6, 30, 54, 78], 18 => [6, 30, 56, 82], 19 => [6, 30, 58, 86],
      20 => [6, 34, 62, 90],
      21 => [6, 28, 50, 72, 94], 22 => [6, 26, 50, 74, 98],
      23 => [6, 30, 54, 78, 102], 24 => [6, 28, 54, 80, 106],
      25 => [6, 32, 58, 84, 110], 26 => [6, 30, 58, 86, 114],
      27 => [6, 34, 62, 90, 118],
      28 => [6, 26, 50, 74, 98, 122], 29 => [6, 30, 54, 78, 102, 126],
      30 => [6, 26, 52, 78, 104, 130], 31 => [6, 30, 56, 82, 108, 134],
      32 => [6, 34, 60, 86, 112, 138], 33 => [6, 30, 58, 86, 114, 142],
      34 => [6, 34, 62, 90, 118, 146],
      35 => [6, 30, 54, 78, 102, 126, 150], 36 => [6, 24, 50, 76, 102, 128, 154],
      37 => [6, 28, 54, 80, 106, 132, 158], 38 => [6, 32, 58, 84, 110, 136, 162],
      39 => [6, 26, 54, 82, 110, 138, 166], 40 => [6, 30, 58, 86, 114, 142, 170]
    }.freeze

    # Version information bits (versions 7-40)
    QR_VERSION_INFO = [
      31892, 34236, 39577, 42195, 48118, 51042, 55367, 58893,
      63784, 68472, 70749, 76311, 79154, 84390, 87683, 92361,
      96236, 102084, 102881, 110507, 110734, 117786, 119615, 126325,
      127568, 133589, 136944, 141498, 145311, 150283, 152622, 158308,
      161089, 167017
    ].freeze

    # GF(256) exp table
    GF_EXP = [
      1, 2, 4, 8, 16, 32, 64, 128, 29, 58, 116, 232, 205, 135, 19, 38, 76, 152, 45, 90,
      180, 117, 234, 201, 143, 3, 6, 12, 24, 48, 96, 192, 157, 39, 78, 156, 37, 74, 148, 53,
      106, 212, 181, 119, 238, 193, 159, 35, 70, 140, 5, 10, 20, 40, 80, 160, 93, 186, 105, 210,
      185, 111, 222, 161, 95, 190, 97, 194, 153, 47, 94, 188, 101, 202, 137, 15, 30, 60, 120, 240,
      253, 231, 211, 187, 107, 214, 177, 127, 254, 225, 223, 163, 91, 182, 113, 226, 217, 175, 67, 134,
      17, 34, 68, 136, 13, 26, 52, 104, 208, 189, 103, 206, 129, 31, 62, 124, 248, 237, 199, 147,
      59, 118, 236, 197, 151, 51, 102, 204, 133, 23, 46, 92, 184, 109, 218, 169, 79, 158, 33, 66,
      132, 21, 42, 84, 168, 77, 154, 41, 82, 164, 85, 170, 73, 146, 57, 114, 228, 213, 183, 115,
      230, 209, 191, 99, 198, 145, 63, 126, 252, 229, 215, 179, 123, 246, 241, 255, 227, 219, 171, 75,
      150, 49, 98, 196, 149, 55, 110, 220, 165, 87, 174, 65, 130, 25, 50, 100, 200, 141, 7, 14,
      28, 56, 112, 224, 221, 167, 83, 166, 81, 162, 89, 178, 121, 242, 249, 239, 195, 155, 43, 86,
      172, 69, 138, 9, 18, 36, 72, 144, 61, 122, 244, 245, 247, 243, 251, 235, 203, 139, 11, 22,
      44, 88, 176, 125, 250, 233, 207, 131, 27, 54, 108, 216, 173, 71, 142, 1
    ].freeze

    # GF(256) log table
    GF_LOG = [
      0, 0, 1, 25, 2, 50, 26, 198, 3, 223, 51, 238, 27, 104, 199, 75, 4, 100, 224, 14,
      52, 141, 239, 129, 28, 193, 105, 248, 200, 8, 76, 113, 5, 138, 101, 47, 225, 36, 15, 33,
      53, 147, 142, 218, 240, 18, 130, 69, 29, 181, 194, 125, 106, 39, 249, 185, 201, 154, 9, 120,
      77, 228, 114, 166, 6, 191, 139, 98, 102, 221, 48, 253, 226, 152, 37, 179, 16, 145, 34, 136,
      54, 208, 148, 206, 143, 150, 219, 189, 241, 210, 19, 92, 131, 56, 70, 64, 30, 66, 182, 163,
      195, 72, 126, 110, 107, 58, 40, 84, 250, 133, 186, 61, 202, 94, 155, 159, 10, 21, 121, 43,
      78, 212, 229, 172, 115, 243, 167, 87, 7, 112, 192, 247, 140, 128, 99, 13, 103, 74, 222, 237,
      49, 197, 254, 24, 227, 165, 153, 119, 38, 184, 180, 124, 17, 68, 146, 217, 35, 32, 137, 46,
      55, 63, 209, 91, 149, 188, 207, 205, 144, 135, 151, 178, 220, 252, 190, 97, 242, 86, 211, 171,
      20, 42, 93, 158, 132, 60, 57, 83, 71, 109, 65, 162, 31, 45, 67, 216, 183, 123, 164, 118,
      196, 23, 73, 236, 127, 12, 111, 246, 108, 161, 59, 82, 41, 157, 85, 170, 251, 96, 134, 177,
      187, 204, 62, 90, 203, 89, 95, 176, 156, 169, 160, 81, 11, 245, 22, 235, 122, 117, 44, 215,
      79, 174, 213, 233, 230, 231, 173, 232, 116, 214, 244, 234, 168, 80, 88, 175
    ].freeze

    # Generator polynomials keyed by ECC word count
    QR_GEN_POLYS = {
      7  => [127, 122, 154, 164, 11, 68, 117],
      10 => [216, 194, 159, 111, 199, 94, 95, 113, 157, 193],
      13 => [137, 73, 227, 17, 177, 17, 52, 13, 46, 43, 83, 132, 120],
      15 => [29, 196, 111, 163, 112, 74, 10, 105, 105, 139, 132, 151, 32, 134, 26],
      16 => [59, 13, 104, 189, 68, 209, 30, 8, 163, 65, 41, 229, 98, 50, 36, 59],
      17 => [119, 66, 83, 120, 119, 22, 197, 83, 249, 41, 143, 134, 85, 53, 125, 99, 79],
      18 => [239, 251, 183, 113, 149, 175, 199, 215, 240, 220, 73, 82, 173, 75, 32, 67, 217, 146],
      20 => [152, 185, 240, 5, 111, 99, 6, 220, 112, 150, 69, 36, 187, 22, 228, 198, 121, 121, 165, 174],
      22 => [89, 179, 131, 176, 182, 244, 19, 189, 69, 40, 28, 137, 29, 123, 67, 253, 86, 218, 230, 26, 145, 245],
      24 => [122, 118, 169, 70, 178, 237, 216, 102, 115, 150, 229, 73, 130, 72, 61, 43, 206, 1, 237, 247, 127, 217, 144, 117],
      26 => [246, 51, 183, 4, 136, 98, 199, 152, 77, 56, 206, 24, 145, 40, 209, 117, 233, 42, 135, 68, 70, 144, 146, 77, 43, 94],
      28 => [252, 9, 28, 13, 18, 251, 208, 150, 103, 174, 100, 41, 167, 12, 247, 56, 117, 119, 233, 127, 181, 100, 121, 147, 176, 74, 58, 197],
      30 => [212, 246, 77, 73, 195, 192, 75, 98, 5, 70, 103, 177, 22, 217, 138, 51, 181, 246, 72, 25, 18, 46, 228, 74, 216, 195, 11, 106, 130, 150]
    }.freeze

    # Capacity tables
    QR_ARRAY_CAP = [
      0, 26, 44, 70, 100, 134, 172, 196, 242, 292,
      346, 404, 466, 532, 581, 655, 733, 815, 901, 991,
      1085, 1156, 1258, 1364, 1474, 1588, 1706, 1828, 1921, 2051,
      2185, 2323, 2465, 2611, 2761, 2876, 3034, 3196, 3362, 3532,
      3706
    ].freeze

    QR_ARRAY2_EXTRA = [
      0, 0, 7, 7, 7, 7, 7, 0, 0, 0,
      0, 0, 0, 0, 3, 3, 3, 3, 3, 3,
      3, 4, 4, 4, 4, 4, 4, 4, 3, 3,
      3, 3, 3, 3, 3, 0, 0, 0, 0, 0,
      0
    ].freeze

    # qrArray3Cap[4][41] - bit capacity by ECC level and version
    QR_ARRAY3_CAP = [
      [0, 152, 272, 440, 640, 864, 1088, 1248, 1552, 1856,
       2192, 2592, 2960, 3424, 3688, 4184, 4712, 5176, 5768, 6360,
       6888, 7456, 8048, 8752, 9392, 10208, 10960, 11744, 12248, 13048,
       13880, 14744, 15640, 16568, 17528, 18448, 19472, 20528, 21616, 22496,
       23648],
      [0, 128, 224, 352, 512, 688, 864, 992, 1232, 1456,
       1728, 2032, 2320, 2672, 2920, 3320, 3624, 4056, 4504, 5016,
       5352, 5712, 6256, 6880, 7312, 8000, 8496, 9024, 9544, 10136,
       10984, 11640, 12328, 13048, 13800, 14496, 15312, 15936, 16816, 17728,
       18672],
      [0, 104, 176, 272, 384, 496, 608, 704, 880, 1056,
       1232, 1440, 1648, 1952, 2088, 2360, 2600, 2936, 3176, 3560,
       3880, 4096, 4544, 4912, 5312, 5744, 6032, 6464, 6968, 7288,
       7880, 8264, 8920, 9368, 9848, 10288, 10832, 11408, 12016, 12656,
       13328],
      [0, 72, 128, 208, 288, 368, 480, 528, 688, 800,
       976, 1120, 1264, 1440, 1576, 1784, 2024, 2264, 2504, 2728,
       3080, 3248, 3536, 3712, 4112, 4304, 4768, 5024, 5288, 5608,
       5960, 6344, 6760, 7208, 7688, 7888, 8432, 8768, 9136, 9776,
       10208]
    ].freeze

    # Format information strings (32 patterns)
    QR_FORMAT_INFO = [
      "111011111000100", "111001011110011", "111110110101010", "111100010011101",
      "110011000101111", "110001100011000", "110110001000001", "110100101110110",
      "101010000010010", "101000100100101", "101111001111100", "101101101001011",
      "100010111111001", "100000011001110", "100111110010111", "100101010100000",
      "011010101011111", "011000001101000", "011111100110001", "011101000000110",
      "010010010110100", "010000110000011", "010111011011010", "010101111101101",
      "001011010001001", "001001110111110", "001110011100111", "001100111010000",
      "000011101100010", "000001001010101", "000110100001100", "000100000111011"
    ].freeze

    def initialize
      @l_qr_set_ecc_rate  = QR_ECC_MEDIUM15
      @l_qr_set_text_type = QR_TEXT_AUTO
      @l_qr_quit_zone     = 4
      @encode_mode        = "Z"

      @l_format_first_x  = [0, 1, 2, 3, 4, 5, 7, 8, 8, 8, 8, 8, 8, 8, 8]
      @l_format_first_y  = [8, 8, 8, 8, 8, 8, 8, 8, 7, 5, 4, 3, 2, 1, 0]
      @l_format_second_x = Array.new(15, 0)
      @l_format_second_y = Array.new(15, 0)

      @l_deploy_table_x = []
      @l_deploy_table_y = []
      @l_deploy_order_x = []
      @l_deploy_order_y = []
      @l_rs_block        = []

      @l_qr_set_version               = 0
      @l_qr_inquire_version           = 0
      @l_qr_inquire_text_type         = 0
      @l_qr_inquire_numeric_bits      = 0
      @l_qr_inquire_alpha_numeric_bits = 0
      @l_qr_inquire_kanji_bits        = 0
      @l_qr_inquire_binary_bits       = 0
      @l_qr_inquire_module_size       = 0

      @patt = nil
    end

    private

    def set_encode_mode(mode)
      @encode_mode = mode
      case mode
      when "N"
        @l_qr_set_text_type = QR_TEXT_NUMERIC
      when "A"
        @l_qr_set_text_type = QR_TEXT_ALPHA_NUMERIC
      else
        @l_qr_set_text_type = QR_TEXT_BINARY
      end
    end

    def set_version(version)
      @l_qr_set_version = version if version >= 0 && version <= 40
    end

    def set_ecc_rate(rate)
      @l_qr_set_ecc_rate = rate
    end

    # ---------------------------------------------------------------------------
    # Internal methods
    # ---------------------------------------------------------------------------

    def qr_get_1d_address(px, py, symbol_size)
      px + py * (symbol_size + 1)
    end

    def qr_code_get_format_x(p_version)
      arr = Array.new(15, 0)
      7.times { |i| arr[i] = 8 }
      8.times { |i| arr[i + 7] = 4 * p_version + 9 + i }
      arr
    end

    def qr_code_get_format_y(p_version)
      arr = Array.new(15, 0)
      8.times { |i| arr[i + 7] = 8 }
      7.times { |i| arr[i] = 4 * p_version + 16 - i }
      arr
    end

    def qr_code_inquire_bits(data)
      length = data.length

      if @encode_mode == "N"
        rem = length % 3
        if rem == 0
          @l_qr_inquire_numeric_bits = ((length / 3.0).floor * 10.0).round
        elsif rem == 1
          @l_qr_inquire_numeric_bits = ((length / 3.0).floor * 10.0 + 4.0).round
        elsif rem == 2
          @l_qr_inquire_numeric_bits = ((length / 3.0).floor * 10.0 + 7.0).round
        end
        @l_qr_inquire_alpha_numeric_bits = 0
        @l_qr_inquire_kanji_bits = 0
      elsif @encode_mode == "A"
        rem = length % 2
        if rem == 0
          @l_qr_inquire_alpha_numeric_bits = ((length / 2.0).floor * 11.0).round
        elsif rem == 1
          @l_qr_inquire_alpha_numeric_bits = ((length / 2.0).floor * 11.0 + 6.0).round
        end
        @l_qr_inquire_numeric_bits = 0
        @l_qr_inquire_kanji_bits = 0
      else
        @l_qr_inquire_numeric_bits = 0
        @l_qr_inquire_alpha_numeric_bits = 0
        @l_qr_inquire_kanji_bits = 0
      end
      @l_qr_inquire_binary_bits = length * 8
    end

    def qr_code_set_inquire_text_type(_data)
      if @encode_mode == "N"
        QR_TEXT_NUMERIC
      elsif @encode_mode == "A"
        QR_TEXT_ALPHA_NUMERIC
      else
        QR_TEXT_BINARY
      end
    end

    def qr_code_get_symbol_data(p_version)
      num = p_version * 4 + 17
      array3 = Array.new(num * (num + 1), 0)
      arr = [3, num - 4]

      ap = QR_ALIGNMENT_PATTERNS[p_version] || [0]

      ga = method(:qr_get_1d_address)

      arr.length.times do |jj|
        arr.length.times do |kk|
          num8 = arr[jj]
          num9 = arr[kk]
          next if jj == arr.length - 1 && kk == arr.length - 1

          # Center and immediate neighbors
          (-1..1).each do |dx|
            (-1..1).each do |dy|
              array3[ga.call(num8 + dx, num9 + dy, num)] = 3
            end
          end

          # Outer ring
          (-3..3).each do |i|
            array3[ga.call(num8 + i, num9 - 3, num)] = 3
            array3[ga.call(num8 + i, num9 + 3, num)] = 3
          end
          (-2..2).each do |i|
            array3[ga.call(num8 - 3, num9 + i, num)] = 3
            array3[ga.call(num8 + 3, num9 + i, num)] = 3
          end

          # Inner ring
          (-2..2).each do |i|
            array3[ga.call(num8 + i, num9 - 2, num)] = 2
            array3[ga.call(num8 + i, num9 + 2, num)] = 2
          end
          (-1..1).each do |i|
            array3[ga.call(num8 - 2, num9 + i, num)] = 2
            array3[ga.call(num8 + 2, num9 + i, num)] = 2
          end

          # Border markings for specific corners
          if jj == 0 && kk == 0
            (-3..4).each do |i|
              array3[ga.call(num8 + 4, num9 + i, num)] = 8
            end
            (-3..4).each do |i|
              array3[ga.call(num8 + i, num9 + 4, num)] = 8
            end
          end

          if jj == 0 && kk == 1
            (-4..3).each do |i|
              array3[ga.call(num8 + 4, num9 + i, num)] = 8
            end
            (-4..3).each do |i|
              array3[ga.call(num8 - i, num9 - 4, num)] = 8
            end
          end

          if jj == 1 && kk == 0
            (-3..4).each do |i|
              array3[ga.call(num8 - 4, num9 + i, num)] = 8
            end
            (-4..3).each do |i|
              array3[ga.call(num8 + i, num9 + 4, num)] = 8
            end
          end
        end
      end

      # Alignment patterns
      num16 = ap.length - 1
      (0..num16).each do |l_idx|
        ap.length.times do |m_idx|
          num18 = ap[l_idx]
          num19 = ap[m_idx]
          next if (l_idx == 0 && m_idx == 0) ||
                  (l_idx == 0 && m_idx == ap.length - 1) ||
                  (m_idx == 0 && l_idx == ap.length - 1)

          array3[ga.call(num18, num19, num)] = 5
          # Outer ring
          (-2..2).each do |i|
            array3[ga.call(num18 + i, num19 - 2, num)] = 5
            array3[ga.call(num18 + i, num19 + 2, num)] = 5
          end
          (-1..1).each do |i|
            array3[ga.call(num18 - 2, num19 + i, num)] = 5
            array3[ga.call(num18 + 2, num19 + i, num)] = 5
          end
          # Inner ring
          (-1..1).each do |dx|
            (-1..1).each do |dy|
              next if dx == 0 && dy == 0
              array3[ga.call(num18 + dx, num19 + dy, num)] = 4
            end
          end
          array3[ga.call(num18, num19, num)] = 5 # center
        end
      end

      # Timing patterns
      num20 = num - 9
      n = 8
      while n <= num20
        array3[ga.call(n, 6, num)] |= 0x11
        array3[ga.call(6, n, num)] |= 0x11
        if n != num20
          array3[ga.call(n + 1, 6, num)] |= 0x10
          array3[ga.call(6, n + 1, num)] |= 0x10
        end
        n += 2
      end

      array3[ga.call(8, num - 8, num)] = 129

      # Version information
      if p_version >= 7
        num21 = QR_VERSION_INFO[p_version - 7]
        6.times do |num22|
          3.times do |num23|
            num24 = num21 & 1
            if num24 == 1
              array3[ga.call(num - 11 + num23, num22, num)] |= 0x21
              array3[ga.call(num22, num - 11 + num23, num)] |= 0x21
            else
              array3[ga.call(num - 11 + num23, num22, num)] |= 0x20
              array3[ga.call(num22, num - 11 + num23, num)] |= 0x20
            end
            num21 >>= 1
          end
        end
      end

      fmt_x = qr_code_get_format_x(@l_qr_inquire_version)
      fmt_y = qr_code_get_format_y(@l_qr_inquire_version)
      @l_format_second_x = fmt_x
      @l_format_second_y = fmt_y
      15.times do |num25|
        array3[ga.call(@l_format_first_x[num25], @l_format_first_y[num25], num)] |= 0x40
        array3[ga.call(@l_format_second_x[num25], @l_format_second_y[num25], num)] |= 0x40
      end

      array3
    end

    def qr_code_trans_rs_block(p_deploy_data)
      ecc_idx = @l_qr_set_ecc_rate
      ver_idx = @l_qr_inquire_version - 1
      num  = G_DATA_WORDS[ecc_idx][ver_idx]
      num2 = G_EXTRA_DATA_WORDS[ecc_idx][ver_idx]
      num3 = G_ECC_WORDS[ecc_idx][ver_idx].to_f
      num4 = G_RS_BLOCK_COUNT[ecc_idx][ver_idx]

      arr = Array.new(p_deploy_data.length, 0)
      num5 = 0
      i = 0
      array2 = Array.new(129, 0)
      num6 = 0

      num.times do |jj|
        num4.times do |kk|
          num9 = 0
          if kk > num4 - num2
            num9 = kk - (num4 - num2)
            if jj == 0
              array2[num6] = jj + kk * num + num9 - 1
              num6 += 1
            end
          end
          8.times do |num10|
            num5 = (jj + kk * num + num9) * 8 + num10
            arr[num5] = p_deploy_data[i]
            i += 1
          end
        end
      end

      if num2 > 0
        array2[num6] = num * num4 + num2 - 1
        num6 += 1
      end

      num6.times do |l_idx|
        8.times do |num12|
          num5 = array2[l_idx] * 8 + num12
          arr[num5] = p_deploy_data[i]
          i += 1
        end
      end

      num13 = i
      num14 = (num3 - 1.0).round
      (0..num14).each do |m|
        num4.times do |n|
          8.times do |num16|
            num5 = (num13 + (m.to_f + n.to_f * num3) * 8.0 + num16.to_f).round
            arr[num5] = p_deploy_data[i]
            i += 1
          end
        end
      end

      while i < p_deploy_data.length
        arr[i] = p_deploy_data[i]
        i += 1
      end

      arr
    end

    def qr_get_alpha_numeric_code(c)
      if c >= 0x30 && c <= 0x39
        return c - 0x30
      end
      if c >= 0x41 && c <= 0x5A
        return c - 0x41 + 10
      end
      mapping = {
        0x20 => 36, 0x24 => 37, 0x25 => 38, 0x2A => 39, 0x2B => 40,
        0x2D => 41, 0x2E => 42, 0x2F => 43, 0x3A => 44
      }
      mapping.fetch(c, 0)
    end

    def qr_code_get_8bit_data(p_input_data, p_input_data_bits, p_max_bytes)
      arr = Array.new(p_max_bytes, 0)
      num_items = p_input_data_bits.length
      num2 = 0
      num3 = 8
      num4 = 0

      num_items.times do |i|
        num4 += p_input_data_bits[i]
      end

      num6 = ((num4 - 1) / 8.0).floor
      (0..num6).each do |jj|
        arr[jj] = 0
      end

      num_items.times do |k|
        num8 = p_input_data[k]
        num9 = p_input_data_bits[k]
        break if num9 == 0

        flag = true
        while flag
          if num3 > num9
            arr[num2] = ((arr[num2] << (num9 & 7)) | num8) & 0xFF
            num3 -= num9
            flag = false
          else
            num9 -= num3
            arr[num2] = ((arr[num2] << (num3 & 7)) | (num8 >> num9)) & 0xFF
            if num9 == 0
              flag = false
            else
              num8 &= (1 << num9) - 1
              flag = true
            end
            num2 += 1
            num3 = 8
          end
        end
      end

      if num3 != 8
        arr[num2] = (arr[num2] << (num3 & 7)) & 0xFF
      else
        num2 -= 1
      end

      if num2 < p_max_bytes - 1
        flag = true
        while num2 < p_max_bytes - 1
          num2 += 1
          if flag
            arr[num2] = 236
          else
            arr[num2] = 17
          end
          flag = !flag
        end
      end

      arr
    end

    def qr_code_get_byte_data(p_input1, p_input2)
      if p_input1.length > p_input2.length
        arr  = p_input1.dup
        arr2 = p_input2.dup
      else
        arr  = p_input2.dup
        arr2 = p_input1.dup
      end

      num_a = arr.length
      num2  = arr2.length
      arr3  = Array.new(num_a + 1, 0)

      num_a.times do |i|
        if i < num2
          arr3[i] = arr[i] ^ arr2[i]
        else
          arr3[i] = arr[i]
        end
      end

      arr3
    end

    def qr_code_get_total_data(p_input_data, p_ecc, p_data, p_total)
      ecc_idx = @l_qr_set_ecc_rate
      ver_idx = @l_qr_inquire_version - 1
      num  = G_DATA_WORDS[ecc_idx][ver_idx]
      num2 = G_EXTRA_DATA_WORDS[ecc_idx][ver_idx]
      num3 = G_ECC_WORDS[ecc_idx][ver_idx].to_f
      num4 = G_RS_BLOCK_COUNT[ecc_idx][ver_idx]

      array4 = QR_GEN_POLYS[p_ecc] || []

      # Build multiplication table
      arr_table = Array.new(256) { Array.new(array4.length, 0) }
      array4.length.times do |i|
        num6_val = GF_LOG[array4[i]]
        256.times do |num7|
          num8 = (num6_val + GF_LOG[num7]) % 255
          num9 = GF_EXP[num8]
          arr_table[num7][i] = num9
        end
      end

      # RS block sizes
      @l_rs_block = Array.new(num4, 0)
      num4.times do |jj|
        if jj >= num4 - num2
          @l_rs_block[jj] = (num.to_f + num3 + 1.0).round
        else
          @l_rs_block[jj] = (num.to_f + num3).round
        end
      end

      # Split input data into RS blocks
      array5 = Array.new(@l_rs_block.length) { [] }
      array6 = Array.new(p_total, 0)
      p_input_data.each_with_index do |v, idx|
        break if idx >= p_total
        array6[idx] = v
      end

      @l_rs_block.length.times do |num10_idx|
        @l_rs_block.length.times do |k|
          size = @l_rs_block[num10_idx] - p_ecc
          array5[k] = Array.new(size + 1, 0)
        end
      end

      num11 = 0
      num12 = 0
      p_data.times do |num10|
        array5[num12][num11] = p_input_data[num10]
        num11 += 1
        if num11 >= @l_rs_block[num12] - p_ecc
          num11 = 0
          num12 += 1
        end
      end

      @l_rs_block.length.times do |num12_idx|
        array8 = array5[num12_idx].dup
        num15 = @l_rs_block[num12_idx]
        num16 = num15 - p_ecc

        cnt = num16
        while cnt > 0
          b = array8[0]
          if b != 0
            array9 = array8[1..].dup
            array10 = Array.new(p_ecc, 0)
            p_ecc.times do |l_idx|
              array10[l_idx] = arr_table[b][l_idx]
            end
            array8 = qr_code_get_byte_data(array9, array10)
          elsif p_ecc < array8.length
            array8 = array8[1..]
          else
            array12 = Array.new(p_ecc, 0)
            (p_ecc).times do |idx|
              break if idx >= array8.length - 1
              array12[idx] = array8[idx + 1]
            end
            array12[p_ecc - 1] = 0
            array8 = array12
          end
          cnt -= 1
        end

        dest_start = p_input_data.length + num12_idx * p_ecc
        p_ecc.times do |idx|
          if dest_start + idx < array6.length && idx < array8.length
            array6[dest_start + idx] = array8[idx]
          end
        end
      end

      array6
    end

    def qr_code_get_mask_bit(mask_no, py, px)
      case mask_no
      when 0
        (px + py) % 2 == 0
      when 1
        px % 2 == 0
      when 2
        py % 3 == 0
      when 3
        (px + py) % 3 == 0
      when 4
        (px / 2 + py / 3) % 2 == 0
      when 5
        (px * py) % 2 + (px * py) % 3 == 0
      when 6
        ((px * py) % 2 + (px * py) % 3) % 2 == 0
      when 7
        ((px * py) % 3 + (px + py) % 2) % 2 == 0
      else
        false
      end
    end

    def qr_code_get_mask_bit_group(py, px)
      num = 0
      8.times do |m|
        if qr_code_get_mask_bit(m, py, px)
          num += 1 << m
        end
      end
      num
    end

    def qr_code_choose_mask_number(p_module_data, p_extra_bits)
      length = p_module_data.length
      array  = Array.new(8, 0)
      array2 = Array.new(8, 0)
      array3 = Array.new(8, 0)
      array4 = Array.new(8, 0)
      array5 = Array.new(8, 0)
      array6 = Array.new(8, 0)
      array7 = Array.new(9, 0)
      array8 = Array.new(9, 0)
      array9 = Array.new(9, false)
      array10 = Array.new(9, false)

      length.times do |i|
        8.times do |n|
          array7[n] = 0
          array8[n] = 0
          array9[n] = false
          array10[n] = false
        end

        length.times do |jj|
          num = 0
          num2 = 0
          if jj > 0 && i > 0
            num = p_module_data[jj][i] & p_module_data[jj - 1][i] &
                  p_module_data[jj][i - 1] & p_module_data[jj - 1][i - 1]
            num2 = p_module_data[jj][i] | p_module_data[jj - 1][i] |
                   p_module_data[jj][i - 1] | p_module_data[jj - 1][i - 1]
          end

          8.times do |m|
            array7[m] = ((array7[m] & 0x3F) << 1) | ((p_module_data[jj][i] >> (m & 7)) & 1)
            array8[m] = ((array8[m] & 0x3F) << 1) | ((p_module_data[i][jj] >> (m & 7)) & 1)

            if p_module_data[jj][i] & (1 << m) != 0
              array4[m] += 1
            end
            if array7[m] == 0b1011101
              array3[m] += 40
            end
            if array8[m] == 0b1011101
              array3[m] += 40
            end
            if jj > 0 && i > 0
              if (num & 1) != 0 || (num2 & 1) == 0
                array2[m] += 3
              end
              num >>= 1
              num2 >>= 1
            end

            if (array7[m] & 0x1F) == 0 || (array7[m] & 0x1F) == 31
              if jj > 3
                if array9[m]
                  array[m] += 1
                else
                  array[m] += 3
                  array9[m] = true
                end
              end
            else
              array9[m] = false
            end

            if (array8[m] & 0x1F) == 0 || (array8[m] & 0x1F) == 31
              if jj > 3
                if array10[m]
                  array[m] += 1
                else
                  array[m] += 3
                  array10[m] = true
                end
              end
            else
              array10[m] = false
            end
          end
        end
      end

      num8 = 0
      result = 0
      array11 = [90, 80, 70, 60, 50, 40, 30, 20, 10, 0, 0, 10, 20, 30, 40, 50, 60, 70, 80, 90, 90]

      8.times do |n|
        idx = ((20.0 * array4[n].to_f) / p_extra_bits.to_f).round
        idx = 0 if idx < 0
        idx = 20 if idx > 20
        array5[n] = array11[idx]
        array6[n] = array[n] + array2[n] + array3[n] + array5[n]
      end

      8.times do |o|
        if array6[o] < num8 || o == 0
          result = o
          num8 = array6[o]
        end
      end

      result
    end

    def qr_get_data(data)
      num4 = data.length
      qr_code_inquire_bits(data)
      @l_qr_inquire_text_type = qr_code_set_inquire_text_type(data)
      qr_text_types = @l_qr_inquire_text_type

      if qr_text_types == QR_TEXT_KANJI
        qr_text_types = QR_TEXT_BINARY
        @l_qr_inquire_text_type = QR_TEXT_BINARY
      end

      array6 = Array.new(num4 + 32, 0)
      array7 = Array.new(num4 + 32, 0)

      return [[0]] if num4 <= 0

      array7[0] = 4
      num = 0 # index into array6/array7

      # Version-dependent extra bits for character count indicator
      array5_ver = nil

      if qr_text_types == QR_TEXT_NUMERIC
        array5_ver = [
          0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
          2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
          2, 2, 2, 2, 2, 2, 2, 4, 4, 4,
          4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4
        ]
        array6[num] = 1
        num += 1
        array6[num] = num4
        array7[num] = 10
        num += 1

        num4.times do |jj|
          b = data[jj] - 0x30
          if jj % 3 == 0
            array6[num] = b
            array7[num] = 4
          else
            array6[num] = array6[num] * 10 + b
            if jj % 3 == 1
              array7[num] = 7
            else
              array7[num] = 10
              num += 1 if jj < num4 - 1
            end
          end
        end
        num += 1
      else
        # Binary / default
        array5_ver = [
          0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
          8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
          8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
          8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8
        ]
        array6[num] = 4 # Binary mode indicator
        num += 1
        array6[num] = num4
        array7[num] = 8
        num += 1

        num4.times do |i|
          array6[i + num] = data[i] & 0xFF
          array7[i + num] = 8
        end
        num += num4
      end

      # Calculate total bits
      num2 = 0
      num.times do |l_idx|
        num2 += array7[l_idx]
      end

      # num5 is the index of the character count indicator
      num5 = 1

      # Auto version selection
      @l_qr_inquire_version = 1
      num12 = 1
      num3 = 0
      while num12 <= 40
        if QR_ARRAY3_CAP[@l_qr_set_ecc_rate][num12] >= num2 + array5_ver[@l_qr_inquire_version]
          num3 = QR_ARRAY3_CAP[@l_qr_set_ecc_rate][num12]
          break
        end
        @l_qr_inquire_version += 1
        num12 += 1
      end

      raise "data too long for QR Code" if @l_qr_inquire_version > 40

      # Version adjustment (C# QR.cs compatible logic) - QR version bug fix
      if @l_qr_inquire_version != 0
        if @l_qr_inquire_version > @l_qr_set_version
          num3 = QR_ARRAY3_CAP[@l_qr_set_ecc_rate][@l_qr_inquire_version]
        else
          @l_qr_inquire_version = @l_qr_set_version
          num3 = QR_ARRAY3_CAP[@l_qr_set_ecc_rate][@l_qr_inquire_version]
        end
      end

      num2 += array5_ver[@l_qr_inquire_version]
      array7[num5] = array7[num5] + array5_ver[@l_qr_inquire_version]
      num14 = QR_ARRAY_CAP[@l_qr_inquire_version]
      num15 = QR_ARRAY2_EXTRA[@l_qr_inquire_version] + (num14 << 3)

      fmt_x = qr_code_get_format_x(@l_qr_inquire_version)
      fmt_y = qr_code_get_format_y(@l_qr_inquire_version)
      @l_format_second_x = fmt_x
      @l_format_second_y = fmt_y

      p_ecc = G_ECC_WORDS[@l_qr_set_ecc_rate][@l_qr_inquire_version - 1]
      num16 = (num3 / 8.0).round
      num17 = 4 * @l_qr_inquire_version + 17
      @l_qr_inquire_module_size = num17

      array9 = qr_code_get_symbol_data(@l_qr_inquire_version)

      @l_deploy_table_x = Array.new(num15 + 1, 0)
      @l_deploy_table_y = Array.new(num15 + 1, 0)
      @l_deploy_order_x = Array.new(num15, 0)
      @l_deploy_order_y = Array.new(num15, 0)

      # Data placement order
      num19 = 0
      m_val = num17 - 1
      while m_val >= 3
        (num17 - 1).downto(0) do |n_val|
          num20 = n_val * (num17 + 1) + m_val
          if (array9[num20] & 0xFE) == 0
            @l_deploy_table_x[num19] = m_val
            @l_deploy_table_y[num19] = n_val
            num19 += 1
          end
          num20 = n_val * (num17 + 1) + m_val - 1
          if (array9[num20] & 0xFE) == 0
            @l_deploy_table_x[num19] = m_val - 1
            @l_deploy_table_y[num19] = n_val
            num19 += 1
          end
        end

        m_val = 7 if m_val == 8

        num17.times do |num22|
          num23 = num22 * (num17 + 1) + m_val - 2
          if (array9[num23] & 0xFE) == 0
            @l_deploy_table_x[num19] = m_val - 2
            @l_deploy_table_y[num19] = num22
            num19 += 1
          end
          num23 = num22 * (num17 + 1) + m_val - 3
          if (array9[num23] & 0xFE) == 0
            @l_deploy_table_x[num19] = m_val - 3
            @l_deploy_table_y[num19] = num22
            num19 += 1
          end
        end
        m_val -= 4
      end

      @l_deploy_order_x = qr_code_trans_rs_block(@l_deploy_table_x)
      @l_deploy_order_y = qr_code_trans_rs_block(@l_deploy_table_y)

      # Terminator
      if num2 <= num3 - 4
        array6[num] = 0
        array7[num] = 4
      elsif num2 < num3
        array6[num] = 0
        array7[num] = num3 - num2
      end

      # Convert to 8-bit data
      p_input_data = qr_code_get_8bit_data(array6, array7, num16)

      # Add error correction data
      array10 = qr_code_get_total_data(p_input_data, p_ecc, num16, num14)

      # Initialize QR matrix
      array11 = Array.new(num17) { Array.new(num17, 0) }

      # Place data into matrix
      num14.times do |num29|
        b5 = array10[num29]
        7.downto(0) do |num30|
          num31 = num29 * 8 + num30
          num32 = qr_code_get_mask_bit_group(
            @l_deploy_order_x[num31], @l_deploy_order_y[num31])
          array11[@l_deploy_order_x[num31]][@l_deploy_order_y[num31]] =
            (255 * (b5 & 1)) ^ num32
          b5 >>= 1
        end
      end

      num33 = QR_ARRAY2_EXTRA[@l_qr_inquire_version]
      while num33 > 0
        num34 = num33 + num14 * 8 - 1
        num32 = qr_code_get_mask_bit_group(
          @l_deploy_order_x[num34], @l_deploy_order_y[num34])
        array11[@l_deploy_order_x[num34]][@l_deploy_order_y[num34]] = 0xFF ^ num32
        num33 -= 1
      end

      # Choose mask pattern
      b6 = qr_code_choose_mask_number(
        array11, QR_ARRAY2_EXTRA[@l_qr_inquire_version] + num14 * 8)
      b7 = (2.0 ** b6.to_f).round
      b8 = b6 + @l_qr_set_ecc_rate * 8

      # Place format information
      15.times do |num35|
        b9 = QR_FORMAT_INFO[b8][num35].ord - '0'.ord
        array11[@l_format_first_x[num35]][@l_format_first_y[num35]] = b9 * 255
        array11[@l_format_second_x[num35]][@l_format_second_y[num35]] = b9 * 255
      end

      # Build final matrix
      array12 = Array.new(num17) { Array.new(num17, 0) }
      num36 = 0
      num17.times do |num38|
        num17.times do |num40|
          if (array11[num40][num38] & b7) != 0 || (array9[num36] & 1) == 1
            array12[num40][num38] = 1 | (array9[num36] & 0xFE)
          else
            array12[num40][num38] = 0 | (array9[num36] & 0xFE)
          end
          num36 += 1
        end
        num36 += 1
      end

      array12
    end

    def make_arr(data)
      raise "empty data" if data.nil? || data.empty?

      array = qr_get_data(data)
      size = array.length
      raise "QR encoding produced empty array" if size <= 0

      @patt = Array.new(size) { |i|
        Array.new(size) { |jj|
          b = array[i][jj]
          (b & 1) == 1
        }
      }
    end
  end
end
