# frozen_string_literal: true

require_relative "pdf417_tables"

module BarcodePao
  # PDF417 error level constants.
  PDF417_ERROR_AUTO   = -1
  PDF417_ERROR_LEVEL0 = 0
  PDF417_ERROR_LEVEL1 = 1
  PDF417_ERROR_LEVEL2 = 2
  PDF417_ERROR_LEVEL3 = 3
  PDF417_ERROR_LEVEL4 = 4
  PDF417_ERROR_LEVEL5 = 5
  PDF417_ERROR_LEVEL6 = 6
  PDF417_ERROR_LEVEL7 = 7
  PDF417_ERROR_LEVEL8 = 8

  # PDF417 size mode constants.
  PDF417_SIZE_AUTO            = 0
  PDF417_SIZE_ROWS            = 1
  PDF417_SIZE_COLUMNS         = 2
  PDF417_SIZE_COLUMNS_AND_ROWS = 3

  class PDF417 < BarcodeBase2D
    attr_accessor :error_correction_level, :columns, :rows, :size_mode,
                  :aspect_ratio, :y_height, :quiet_zone_width, :use_auto_error_level

    def initialize(output_format = FORMAT_PNG)
      super(output_format)
      @error_correction_level = PDF417_ERROR_LEVEL2
      @columns = 0
      @rows = 0
      @size_mode = PDF417_SIZE_AUTO
      @aspect_ratio = 0.5
      @y_height = 3
      @quiet_zone_width = 2
      @use_auto_error_level = true
    end

    # --- Settings ---

    def set_error_level(level)
      if level >= PDF417_ERROR_AUTO && level <= PDF417_ERROR_LEVEL8
        @error_correction_level = level
      end
    end

    def set_columns(cols)
      @columns = cols if cols >= 0 && cols <= 30
    end

    def set_rows(r)
      @rows = r if r >= 0 && r <= 90
    end

    def set_size_mode(mode)
      if mode >= PDF417_SIZE_AUTO && mode <= PDF417_SIZE_COLUMNS_AND_ROWS
        @size_mode = mode
      end
    end

    def set_aspect_ratio(ratio)
      @aspect_ratio = ratio if ratio >= 0.001 && ratio <= 1000.0
    end

    def set_y_height(h)
      @y_height = h if h > 0
    end

    def set_quiet_zone_width(w)
      @quiet_zone_width = w if w >= 0
    end

    def set_use_auto_error_level(on)
      @use_auto_error_level = on
    end

    # --- GetPattern ---

    # Generates the PDF417 pattern as a row-major bool matrix (true=black).
    def get_pattern(code)
      data = code.bytes
      raise "empty input" if data.empty?

      enc = PDF417Encoder.new
      apply_settings_cal(enc, data)
      enc.paint_code
      result = enc.convert_to_bool_matrix
      raise "failed to generate pattern" if result.nil? || result.empty?
      result
    end

    # --- Draw ---

    # Renders the PDF417 barcode to the internal buffer (SVG or PNG).
    def draw(code, width, height)
      raise "width and height must be positive" if width <= 0 || height <= 0

      patt = get_pattern(code)
      num_rows = patt.length
      raise "empty pattern" if num_rows == 0
      num_cols = patt[0].length
      raise "empty pattern" if num_cols == 0

      if svg_output?
        draw_svg_pdf417(patt, num_rows, num_cols, width, height)
      else
        draw_png_pdf417(patt, num_rows, num_cols, width, height)
      end
    end

    private

    # --- Internal: apply settings to encoder ---

    def apply_settings_cal(enc, data)
      enc.error_level = @error_correction_level
      enc.code_columns = @columns
      enc.code_rows = @rows
      enc.aspect_ratio = @aspect_ratio
      enc.y_height = @y_height
      enc.use_auto_error_level = @use_auto_error_level
      enc.text = data

      case @size_mode
      when PDF417_SIZE_AUTO
        enc.size_kind = PDF417Encoder::SIZE_AUTO
      when PDF417_SIZE_COLUMNS
        enc.size_kind = PDF417Encoder::SIZE_COLUMNS
      when PDF417_SIZE_ROWS
        enc.size_kind = PDF417Encoder::SIZE_ROWS
      when PDF417_SIZE_COLUMNS_AND_ROWS
        enc.size_kind = PDF417Encoder::SIZE_COLUMNS_AND_ROWS
      end
    end

    # --- SVG rendering ---

    def draw_svg_pdf417(patt, num_rows, num_cols, width, height)
      if @fit_width
        module_w = width.to_f / num_cols
        module_h = height.to_f / num_rows
      else
        module_w = (width / num_cols).to_f
        module_h = (height / num_rows).to_f
        module_w = 1.0 if module_w < 1.0
        module_h = 1.0 if module_h < 1.0
      end

      aw = (module_w * num_cols).ceil
      ah = (module_h * num_rows).ceil

      svg_begin(aw, ah)
      svg_rect(0, 0, aw.to_f, ah.to_f, @background_color)

      adj = @px_adjust_black.to_f
      num_rows.times do |r|
        num_cols.times do |c|
          if patt[r][c]
            dw = module_w + adj + 0.5
            dh = module_h + adj + 0.5
            dw = 0 if dw < 0
            dh = 0 if dh < 0
            svg_rect(c * module_w, r * module_h, dw, dh, @foreground_color)
          end
        end
      end

      svg_end
    end

    # --- PNG rendering ---

    def draw_png_pdf417(patt, num_rows, num_cols, width, height)
      if @fit_width
        module_w = width.to_f / num_cols
        module_h = height.to_f / num_rows
      else
        module_w = (width / num_cols).to_f
        module_h = (height / num_rows).to_f
        module_w = 1.0 if module_w < 1.0
        module_h = 1.0 if module_h < 1.0
      end

      aw = (module_w * num_cols).ceil
      ah = (module_h * num_rows).ceil

      require_relative "render_png"
      img = PNGImage.new(aw, ah, @background_color)

      adj = @px_adjust_black
      num_rows.times do |r|
        num_cols.times do |c|
          if patt[r][c]
            draw_x = (c * module_w).to_i
            draw_y = (r * module_h).to_i
            draw_w = (module_w + adj.to_f).to_i
            draw_h = (module_h + adj.to_f).to_i
            draw_w = 1 if draw_w < 1
            draw_h = 1 if draw_h < 1
            end_x = draw_x + draw_w
            end_y = draw_y + draw_h
            end_x = aw if end_x > aw
            end_y = ah if end_y > ah
            img.fill_rect(draw_x, draw_y, end_x, end_y, @foreground_color)
          end
        end
      end

      if BarcodePao.trial_mode?
        img.draw_sample_overlay(0, 0, aw, ah)
      end

      @image_buffer = img.to_png
    end
  end

  # ===========================================================================
  # PDF417Encoder - internal encoding engine
  # ===========================================================================
  class PDF417Encoder
    # Internal constants matching the C++/Go implementation.
    START_PATTERN       = 0x1fea8
    STOP_PATTERN        = 0x3fa29
    START_CODE_SIZE     = 17
    STOP_SIZE           = 18
    MOD                 = 929
    ALPHA               = 0x10000
    LOWER               = 0x20000
    MIXED               = 0x40000
    PUNCTUATION         = 0x80000
    IS_BYTE             = 0x100000
    BYTE_SHIFT          = 913
    PL                  = 25
    LL                  = 27
    AS                  = 27
    ML                  = 28
    AL                  = 28
    PS                  = 29
    PAL                 = 29
    SPACE               = 26
    TEXT_MODE            = 900
    BYTE_MODE_6          = 924
    BYTE_MODE            = 901
    NUMERIC_MODE         = 902
    ABSOLUTE_MAX_TEXT_SIZE = 5420
    MAX_DATA_CODEWORDS    = 926

    # Internal size kind constants.
    SIZE_AUTO            = 0
    SIZE_COLUMNS         = 1
    SIZE_ROWS            = 2
    SIZE_COLUMNS_AND_ROWS = 3

    MIXED_SET       = "0123456789&\r\t,:#-.$/+%*=^"
    PUNCTUATION_SET = ";<>@[\\]_`~!\r\t,:\n-.$/\"|*()?{}'"

    attr_accessor :out_bits, :bit_ptr, :bit_columns,
                  :error_level, :len_codewords, :code_columns, :code_rows,
                  :use_auto_error_level, :aspect_ratio, :y_height,
                  :options, :size_kind, :codewords, :text, :cw_ptr,
                  :segment_list

    def initialize
      @out_bits = []
      @bit_ptr = 0
      @bit_columns = 0
      @error_level = 2
      @len_codewords = 0
      @code_columns = 0
      @code_rows = 0
      @use_auto_error_level = true
      @aspect_ratio = 0.5
      @y_height = 3
      @options = 0
      @size_kind = SIZE_AUTO
      @codewords = []
      @text = []
      @cw_ptr = 0
      @segment_list = []
    end

    def init_block
      @codewords = Array.new(MAX_DATA_CODEWORDS + 2, 0)
    end

    # --- Bit output ---

    def out_codeword17(codeword)
      byte_ptr = @bit_ptr >> 3
      bit = @bit_ptr - byte_ptr * 8
      @out_bits[byte_ptr] = (@out_bits[byte_ptr] || 0) | ((codeword >> (9 + bit)) & 0xFF)
      byte_ptr += 1
      @out_bits[byte_ptr] = (@out_bits[byte_ptr] || 0) | ((codeword >> (1 + bit)) & 0xFF)
      byte_ptr += 1
      codeword <<= 8
      @out_bits[byte_ptr] = (@out_bits[byte_ptr] || 0) | ((codeword >> (1 + bit)) & 0xFF)
      @bit_ptr += 17
    end

    def out_codeword18(codeword)
      byte_ptr = @bit_ptr >> 3
      bit = @bit_ptr - byte_ptr * 8
      @out_bits[byte_ptr] = (@out_bits[byte_ptr] || 0) | ((codeword >> (10 + bit)) & 0xFF)
      byte_ptr += 1
      @out_bits[byte_ptr] = (@out_bits[byte_ptr] || 0) | ((codeword >> (2 + bit)) & 0xFF)
      byte_ptr += 1
      codeword <<= 8
      @out_bits[byte_ptr] = (@out_bits[byte_ptr] || 0) | ((codeword >> (2 + bit)) & 0xFF)
      if bit == 7
        byte_ptr += 1
        @out_bits[byte_ptr] = (@out_bits[byte_ptr] || 0) | 0x80
      end
      @bit_ptr += 18
    end

    def out_codeword(codeword)
      out_codeword17(codeword)
    end

    def out_start_pattern
      out_codeword17(START_PATTERN)
    end

    def out_stop_pattern
      out_codeword18(STOP_PATTERN)
    end

    # --- Paint code ---

    def out_paint_code
      @bit_columns = START_CODE_SIZE * (@code_columns + 3) + STOP_SIZE
      bytes_per_row = (@bit_columns - 1) / 8 + 1
      len_bits = bytes_per_row * @code_rows
      @out_bits = Array.new(len_bits, 0)

      code_ptr = 0
      @code_rows.times do |row|
        @bit_ptr = bytes_per_row * 8 * row
        row_mod = row % 3
        cluster = PDF417Tables::CLUSTERS[row_mod]

        out_start_pattern

        # Left edge indicator
        case row_mod
        when 0
          edge = 30 * (row / 3) + (@code_rows - 1) / 3
        when 1
          edge = 30 * (row / 3) + @error_level * 3 + (@code_rows - 1) % 3
        else
          edge = 30 * (row / 3) + @code_columns - 1
        end
        out_codeword(cluster[edge])

        # Data codewords
        @code_columns.times do |_col|
          break if code_ptr >= @len_codewords
          cw = @codewords[code_ptr]
          break if cw < 0 || cw >= cluster.length
          out_codeword(cluster[cw])
          code_ptr += 1
        end

        # Right edge indicator
        case row_mod
        when 0
          edge = 30 * (row / 3) + @code_columns - 1
        when 1
          edge = 30 * (row / 3) + (@code_rows - 1) / 3
        else
          edge = 30 * (row / 3) + @error_level * 3 + (@code_rows - 1) % 3
        end
        out_codeword(cluster[edge])

        out_stop_pattern
      end

      if (@options & 0x02) != 0 # PDF417_INVERT_BITMAP
        @out_bits.length.times do |k|
          @out_bits[k] ^= 0xFF
        end
      end
    end

    # --- Error correction ---

    def calculate_error_correction(dest)
      el = @error_level
      el = 0 if el < 0 || el > 8
      a = PDF417Tables::ERROR_LEVELS[el]
      a_length = 2 << el
      a_length.times do |k|
        @codewords[dest + k] = 0
      end
      last_e = a_length - 1
      @len_codewords.times do |k|
        t1 = @codewords[k] + @codewords[dest]
        a_length.times do |e|
          t2 = (t1 * a[last_e - e]) % MOD
          t3 = MOD - t2
          nxt = 0
          nxt = @codewords[dest + e + 1] if e != last_e
          @codewords[dest + e] = (nxt + t3) % MOD
        end
      end
      a_length.times do |k|
        @codewords[dest + k] = (MOD - @codewords[dest + k]) % MOD
      end
    end

    # --- Text type/value ---

    def get_text_type_and_value(max_length, idx)
      return 0 if idx >= max_length
      c = @text[idx] & 0xFF
      ch = c

      if ch >= 65 && ch <= 90 # 'A'..'Z'
        return ALPHA + c - 65
      end
      if ch >= 97 && ch <= 122 # 'a'..'z'
        return LOWER + c - 97
      end
      if ch == 32 # ' '
        return ALPHA + LOWER + MIXED + SPACE
      end

      ms = MIXED_SET.index(ch.chr)
      ps = PUNCTUATION_SET.index(ch.chr)
      ms = ms.nil? ? -1 : ms
      ps = ps.nil? ? -1 : ps

      if ms < 0 && ps < 0
        return IS_BYTE + c
      end
      if ms == ps
        return MIXED + PUNCTUATION + ms
      end
      if ms >= 0
        return MIXED + ms
      end
      PUNCTUATION + ps
    end

    # --- Text compaction ---

    def text_compaction(start, length)
      dest = Array.new(ABSOLUTE_MAX_TEXT_SIZE * 2, 0)
      mode = ALPHA
      ptr = 0
      full_bytes = 0
      length += start
      k = start
      while k < length
        v = get_text_type_and_value(length, k)
        if (v & mode) != 0
          dest[ptr] = v & 0xFF
          ptr += 1
          k += 1
          next
        end
        if (v & IS_BYTE) != 0
          if (ptr & 1) != 0
            if (mode & PUNCTUATION) != 0
              dest[ptr] = PAL
            else
              dest[ptr] = PS
            end
            ptr += 1
            mode = ALPHA if (mode & PUNCTUATION) != 0
          end
          dest[ptr] = BYTE_SHIFT
          ptr += 1
          dest[ptr] = v & 0xFF
          ptr += 1
          full_bytes += 2
          k += 1
          next
        end

        if mode == ALPHA
          if (v & LOWER) != 0
            dest[ptr] = LL
            ptr += 1
            dest[ptr] = v & 0xFF
            ptr += 1
            mode = LOWER
          elsif (v & MIXED) != 0
            dest[ptr] = ML
            ptr += 1
            dest[ptr] = v & 0xFF
            ptr += 1
            mode = MIXED
          elsif (get_text_type_and_value(length, k + 1) &
                 get_text_type_and_value(length, k + 2) & PUNCTUATION) != 0
            dest[ptr] = ML
            ptr += 1
            dest[ptr] = PL
            ptr += 1
            dest[ptr] = v & 0xFF
            ptr += 1
            mode = PUNCTUATION
          else
            dest[ptr] = PS
            ptr += 1
            dest[ptr] = v & 0xFF
            ptr += 1
          end
        elsif mode == LOWER
          if (v & ALPHA) != 0
            if (get_text_type_and_value(length, k + 1) &
                get_text_type_and_value(length, k + 2) & ALPHA) != 0
              dest[ptr] = ML
              ptr += 1
              dest[ptr] = AL
              ptr += 1
              mode = ALPHA
            else
              dest[ptr] = AS
              ptr += 1
            end
            dest[ptr] = v & 0xFF
            ptr += 1
          elsif (v & MIXED) != 0
            dest[ptr] = ML
            ptr += 1
            dest[ptr] = v & 0xFF
            ptr += 1
            mode = MIXED
          elsif (get_text_type_and_value(length, k + 1) &
                 get_text_type_and_value(length, k + 2) & PUNCTUATION) != 0
            dest[ptr] = ML
            ptr += 1
            dest[ptr] = PL
            ptr += 1
            dest[ptr] = v & 0xFF
            ptr += 1
            mode = PUNCTUATION
          else
            dest[ptr] = PS
            ptr += 1
            dest[ptr] = v & 0xFF
            ptr += 1
          end
        elsif mode == MIXED
          if (v & LOWER) != 0
            dest[ptr] = LL
            ptr += 1
            dest[ptr] = v & 0xFF
            ptr += 1
            mode = LOWER
          elsif (v & ALPHA) != 0
            dest[ptr] = AL
            ptr += 1
            dest[ptr] = v & 0xFF
            ptr += 1
            mode = ALPHA
          elsif (get_text_type_and_value(length, k + 1) &
                 get_text_type_and_value(length, k + 2) & PUNCTUATION) != 0
            dest[ptr] = PL
            ptr += 1
            dest[ptr] = v & 0xFF
            ptr += 1
            mode = PUNCTUATION
          else
            dest[ptr] = PS
            ptr += 1
            dest[ptr] = v & 0xFF
            ptr += 1
          end
        elsif mode == PUNCTUATION
          dest[ptr] = PAL
          ptr += 1
          mode = ALPHA
          k -= 1 # re-process this character
        end

        k += 1
      end

      if (ptr & 1) != 0
        dest[ptr] = PS
        ptr += 1
      end
      size = (ptr + full_bytes) / 2
      raise "the text is too big" if size + @cw_ptr > MAX_DATA_CODEWORDS

      length_end = ptr
      ptr = 0
      while ptr < length_end
        v = dest[ptr]
        ptr += 1
        if v >= 30
          @codewords[@cw_ptr] = v
          @cw_ptr += 1
          @codewords[@cw_ptr] = dest[ptr]
          @cw_ptr += 1
          ptr += 1
        else
          @codewords[@cw_ptr] = v * 30 + dest[ptr]
          @cw_ptr += 1
          ptr += 1
        end
      end
    end

    # --- Number compaction ---

    def basic_number_compaction(start, length)
      ret = @cw_ptr
      ret_last = length / 3
      @cw_ptr += ret_last + 1
      (ret_last + 1).times do |k|
        @codewords[ret + k] = 0
      end
      @codewords[ret + ret_last] = 1
      end_pos = length + start
      (start...end_pos).each do |ni|
        ret_last.downto(0) do |k|
          @codewords[ret + k] *= 10
        end
        @codewords[ret + ret_last] += @text[ni] - 48 # '0' = 48
        ret_last.downto(1) do |k|
          @codewords[ret + k - 1] += @codewords[ret + k] / 900
          @codewords[ret + k] %= 900
        end
      end
    end

    def number_compaction(start, length)
      full = (length / 44) * 15
      sz = length % 44
      if sz == 0
        size = full
      else
        size = full + sz / 3 + 1
      end
      raise "the text is too big" if size + @cw_ptr > MAX_DATA_CODEWORDS

      end_pos = length + start
      k = start
      while k < end_pos
        seg_sz = end_pos - k
        seg_sz = 44 if seg_sz > 44
        basic_number_compaction(k, seg_sz)
        k += 44
      end
    end

    # --- Byte compaction ---

    def byte_compaction6(start)
      length = 6
      ret = @cw_ptr
      ret_last = 4
      @cw_ptr += ret_last + 1
      (ret_last + 1).times do |k|
        @codewords[ret + k] = 0
      end
      end_pos = length + start
      (start...end_pos).each do |ni|
        ret_last.downto(0) do |k|
          @codewords[ret + k] *= 256
        end
        @codewords[ret + ret_last] += @text[ni] & 0xFF
        ret_last.downto(1) do |k|
          @codewords[ret + k - 1] += @codewords[ret + k] / 900
          @codewords[ret + k] %= 900
        end
      end
    end

    def byte_compaction(start, length)
      size = (length / 6) * 5 + (length % 6)
      raise "the text is too big" if size + @cw_ptr > MAX_DATA_CODEWORDS

      end_pos = length + start
      k = start
      while k < end_pos
        sz = end_pos - k
        sz = 6 if sz > 6
        if sz < 6
          sz.times do |j|
            @codewords[@cw_ptr] = @text[k + j] & 0xFF
            @cw_ptr += 1
          end
        else
          byte_compaction6(k)
        end
        k += 6
      end
    end

    # --- Break string ---

    def break_string
      text_length = @text.length
      last_p = 0
      start_n = 0
      nd = 0

      text_length.times do |k|
        c = @text[k] & 0xFF
        ch = c
        if ch >= 48 && ch <= 57 # '0'..'9'
          start_n = k if nd == 0
          nd += 1
          next
        end
        if nd >= 13
          if last_p != start_n
            c2 = @text[last_p] & 0xFF
            ch2 = c2
            last_txt = (ch2 >= 32 && ch2 < 0x7f) || ch2 == 13 || ch2 == 10 || ch2 == 9
            (last_p...start_n).each do |j|
              c2 = @text[j] & 0xFF
              ch2 = c2
              txt = (ch2 >= 32 && ch2 < 0x7f) || ch2 == 13 || ch2 == 10 || ch2 == 9
              if txt != last_txt
                seg_t = last_txt ? "T" : "B"
                @segment_list << { type: seg_t, start: last_p, end: j }
                last_p = j
                last_txt = txt
              end
            end
            seg_t = last_txt ? "T" : "B"
            @segment_list << { type: seg_t, start: last_p, end: start_n }
          end
          @segment_list << { type: "N", start: start_n, end: k }
          last_p = k
        end
        nd = 0
      end

      if nd < 13
        start_n = text_length
      end
      if last_p != start_n
        c2 = @text[last_p] & 0xFF
        ch2 = c2
        last_txt = (ch2 >= 32 && ch2 < 0x7f) || ch2 == 13 || ch2 == 10 || ch2 == 9
        (last_p...start_n).each do |j|
          c2 = @text[j] & 0xFF
          ch2 = c2
          txt = (ch2 >= 32 && ch2 < 0x7f) || ch2 == 13 || ch2 == 10 || ch2 == 9
          if txt != last_txt
            seg_t = last_txt ? "T" : "B"
            @segment_list << { type: seg_t, start: last_p, end: j }
            last_p = j
            last_txt = txt
          end
        end
        seg_t = last_txt ? "T" : "B"
        @segment_list << { type: seg_t, start: last_p, end: start_n }
      end
      if nd >= 13
        @segment_list << { type: "N", start: start_n, end: text_length }
      end

      # Merge pass 1: single-byte segments between text segments
      k = 0
      while k < @segment_list.length
        v = @segment_list[k]
        vp = k > 0 ? @segment_list[k - 1] : nil
        vn = (k + 1) < @segment_list.length ? @segment_list[k + 1] : nil
        if v[:type] == "B" && (v[:end] - v[:start]) == 1
          if !vp.nil? && !vn.nil? &&
             vp[:type] == "T" && vn[:type] == "T" &&
             (vp[:end] - vp[:start]) + (vn[:end] - vn[:start]) >= 3
            vp[:end] = vn[:end]
            @segment_list.delete_at(k + 1)
            @segment_list.delete_at(k)
            k = 0
            next
          end
        end
        k += 1
      end

      # Merge pass 2: absorb short neighbors into long text segments
      k = 0
      while k < @segment_list.length
        v = @segment_list[k]
        if v[:type] == "T" && (v[:end] - v[:start]) >= 5
          redo_flag = false
          if k > 0
            vp = @segment_list[k - 1]
            if (vp[:type] == "B" && (vp[:end] - vp[:start]) == 1) || vp[:type] == "T"
              redo_flag = true
              v[:start] = vp[:start]
              @segment_list.delete_at(k - 1)
              k -= 1
              v = @segment_list[k]
            end
          end
          if (k + 1) < @segment_list.length
            vn = @segment_list[k + 1]
            if (vn[:type] == "B" && (vn[:end] - vn[:start]) == 1) || vn[:type] == "T"
              redo_flag = true
              v[:end] = vn[:end]
              @segment_list.delete_at(k + 1)
            end
          end
          if redo_flag
            k = 0
            next
          end
        end
        k += 1
      end

      # Merge pass 3: absorb short text segments into byte segments
      k = 0
      while k < @segment_list.length
        v = @segment_list[k]
        if v[:type] == "B"
          redo_flag = false
          if k > 0
            vp = @segment_list[k - 1]
            if (vp[:type] == "T" && (vp[:end] - vp[:start]) < 5) || vp[:type] == "B"
              redo_flag = true
              v[:start] = vp[:start]
              @segment_list.delete_at(k - 1)
              k -= 1
              v = @segment_list[k]
            end
          end
          if (k + 1) < @segment_list.length
            vn = @segment_list[k + 1]
            if (vn[:type] == "T" && (vn[:end] - vn[:start]) < 5) || vn[:type] == "B"
              redo_flag = true
              v[:end] = vn[:end]
              @segment_list.delete_at(k + 1)
            end
          end
          if redo_flag
            k = 0
            next
          end
        end
        k += 1
      end

      # Special case: single text segment of all digits
      if @segment_list.length == 1
        v = @segment_list[0]
        if v[:type] == "T" && (v[:end] - v[:start]) >= 8
          all_digits = true
          (v[:start]...v[:end]).each do |kk|
            ch = @text[kk] & 0xFF
            if ch < 48 || ch > 57
              all_digits = false
              break
            end
          end
          v[:type] = "N" if all_digits
        end
      end
    end

    # --- Assemble ---

    def assemble
      return if @segment_list.empty?
      @cw_ptr = 1
      @segment_list.each_with_index do |v, k|
        seg_len = v[:end] - v[:start]
        case v[:type]
        when "T"
          if k != 0
            @codewords[@cw_ptr] = TEXT_MODE
            @cw_ptr += 1
          end
          text_compaction(v[:start], seg_len)
        when "N"
          @codewords[@cw_ptr] = NUMERIC_MODE
          @cw_ptr += 1
          number_compaction(v[:start], seg_len)
        when "B"
          if seg_len % 6 != 0
            @codewords[@cw_ptr] = BYTE_MODE
          else
            @codewords[@cw_ptr] = BYTE_MODE_6
          end
          @cw_ptr += 1
          byte_compaction(v[:start], seg_len)
        end
      end
    end

    # --- Max possible error level ---

    def max_possible_error_level(remain)
      level = 8
      size = 512
      while level > 0
        return level if remain >= size
        level -= 1
        size >>= 1
      end
      0
    end

    # --- Max square ---

    def max_square
      if @code_columns > 21
        @code_columns = 29
        @code_rows = 32
      else
        @code_columns = 16
        @code_rows = 58
      end
      MAX_DATA_CODEWORDS + 2
    end

    # --- Paint code (main entry) ---

    def paint_code
      init_block

      raise "text cannot be empty" if @text.empty?
      raise "the text is too big" if @text.length > ABSOLUTE_MAX_TEXT_SIZE

      @segment_list = []
      break_string
      assemble
      @segment_list = []
      @codewords[0] = @cw_ptr
      @len_codewords = @cw_ptr

      max_err = max_possible_error_level(MAX_DATA_CODEWORDS + 2 - @len_codewords)

      if @use_auto_error_level
        if @len_codewords < 41
          @error_level = 2
        elsif @len_codewords < 161
          @error_level = 3
        elsif @len_codewords < 321
          @error_level = 4
        else
          @error_level = 5
        end
      end

      if @error_level < 0
        @error_level = 0
      elsif @error_level > max_err
        @error_level = max_err
      end

      @code_columns = 1 if @code_columns < 1
      @code_columns = 30 if @code_columns > 30

      @code_rows = 3 if @code_rows < 3
      @code_rows = 90 if @code_rows > 90

      len_err = 2 << @error_level
      fixed_column = @size_kind != SIZE_ROWS
      skip_row_col_adjust = false
      tot = @len_codewords + len_err

      if @size_kind == SIZE_COLUMNS_AND_ROWS
        tot = @code_columns * @code_rows
        tot = max_square if tot > MAX_DATA_CODEWORDS + 2
        if tot < @len_codewords + len_err
          tot = @len_codewords + len_err
        else
          skip_row_col_adjust = true
        end
      elsif @size_kind == SIZE_AUTO
        fixed_column = true
        @aspect_ratio = 0.001 if @aspect_ratio < 0.001
        @aspect_ratio = 1000 if @aspect_ratio > 1000
        b = 73 * @aspect_ratio - 4
        c = (-b + Math.sqrt(b * b + 4 * 17 * @aspect_ratio *
            (@len_codewords + len_err).to_f * @y_height.to_f)) /
            (2 * 17 * @aspect_ratio)
        @code_columns = (c + 0.5).to_i
        @code_columns = 1 if @code_columns < 1
        @code_columns = 30 if @code_columns > 30
      end

      unless skip_row_col_adjust
        if fixed_column
          @code_rows = (tot - 1) / @code_columns + 1
          if @code_rows < 3
            @code_rows = 3
          elsif @code_rows > 90
            @code_rows = 90
            @code_columns = (tot - 1) / 90 + 1
          end
        else
          @code_columns = (tot - 1) / @code_rows + 1
          if @code_columns > 30
            @code_columns = 30
            @code_rows = (tot - 1) / 30 + 1
          end
        end
        tot = @code_rows * @code_columns
      end

      tot = max_square if tot > MAX_DATA_CODEWORDS + 2

      @error_level = max_possible_error_level(tot - @len_codewords)
      len_err = 2 << @error_level
      pad = tot - len_err - @len_codewords
      @cw_ptr = @len_codewords
      while pad > 0
        @codewords[@cw_ptr] = TEXT_MODE
        @cw_ptr += 1
        pad -= 1
      end
      @codewords[0] = @cw_ptr
      @len_codewords = @cw_ptr
      calculate_error_correction(@len_codewords)
      @len_codewords = tot

      out_paint_code
    end

    # --- Convert to bool matrix ---

    def convert_to_bool_matrix
      return nil if @out_bits.nil? || @out_bits.empty? || @code_rows == 0 || @bit_columns == 0

      rows = @code_rows
      bit_cols = @bit_columns
      bytes_per_row = (bit_cols + 7) / 8
      result = Array.new(rows) { Array.new(bit_cols, false) }
      rows.times do |r|
        bit_cols.times do |c|
          byte_index = r * bytes_per_row + c / 8
          bit_index = 7 - (c % 8)
          if byte_index < @out_bits.length
            result[r][c] = ((@out_bits[byte_index] || 0) & (1 << bit_index)) != 0
          end
        end
      end
      result
    end
  end
end
