# frozen_string_literal: true

module BarcodePao
  NW7_CHARS = "ABCD.+:/$-0123456789"
  NW7_PATTERNS = [
    0x1a, 0x29, 0x0b, 0x0e, 0x54, 0x15, 0x45, 0x51, 0x18, 0x0c,
    0x03, 0x06, 0x09, 0x60, 0x12, 0x42, 0x21, 0x24, 0x30, 0x48,
  ].freeze

  class NW7 < BarcodeBase1D
    attr_accessor :show_start_stop

    def initialize(output_format = FORMAT_PNG)
      super(output_format)
      @show_start_stop = true
    end

    def encode(code)
      raise "empty string" if code.nil? || code.empty?
      s = code.upcase

      start_char = s[0]
      stop_char = s[-1]
      is_start_valid = start_char >= "A" && start_char <= "D"
      is_stop_valid = stop_char >= "A" && stop_char <= "D"

      if is_start_valid && is_stop_valid
        encoded = s
      elsif is_start_valid || is_stop_valid
        raise "invalid start/stop character combination in NW7: #{s}"
      else
        encoded = "C#{s}C"
      end

      result = []
      encoded.length.times do |i|
        result << 2 if i > 0 # inter-character gap
        char_index = NW7_CHARS.index(encoded[i])
        raise "invalid character in NW7: #{encoded[i]}" unless char_index
        pattern = NW7_PATTERNS[char_index]

        # 7-bit pattern: MSB first, 1=wide(3), 0=narrow(1)
        mask = 0x40
        7.times do |j|
          result << ((pattern & (mask >> j)) != 0 ? 3 : 1)
        end
      end
      result
    end

    def draw(code, width, height)
      draw_1d(code, width, height) { |c| encode(c) }
    end
  end
end
