# frozen_string_literal: true

module BarcodePao
  NEC2OF5_DIGIT_PATTERNS = [
    0x18, 0x03, 0x05, 0x06, 0x09, 0x0a, 0x0c, 0x11, 0x12, 0x14,
  ].freeze
  NEC2OF5_START = 0x5 # 101
  NEC2OF5_STOP  = 0x3 # 011

  class NEC2of5 < BarcodeBase1D
    def initialize(output_format = FORMAT_PNG)
      super(output_format)
    end

    def encode(code)
      raise "empty string" if code.nil? || code.empty?
      code.each_char { |c| raise "NEC 2 of 5 barcode requires numeric digits only" unless c >= "0" && c <= "9" }

      wide = @min_line_width.odd? ? 3 : 2
      narrow = 1
      result = []

      # Start code (3 bits: 101)
      chk = 0x4
      3.times do |j|
        if j.even?
          result << ((NEC2OF5_START & (chk >> j)) != 0 ? wide : narrow)
        else
          result << narrow
        end
      end
      result << narrow

      # Data encoding
      code.length.times do |i|
        digit = code[i].to_i
        pattern = NEC2OF5_DIGIT_PATTERNS[digit]
        chk = 0x10
        5.times do |j|
          if j.even?
            result << ((pattern & (chk >> j)) != 0 ? wide : narrow)
          else
            result << narrow
          end
        end
        result << narrow if i < code.length - 1
      end

      # Stop code (3 bits: 011)
      chk = 0x4
      3.times do |j|
        if j.even?
          result << ((NEC2OF5_STOP & (chk >> j)) != 0 ? wide : narrow)
        else
          result << ((NEC2OF5_STOP & (chk >> j)) != 0 ? wide : narrow)
        end
      end

      result
    end

    def draw(code, width, height)
      draw_1d(code, width, height) { |c| encode(c) }
    end
  end
end
