# frozen_string_literal: true

module BarcodePao
  # JAN/UPC extended guard bar rendering parameters and methods.
  class JANUPCParams
    attr_accessor :text_height, :normal_bar_height, :tall_bar_height,
                  :svg_font_size, :text_y, :digit_width, :left_margin,
                  :char_width, :barcode_width, :barcode_start_x,
                  :module_width, :pos_a

    def initialize
      @pos_a = 0
    end
  end

  module JANUPCRenderer
    def compute_jan_upc_params(height, tf_scale, tvo_scale, pattern, width, cls_name)
      p = JANUPCParams.new

      base_ratio = 0.2
      if height < 50
        nr = base_ratio * 1.25
        base_ratio = nr < 0.25 ? nr : 0.25
      elsif height > 200
        nr = base_ratio * 0.75
        base_ratio = nr > 0.15 ? nr : 0.15
      end

      p.text_height = (height * base_ratio * tf_scale).to_i.to_f
      p.text_height = 8.0 if p.text_height < 8

      vo_ratio = 0.05
      if height < 50
        nr = vo_ratio * 0.6
        vo_ratio = nr > 0.03 ? nr : 0.03
      elsif height > 200
        nr = vo_ratio * 1.4
        vo_ratio = nr < 0.08 ? nr : 0.08
      end

      vertical_offset_bar = (height * vo_ratio * tvo_scale).to_i.to_f
      vertical_offset_bar = 2.0 if vertical_offset_bar < 2

      p.normal_bar_height = height - p.text_height - vertical_offset_bar
      p.normal_bar_height = 1.0 if p.normal_bar_height < 1

      p.svg_font_size = (p.text_height * 0.8).to_i
      p.svg_font_size = 8 if p.svg_font_size < 8

      vertical_offset_text = (height * 0.03 * tvo_scale).to_i.to_f
      vertical_offset_text = 1.0 if vertical_offset_text < 1
      p.text_y = p.normal_bar_height + vertical_offset_text

      p.digit_width = p.svg_font_size * 0.6
      p.left_margin = p.digit_width * 1.0

      font_height = height * 0.2 * tf_scale
      p.char_width = font_height * 0.8

      long_bar_extra = if cls_name == "JAN8" || cls_name == "JAN13"
                         font_height * 0.275
                       else
                         font_height * 0.495
                       end
      p.tall_bar_height = p.normal_bar_height + long_bar_extra

      p.pos_a = 0

      total_units = pattern.sum

      case cls_name
      when "JAN8"
        p.barcode_width = width.to_f
        p.barcode_start_x = 0.0
      when "JAN13"
        left_margin_width = p.char_width * 1.2
        p.barcode_width = width - left_margin_width
        p.barcode_start_x = left_margin_width
      when "UPCA", "UPCE"
        out_margin = p.char_width * 1.0
        p.barcode_width = width - out_margin * 2
        p.barcode_start_x = out_margin
      else
        p.barcode_width = width.to_f
        p.barcode_start_x = 0.0
      end

      p.module_width = total_units > 0 ? p.barcode_width / total_units : 0.0
      p
    end

    # --- SVG JAN8 ---
    def draw_svg_jan8(pattern, display_code, width, height)
      p = compute_jan_upc_params(height, @text_font_scale, @text_vertical_offset_scale, pattern, width, "JAN8")

      svg_begin(width, height)
      svg_rect(0, 0, width.to_f, height.to_f, @background_color)

      current_x = p.barcode_start_x
      is_bar = true
      seg_idx = 0; seg_cnt = 0
      pos_b = 0.0; pos_c = 0.0; pos_d = 0.0; pos_e = 0.0

      pattern.each do |v|
        bar_w = v * p.module_width
        bar_w = 1.0 if bar_w < 1

        if seg_idx == 0 && seg_cnt >= 3
          pos_b = current_x; seg_idx += 1; seg_cnt = 0
        elsif seg_idx == 1 && seg_cnt >= 16
          pos_c = current_x; seg_idx += 1; seg_cnt = 0
        elsif seg_idx == 2 && seg_cnt >= 5
          pos_d = current_x; seg_idx += 1; seg_cnt = 0
        elsif seg_idx == 3 && seg_cnt >= 16
          pos_e = current_x; seg_idx += 1; seg_cnt = 0
        end

        is_tall = (seg_idx == 0 || seg_idx == 2 || seg_idx == 4) && is_bar
        if is_bar
          bh = is_tall ? p.tall_bar_height : p.normal_bar_height
          svg_rect(current_x, 0, bar_w, bh, @foreground_color)
        end
        seg_cnt += 1
        current_x += bar_w
        is_bar = !is_bar
      end

      mw = p.module_width
      pos_b = p.barcode_start_x + 3 * mw if pos_b == 0
      pos_c = pos_b + 28 * mw if pos_c == 0
      pos_d = pos_c + 5 * mw if pos_d == 0
      pos_e = pos_d + 28 * mw if pos_e == 0

      if display_code.length >= 8
        _draw_jan_upc_text_svg(display_code[0, 4], pos_b, pos_c, p, 4)
        _draw_jan_upc_text_svg(display_code[4, 4], pos_d, pos_e, p, 4)
      end

      svg_end
    end

    # --- SVG JAN13 ---
    def draw_svg_jan13(pattern, display_code, width, height)
      p = compute_jan_upc_params(height, @text_font_scale, @text_vertical_offset_scale, pattern, width, "JAN13")

      svg_begin(width, height)
      svg_rect(0, 0, width.to_f, height.to_f, @background_color)

      current_x = p.barcode_start_x
      is_bar = true
      seg_idx = 0; seg_cnt = 0
      pos_b = 0.0; pos_c = 0.0; pos_d = 0.0; pos_e = 0.0

      pattern.each do |v|
        bar_w = v * p.module_width
        bar_w = 1.0 if bar_w < 1

        if seg_idx == 0 && seg_cnt >= 3
          pos_b = current_x; seg_idx += 1; seg_cnt = 0
        elsif seg_idx == 1 && seg_cnt >= 24
          pos_c = current_x; seg_idx += 1; seg_cnt = 0
        elsif seg_idx == 2 && seg_cnt >= 5
          pos_d = current_x; seg_idx += 1; seg_cnt = 0
        elsif seg_idx == 3 && seg_cnt >= 24
          pos_e = current_x; seg_idx += 1; seg_cnt = 0
        end

        is_tall = (seg_idx == 0 || seg_idx == 2 || seg_idx == 4) && is_bar
        if is_bar
          bh = is_tall ? p.tall_bar_height : p.normal_bar_height
          svg_rect(current_x, 0, bar_w, bh, @foreground_color)
        end
        seg_cnt += 1
        current_x += bar_w
        is_bar = !is_bar
      end

      if display_code.length >= 13
        prefix_x = p.pos_a + p.digit_width * 0.3
        svg_text(prefix_x, p.text_y, display_code[0], p.svg_font_size, @foreground_color, "start")

        _draw_jan_upc_text_svg(display_code[1, 6], pos_b, pos_c, p, 6)
        _draw_jan_upc_text_svg(display_code[7, 6], pos_d, pos_e, p, 6)
      end

      svg_end
    end

    # --- SVG UPCA ---
    def draw_svg_upca(pattern, display_code, width, height)
      p = compute_jan_upc_params(height, @text_font_scale, @text_vertical_offset_scale, pattern, width, "UPCA")

      svg_begin(width, height)
      svg_rect(0, 0, width.to_f, height.to_f, @background_color)

      current_x = p.barcode_start_x
      is_bar = true
      seg_idx = 0; seg_cnt = 0
      pos_b = 0.0; pos_c = 0.0; pos_d = 0.0; pos_e = 0.0

      pattern.each do |v|
        bar_w = v * p.module_width
        bar_w = 1.0 if bar_w < 1

        if seg_idx == 0 && seg_cnt >= 7
          pos_b = current_x; seg_idx += 1; seg_cnt = 0
        elsif seg_idx == 1 && seg_cnt >= 21
          pos_c = current_x; seg_idx += 1; seg_cnt = 0
        elsif seg_idx == 2 && seg_cnt >= 3
          pos_d = current_x; seg_idx += 1; seg_cnt = 0
        elsif seg_idx == 3 && seg_cnt >= 21
          pos_e = current_x; seg_idx += 1; seg_cnt = 0
        end

        is_tall = (seg_idx == 0 || seg_idx == 2 || seg_idx == 4) && is_bar
        if is_bar
          bh = is_tall ? p.tall_bar_height : p.normal_bar_height
          svg_rect(current_x, 0, bar_w, bh, @foreground_color)
        end
        seg_cnt += 1
        current_x += bar_w
        is_bar = !is_bar
      end

      pos_f = current_x

      if display_code.length >= 12
        prefix_x = p.pos_a + p.digit_width * 0.3
        svg_text(prefix_x, p.text_y, display_code[0], p.svg_font_size, @foreground_color, "start")

        _draw_jan_upc_text_svg(display_code[1, 5], pos_b, pos_c, p, 5)
        _draw_jan_upc_text_svg(display_code[6, 5], pos_d, pos_e, p, 5)

        suffix_x = pos_f + p.digit_width * 0.7
        svg_text(suffix_x, p.text_y, display_code[11], p.svg_font_size, @foreground_color, "start")
      end

      svg_end
    end

    # --- SVG UPCE ---
    def draw_svg_upce(pattern, display_code, width, height)
      p = compute_jan_upc_params(height, @text_font_scale, @text_vertical_offset_scale, pattern, width, "UPCE")

      svg_begin(width, height)
      svg_rect(0, 0, width.to_f, height.to_f, @background_color)

      current_x = p.barcode_start_x
      is_bar = true
      pos_b = 0.0; pos_c = 0.0
      accumulated_units = 0

      pattern.each do |v|
        bar_w = v * p.module_width
        bar_w = 1.0 if bar_w < 1

        pos_b = current_x if pos_b == 0 && accumulated_units >= 3
        pos_c = current_x if pos_c == 0 && accumulated_units >= 45

        is_start_guard = accumulated_units < 3
        is_end_guard = accumulated_units >= 45
        is_tall = (is_start_guard || is_end_guard) && is_bar

        if is_bar
          bh = is_tall ? p.tall_bar_height : p.normal_bar_height
          svg_rect(current_x, 0, bar_w, bh, @foreground_color)
        end

        accumulated_units += v
        current_x += bar_w
        is_bar = !is_bar
      end

      pos_b = p.barcode_start_x + 3 * p.module_width if pos_b == 0
      pos_c = p.barcode_start_x + 45 * p.module_width if pos_c == 0

      pos_b += p.char_width * 0.5
      pos_c -= p.char_width * 0.5
      pos_f = current_x

      if display_code.length >= 8
        prefix_x = p.pos_a + p.digit_width * 0.3
        svg_text(prefix_x, p.text_y, display_code[0], p.svg_font_size, @foreground_color, "start")

        _draw_jan_upc_text_svg(display_code[1, 6], pos_b, pos_c, p, 6)

        suffix_x = pos_f + p.digit_width * 0.7
        svg_text(suffix_x, p.text_y, display_code[7], p.svg_font_size, @foreground_color, "start")
      end

      svg_end
    end

    # --- PNG JAN/UPC ---
    def draw_png_jan_upc(pattern, display_code, width, height, cls_name)
      p = compute_jan_upc_params(height, @text_font_scale, @text_vertical_offset_scale, pattern, width, cls_name)

      img = PNGImage.new(width, height, @background_color)

      current_x = p.barcode_start_x
      is_bar = true

      case cls_name
      when "JAN8"
        _draw_png_bars_segmented(img, pattern, p, current_x, [3, 16, 5, 16])
      when "JAN13"
        _draw_png_bars_segmented(img, pattern, p, current_x, [3, 24, 5, 24])
      when "UPCA"
        _draw_png_bars_segmented(img, pattern, p, current_x, [7, 21, 3, 21])
      when "UPCE"
        _draw_png_bars_upce(img, pattern, p, current_x)
      end

      # Draw text
      font_size = p.svg_font_size
      text_y = p.text_y.round

      case cls_name
      when "JAN8"
        if display_code.length >= 8
          _draw_jan_upc_text_png(img, display_code[0, 4], p.barcode_start_x + 3 * p.module_width, p.barcode_start_x + 31 * p.module_width, p.left_margin, text_y, font_size, 4)
          _draw_jan_upc_text_png(img, display_code[4, 4], p.barcode_start_x + 36 * p.module_width, p.barcode_start_x + 64 * p.module_width, p.left_margin, text_y, font_size, 4)
        end
      when "JAN13"
        if display_code.length >= 13
          img.draw_char_centered(display_code[0], (p.pos_a + p.digit_width * 0.3).round, text_y, font_size, @foreground_color)
          _draw_jan_upc_text_png(img, display_code[1, 6], p.barcode_start_x + 3 * p.module_width, p.barcode_start_x + 45 * p.module_width, p.left_margin, text_y, font_size, 6)
          _draw_jan_upc_text_png(img, display_code[7, 6], p.barcode_start_x + 50 * p.module_width, p.barcode_start_x + 92 * p.module_width, p.left_margin, text_y, font_size, 6)
        end
      when "UPCA"
        if display_code.length >= 12
          img.draw_char_centered(display_code[0], (p.pos_a + p.digit_width * 0.3).round, text_y, font_size, @foreground_color)
          _draw_jan_upc_text_png(img, display_code[1, 5], p.barcode_start_x + 10 * p.module_width, p.barcode_start_x + 31 * p.module_width, p.left_margin, text_y, font_size, 5)
          _draw_jan_upc_text_png(img, display_code[6, 5], p.barcode_start_x + 34 * p.module_width, p.barcode_start_x + 55 * p.module_width, p.left_margin, text_y, font_size, 5)
          end_x = p.barcode_start_x + p.barcode_width
          img.draw_char_centered(display_code[11], (end_x + p.digit_width * 0.7).round, text_y, font_size, @foreground_color)
        end
      when "UPCE"
        if display_code.length >= 8
          img.draw_char_centered(display_code[0], (p.pos_a + p.digit_width * 0.3).round, text_y, font_size, @foreground_color)
          start_b = p.barcode_start_x + 3 * p.module_width + p.char_width * 0.5
          end_c = p.barcode_start_x + 45 * p.module_width - p.char_width * 0.5
          _draw_jan_upc_text_png(img, display_code[1, 6], start_b, end_c, p.left_margin, text_y, font_size, 6)
          end_x = p.barcode_start_x + p.barcode_width
          img.draw_char_centered(display_code[7], (end_x + p.digit_width * 0.7).round, text_y, font_size, @foreground_color)
        end
      end

      if BarcodePao.trial_mode?
        img.draw_sample_overlay(0, 0, width, height)
      end

      @image_buffer = img.to_png
    end

    private

    def _draw_jan_upc_text_svg(text, start_pos, end_pos, p, count)
      section = end_pos - start_pos
      eff = section - p.left_margin * 2
      return if eff <= 0

      if @text_even_spacing && count > 1
        spacing = eff / (count - 1).to_f
        count.times do |i|
          digit_x = start_pos + p.left_margin + i * spacing
          svg_text(digit_x, p.text_y, text[i], p.svg_font_size, @foreground_color, "middle")
        end
      else
        center_x = start_pos + section / 2.0
        svg_text(center_x, p.text_y, text, p.svg_font_size, @foreground_color, "middle")
      end
    end

    def _draw_png_bars_segmented(img, pattern, p, start_x, seg_limits)
      current_x = start_x
      is_bar = true
      seg_idx = 0; seg_cnt = 0

      pattern.each do |v|
        bar_w = v * p.module_width
        bar_w = 1.0 if bar_w < 1

        if seg_idx < seg_limits.length && seg_cnt >= seg_limits[seg_idx]
          seg_idx += 1; seg_cnt = 0
        end

        is_tall = (seg_idx == 0 || seg_idx == 2 || seg_idx == 4) && is_bar
        if is_bar
          bh = is_tall ? p.tall_bar_height : p.normal_bar_height
          x1 = current_x.round
          x2 = (current_x + bar_w).round
          img.fill_rect(x1, 0, x2, bh.round, @foreground_color) if x2 > x1
        end
        seg_cnt += 1
        current_x += bar_w
        is_bar = !is_bar
      end
    end

    def _draw_png_bars_upce(img, pattern, p, start_x)
      current_x = start_x
      is_bar = true
      acc_units = 0

      pattern.each do |v|
        bar_w = v * p.module_width
        bar_w = 1.0 if bar_w < 1

        is_start_guard = acc_units < 3
        is_end_guard = acc_units >= 45
        is_tall = (is_start_guard || is_end_guard) && is_bar

        if is_bar
          bh = is_tall ? p.tall_bar_height : p.normal_bar_height
          x1 = current_x.round
          x2 = (current_x + bar_w).round
          img.fill_rect(x1, 0, x2, bh.round, @foreground_color) if x2 > x1
        end
        acc_units += v
        current_x += bar_w
        is_bar = !is_bar
      end
    end

    def _draw_jan_upc_text_png(img, text, start_pos, end_pos, margin, text_y, font_size, count)
      section = end_pos - start_pos
      eff = section - margin * 2
      return if eff <= 0 || count <= 0

      if @text_even_spacing && count > 1
        spacing = eff / (count - 1).to_f
        count.times do |i|
          break if i >= text.length
          digit_x = (start_pos + margin + i * spacing).round
          img.draw_char_centered(text[i], digit_x, text_y, font_size, @foreground_color)
        end
      else
        img.draw_text_centered(text, (start_pos + section / 2).round, text_y, font_size, @foreground_color)
      end
    end
  end

  class BarcodeBase1D
    include JANUPCRenderer
  end
end
