# frozen_string_literal: true

module BarcodePao
  JAN8_LEFT_PATTERNS  = [0x0d, 0x19, 0x13, 0x3d, 0x23, 0x31, 0x2f, 0x3b, 0x37, 0x0b].freeze
  JAN8_RIGHT_PATTERNS = [0x72, 0x66, 0x6c, 0x42, 0x5c, 0x4e, 0x50, 0x44, 0x48, 0x74].freeze

  def self.calculate_check_digit_jan8(src)
    total = 0
    7.times do |i|
      d = src[i].to_i
      total += i.even? ? d * 3 : d
    end
    ((10 - (total % 10)) % 10).to_s
  end

  class JAN8 < BarcodeBase1D
    attr_accessor :extended_guard

    def initialize(output_format = FORMAT_PNG)
      super(output_format)
      @extended_guard = true
    end

    def encode(code)
      raise "empty string" if code.nil? || code.empty?
      code.each_char { |c| raise "JAN8 barcode requires numeric digits only" unless c >= "0" && c <= "9" }

      s = code
      if s.length == 8
        expected = BarcodePao.calculate_check_digit_jan8(s[0, 7])
        raise "invalid check digit" if s[7] != expected
      elsif s.length == 7
        s = s + BarcodePao.calculate_check_digit_jan8(s)
      else
        raise "JAN8 barcode requires 7 or 8 digits"
      end

      left_half = s[0, 4]
      right_half = s[4, 4]

      result = [1, 1, 1] # Start guard

      4.times do |i|
        digit = left_half[i].to_i
        result.concat(BitsUtil.bits_to_runlength(JAN8_LEFT_PATTERNS[digit], 7))
      end

      result.concat([1, 1, 1, 1, 1]) # Center guard

      4.times do |i|
        digit = right_half[i].to_i
        result.concat(BitsUtil.bits_to_runlength(JAN8_RIGHT_PATTERNS[digit], 7))
      end

      result.concat([1, 1, 1]) # End guard
      result
    end

    def draw(code, width, height)
      pattern = encode(code)
      s = code
      s = s + BarcodePao.calculate_check_digit_jan8(s) if s.length == 7

      if svg_output?
        if @show_text && @extended_guard
          draw_svg_jan8(pattern, s, width, height)
        else
          display = @show_text ? s : ""
          draw_svg_bars(pattern, width, height, display)
        end
      else
        draw_png_jan_upc(pattern, s, width, height, "JAN8")
      end
    end
  end
end
