# frozen_string_literal: true

module BarcodePao
  JAN13_PARITY_PATTERNS = [0x00, 0x0b, 0x0d, 0x0e, 0x13, 0x19, 0x1c, 0x15, 0x16, 0x1a].freeze
  JAN13_LEFT_PATTERNS = [
    [0x0d, 0x19, 0x13, 0x3d, 0x23, 0x31, 0x2f, 0x3b, 0x37, 0x0b].freeze, # L-pattern
    [0x27, 0x33, 0x1b, 0x21, 0x1d, 0x39, 0x05, 0x11, 0x09, 0x17].freeze, # G-pattern
  ].freeze
  JAN13_RIGHT_PATTERNS = [0x72, 0x66, 0x6c, 0x42, 0x5c, 0x4e, 0x50, 0x44, 0x48, 0x74].freeze

  def self.calculate_check_digit_jan13(src)
    total = 0
    12.times do |i|
      d = src[i].to_i
      total += i.even? ? d : d * 3
    end
    ((10 - (total % 10)) % 10).to_s
  end

  class JAN13 < BarcodeBase1D
    attr_accessor :extended_guard

    def initialize(output_format = FORMAT_PNG)
      super(output_format)
      @extended_guard = true
    end

    def encode(code)
      raise "empty string" if code.nil? || code.empty?
      code.each_char { |c| raise "JAN13 barcode requires numeric digits only" unless c >= "0" && c <= "9" }

      s = code
      if s.length == 13
        expected = BarcodePao.calculate_check_digit_jan13(s[0, 12])
        raise "invalid check digit" if s[12] != expected
      elsif s.length == 12
        s = s + BarcodePao.calculate_check_digit_jan13(s)
      else
        raise "JAN13 barcode requires 12 or 13 digits"
      end

      first_digit = s[0].to_i
      left_half = s[1, 6]
      right_half = s[7, 6]

      result = [1, 1, 1] # Start guard

      chk_oe = 0x20
      6.times do |i|
        digit = left_half[i].to_i
        flg_oe = (JAN13_PARITY_PATTERNS[first_digit] & (chk_oe >> i)) != 0 ? 1 : 0
        pattern = JAN13_LEFT_PATTERNS[flg_oe][digit]
        result.concat(BitsUtil.bits_to_runlength(pattern, 7))
      end

      result.concat([1, 1, 1, 1, 1]) # Center guard

      6.times do |i|
        digit = right_half[i].to_i
        pattern = JAN13_RIGHT_PATTERNS[digit]
        result.concat(BitsUtil.bits_to_runlength(pattern, 7))
      end

      result.concat([1, 1, 1]) # End guard
      result
    end

    def draw(code, width, height)
      pattern = encode(code)
      s = code
      s = s + BarcodePao.calculate_check_digit_jan13(s) if s.length == 12

      if svg_output?
        if @show_text && @extended_guard
          draw_svg_jan13(pattern, s, width, height)
        else
          display = @show_text ? s : ""
          draw_svg_bars(pattern, width, height, display)
        end
      else
        draw_png_jan_upc(pattern, s, width, height, "JAN13")
      end
    end
  end
end
