# frozen_string_literal: true

module BarcodePao
  ITF_DIGIT_PATTERNS = [
    0x06, 0x11, 0x09, 0x18, 0x05, 0x14, 0x0c, 0x03, 0x12, 0x0a,
  ].freeze

  class ITF < BarcodeBase1D
    def initialize(output_format = FORMAT_PNG)
      super(output_format)
    end

    def encode(code)
      raise "empty string" if code.nil? || code.empty?
      code.each_char { |c| raise "ITF barcode requires numeric digits only" unless c >= "0" && c <= "9" }

      s = code
      s = "0#{s}" if s.length.odd?

      wide = @min_line_width.odd? ? 3 : 2

      result = [1, 1, 1, 1] # Start pattern

      # Encode digit pairs
      (0...s.length).step(2) do |i|
        d1 = s[i].to_i
        d2 = s[i + 1].to_i
        p1 = ITF_DIGIT_PATTERNS[d1]
        p2 = ITF_DIGIT_PATTERNS[d2]

        combined = 0x01
        5.times do |j|
          bit1 = (p1 & (0x10 >> j)) != 0 ? 1 : 0
          bit2 = (p2 & (0x10 >> j)) != 0 ? 1 : 0
          combined = (combined << 1) | bit1
          combined = (combined << 1) | bit2
        end
        combined &= 0x3FF

        10.times do |j|
          result << ((combined & (0x200 >> j)) != 0 ? wide : 1)
        end
      end

      result.push(wide, 1, 1) # Stop pattern
      result
    end

    def draw(code, width, height)
      draw_1d(code, width, height) { |c| encode(c) }
    end
  end
end
