# frozen_string_literal: true

module BarcodePao
  class GS1DataBarLimited < BarcodeBase1D
    # ISO 24724 Limited tables
    LEFT_REG_THRESHOLDS = [0, 183064, 820064, 1000776, 1491021, 1979845, 1996939].freeze
    LEFT_REG_OFFSETS    = [0, 183064, 820064, 1000776, 1491021, 1979845, 1996939].freeze

    T_EVEN_LTD       = [28, 728, 6454, 203, 2408, 1, 16632].freeze
    MODULES_ODD_LTD  = [17, 13, 9, 15, 11, 19, 7].freeze
    MODULES_EVEN_LTD = [9, 13, 17, 11, 15, 7, 19].freeze
    WIDEST_ODD_LTD   = [6, 5, 3, 5, 4, 8, 1].freeze
    WIDEST_EVEN_LTD  = [3, 4, 6, 4, 5, 1, 8].freeze

    CHECKSUM_WEIGHT_LTD = [
      1, 3, 9, 27, 81, 65, 17, 51, 64, 14,
      42, 37, 22, 66, 20, 60, 2, 6, 18, 54,
      73, 41, 34, 13, 39, 28, 84, 74
    ].freeze

    FINDER_PATTERN_LTD = [
      1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 3, 3, 1, 1,
      1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 3, 2, 1, 1,
      1, 1, 1, 1, 1, 1, 1, 1, 1, 3, 3, 1, 1, 1,
      1, 1, 1, 1, 1, 1, 1, 2, 1, 1, 3, 2, 1, 1,
      1, 1, 1, 1, 1, 1, 1, 2, 1, 2, 3, 1, 1, 1,
      1, 1, 1, 1, 1, 1, 1, 3, 1, 1, 3, 1, 1, 1,
      1, 1, 1, 1, 1, 2, 1, 1, 1, 1, 3, 2, 1, 1,
      1, 1, 1, 1, 1, 2, 1, 1, 1, 2, 3, 1, 1, 1,
      1, 1, 1, 1, 1, 2, 1, 2, 1, 1, 3, 1, 1, 1,
      1, 1, 1, 1, 1, 3, 1, 1, 1, 1, 3, 1, 1, 1,
      1, 1, 1, 2, 1, 1, 1, 1, 1, 1, 3, 2, 1, 1,
      1, 1, 1, 2, 1, 1, 1, 1, 1, 2, 3, 1, 1, 1,
      1, 1, 1, 2, 1, 1, 1, 2, 1, 1, 3, 1, 1, 1,
      1, 1, 1, 2, 1, 2, 1, 1, 1, 1, 3, 1, 1, 1,
      1, 1, 1, 3, 1, 1, 1, 1, 1, 1, 3, 1, 1, 1,
      1, 2, 1, 1, 1, 1, 1, 1, 1, 1, 3, 2, 1, 1,
      1, 2, 1, 1, 1, 1, 1, 1, 1, 2, 3, 1, 1, 1,
      1, 2, 1, 1, 1, 1, 1, 2, 1, 1, 3, 1, 1, 1,
      1, 2, 1, 1, 1, 2, 1, 1, 1, 1, 3, 1, 1, 1,
      1, 2, 1, 2, 1, 1, 1, 1, 1, 1, 3, 1, 1, 1,
      1, 3, 1, 1, 1, 1, 1, 1, 1, 1, 3, 1, 1, 1,
      1, 1, 1, 1, 1, 1, 1, 1, 2, 1, 3, 2, 1, 1,
      1, 1, 1, 1, 1, 1, 1, 1, 2, 2, 3, 1, 1, 1,
      1, 1, 1, 1, 1, 1, 1, 2, 2, 1, 3, 1, 1, 1,
      1, 1, 1, 1, 1, 2, 1, 1, 2, 1, 3, 1, 1, 1,
      1, 1, 1, 2, 1, 1, 1, 1, 2, 1, 3, 1, 1, 1,
      1, 2, 1, 1, 1, 1, 1, 1, 2, 1, 3, 1, 1, 1,
      1, 1, 1, 1, 1, 1, 2, 1, 1, 1, 3, 2, 1, 1,
      1, 1, 1, 1, 1, 1, 2, 1, 1, 2, 3, 1, 1, 1,
      1, 1, 1, 2, 1, 1, 2, 1, 1, 1, 3, 1, 1, 1,
      1, 2, 1, 1, 1, 1, 2, 1, 1, 1, 3, 1, 1, 1,
      1, 1, 1, 1, 2, 1, 1, 1, 1, 1, 3, 2, 1, 1,
      1, 1, 1, 1, 2, 1, 1, 1, 1, 2, 3, 1, 1, 1,
      1, 1, 1, 1, 2, 1, 1, 2, 1, 1, 3, 1, 1, 1,
      1, 1, 1, 1, 2, 1, 2, 1, 1, 1, 3, 1, 1, 1,
      1, 1, 1, 1, 2, 2, 1, 1, 1, 1, 3, 1, 1, 1,
      1, 2, 1, 1, 2, 1, 1, 1, 1, 1, 3, 1, 1, 1,
      1, 1, 1, 1, 1, 1, 2, 1, 2, 1, 3, 1, 1, 1,
      1, 1, 1, 1, 1, 1, 2, 2, 1, 1, 3, 1, 1, 1,
      1, 1, 1, 2, 1, 1, 2, 1, 1, 1, 3, 1, 1, 1,
      1, 1, 1, 1, 1, 2, 2, 1, 1, 1, 3, 1, 1, 1,
      1, 1, 1, 2, 2, 1, 1, 1, 1, 1, 3, 1, 1, 1,
      1, 2, 1, 1, 1, 1, 2, 1, 1, 1, 3, 1, 1, 1,
      1, 1, 2, 1, 1, 1, 1, 1, 1, 1, 3, 2, 1, 1,
      1, 1, 2, 1, 1, 1, 1, 1, 1, 2, 3, 1, 1, 1,
      1, 1, 2, 1, 1, 1, 1, 2, 1, 1, 3, 1, 1, 1,
      1, 1, 2, 1, 1, 2, 1, 1, 1, 1, 3, 1, 1, 1,
      1, 1, 2, 2, 1, 1, 1, 1, 1, 1, 3, 1, 1, 1,
      1, 2, 2, 1, 1, 1, 1, 1, 1, 1, 3, 1, 1, 1,
      1, 1, 1, 1, 2, 1, 1, 1, 2, 1, 3, 1, 1, 1,
      1, 1, 1, 1, 2, 1, 1, 2, 2, 1, 3, 1, 1, 1,
      1, 1, 1, 2, 2, 1, 1, 1, 2, 1, 3, 1, 1, 1,
      1, 1, 1, 1, 2, 2, 1, 1, 2, 1, 3, 1, 1, 1,
      1, 2, 1, 1, 2, 1, 1, 1, 2, 1, 3, 1, 1, 1,
      1, 1, 1, 1, 2, 1, 2, 1, 2, 1, 3, 1, 1, 1,
      1, 1, 1, 1, 2, 1, 2, 2, 1, 1, 3, 1, 1, 1,
      1, 1, 1, 2, 2, 1, 2, 1, 1, 1, 3, 1, 1, 1,
      1, 2, 1, 1, 2, 1, 2, 1, 1, 1, 3, 1, 1, 1,
      1, 1, 1, 1, 3, 1, 1, 1, 2, 1, 3, 1, 1, 1,
      1, 2, 1, 1, 2, 2, 1, 1, 1, 1, 3, 1, 1, 1,
      1, 2, 1, 1, 1, 1, 2, 1, 2, 1, 3, 1, 1, 1,
      1, 1, 2, 1, 1, 1, 2, 1, 1, 1, 3, 1, 1, 1,
      1, 1, 2, 1, 2, 1, 1, 1, 1, 1, 3, 1, 1, 1,
      1, 1, 2, 2, 1, 1, 1, 1, 1, 1, 3, 1, 1, 1,
      1, 2, 2, 1, 2, 1, 1, 1, 1, 1, 3, 1, 1, 1,
      1, 2, 2, 1, 1, 1, 1, 1, 2, 1, 3, 1, 1, 1,
      1, 2, 2, 1, 1, 1, 2, 1, 1, 1, 3, 1, 1, 1,
      1, 1, 2, 1, 2, 1, 2, 1, 1, 1, 3, 1, 1, 1,
      1, 1, 2, 1, 2, 2, 1, 1, 1, 1, 3, 1, 1, 1,
      1, 2, 2, 1, 2, 1, 2, 1, 1, 1, 3, 1, 1, 1,
      1, 1, 1, 1, 2, 1, 2, 1, 2, 2, 3, 1, 1, 1,
      1, 1, 1, 1, 2, 2, 2, 1, 2, 1, 3, 1, 1, 1,
      1, 1, 1, 2, 2, 1, 2, 1, 2, 1, 3, 1, 1, 1,
      1, 1, 1, 1, 2, 2, 2, 2, 1, 1, 3, 1, 1, 1,
      1, 2, 1, 1, 2, 1, 2, 1, 2, 1, 3, 1, 1, 1,
      1, 1, 1, 1, 3, 1, 2, 1, 2, 1, 3, 1, 1, 1,
      1, 2, 1, 1, 2, 2, 2, 1, 1, 1, 3, 1, 1, 1,
      1, 2, 1, 1, 3, 1, 2, 1, 1, 1, 3, 1, 1, 1,
      1, 1, 2, 1, 2, 1, 2, 1, 2, 1, 3, 1, 1, 1,
      1, 1, 2, 1, 2, 2, 2, 1, 1, 1, 3, 1, 1, 1,
      1, 1, 2, 2, 2, 1, 2, 1, 1, 1, 3, 1, 1, 1,
      1, 2, 2, 1, 2, 1, 2, 1, 2, 1, 3, 1, 1, 1,
      1, 1, 2, 2, 2, 1, 2, 2, 1, 1, 3, 1, 1, 1,
      1, 2, 2, 1, 2, 2, 2, 1, 1, 1, 3, 1, 1, 1,
      1, 2, 2, 2, 2, 1, 2, 1, 1, 1, 3, 1, 1, 1,
      1, 1, 1, 1, 3, 1, 2, 1, 2, 2, 3, 1, 1, 1,
      1, 1, 1, 1, 3, 1, 2, 2, 2, 1, 3, 1, 1, 1,
      1, 1, 1, 2, 3, 1, 2, 1, 2, 1, 3, 1, 1, 1,
      1, 1, 1, 1, 3, 2, 2, 1, 2, 1, 3, 1, 1, 1,
      1, 2, 1, 1, 3, 1, 2, 1, 2, 1, 3, 1, 1, 1
    ].freeze

    attr_reader :gtin14, :total_widths

    def initialize(output_format = FORMAT_PNG)
      super(output_format)
      @show_text = false
      @gtin14 = ""
      @total_widths = []
    end

    # Returns "(01)GTIN-14".
    def human_readable
      "(01)#{@gtin14}"
    end

    # Encode GS1 DataBar Limited.
    def encode(code)
      code.each_char do |c|
        raise "GS1 DataBar Limited requires numeric digits only" unless c >= "0" && c <= "9"
      end
      raise "input must be 8-13 digits" if code.length < 8 || code.length > 13

      padded = DataBarUtils.pad_left(code, 13)

      unless padded[0] == "0" || padded[0] == "1"
        raise "GS1 DataBar Limited requires leading digit 0 or 1 after padding"
      end

      accum = DataBarUtils.parse_int64(padded)
      left_reg = accum / 2013571
      right_reg = accum % 2013571

      left_group = determine_group_ltd(left_reg)
      right_group = determine_group_ltd(right_reg)

      left_reg -= LEFT_REG_OFFSETS[left_group] if left_group > 0
      right_reg -= LEFT_REG_OFFSETS[right_group] if right_group > 0

      left_odd = left_reg / T_EVEN_LTD[left_group]
      left_even = left_reg % T_EVEN_LTD[left_group]
      right_odd = right_reg / T_EVEN_LTD[right_group]
      right_even = right_reg % T_EVEN_LTD[right_group]

      # Generate width patterns (14 elements each)
      left_widths = Array.new(14, 0)
      right_widths = Array.new(14, 0)

      w = RSSUtils.get_widths(left_odd, MODULES_ODD_LTD[left_group], 7, WIDEST_ODD_LTD[left_group], 1)
      7.times { |i| left_widths[i * 2] = w[i] }
      w = RSSUtils.get_widths(left_even, MODULES_EVEN_LTD[left_group], 7, WIDEST_EVEN_LTD[left_group], 0)
      7.times { |i| left_widths[i * 2 + 1] = w[i] }
      w = RSSUtils.get_widths(right_odd, MODULES_ODD_LTD[right_group], 7, WIDEST_ODD_LTD[right_group], 1)
      7.times { |i| right_widths[i * 2] = w[i] }
      w = RSSUtils.get_widths(right_even, MODULES_EVEN_LTD[right_group], 7, WIDEST_EVEN_LTD[right_group], 0)
      7.times { |i| right_widths[i * 2 + 1] = w[i] }

      # Checksum (modulo 89)
      checksum = 0
      14.times do |i|
        checksum += CHECKSUM_WEIGHT_LTD[i] * left_widths[i]
        checksum += CHECKSUM_WEIGHT_LTD[i + 14] * right_widths[i]
      end
      checksum %= 89

      check_elements = FINDER_PATTERN_LTD[checksum * 14, 14]

      # Build total widths (46 elements)
      total_widths = Array.new(46, 0)
      total_widths[0] = 1
      total_widths[1] = 1
      total_widths[44] = 1
      total_widths[45] = 1

      14.times do |i|
        total_widths[i + 2] = left_widths[i]
      end
      14.times do |i|
        total_widths[i + 16] = check_elements[i]
      end
      14.times do |i|
        total_widths[i + 30] = right_widths[i]
      end

      cd = RSSUtils.calculate_gtin_check_digit(padded)
      @gtin14 = padded + cd.to_s
      @total_widths = total_widths

      total_widths
    end

    # Draw renders the GS1 DataBar Limited barcode.
    def draw(code, width, height)
      pattern = encode(code)
      # DataBar Limited is single-row, starts with space
      if svg_output?
        draw_svg_bars(pattern, width, height, "")
      else
        render_bars_to_png(pattern, width, height, "")
      end
    end

    private

    def determine_group_ltd(reg_value)
      group = 0
      group = 1 if reg_value > 183063
      group = 2 if reg_value > 820063
      group = 3 if reg_value > 1000775
      group = 4 if reg_value > 1491020
      group = 5 if reg_value > 1979844
      group = 6 if reg_value > 1996938
      group
    end
  end
end
