# frozen_string_literal: true

module BarcodePao
  # ExpandedSymbolType specifies the GS1 DataBar Expanded variant.
  UNSTACKED   = 0
  STACKED_EXP = 1

  # Encoding modes
  ENC_NUMERIC      = 0
  ENC_ALPHA        = 1
  ENC_ISO          = 2
  ENC_INVALID      = 3
  ENC_ANY_ENC      = 4
  ENC_ALPHA_OR_ISO = 5

  class GS1DataBarExpanded < BarcodeBase1D
    # ISO 24724 Tables for GS1 DataBar Expanded
    G_SUM_EXP       = [0, 348, 1388, 2948, 3988].freeze
    T_EVEN_EXP      = [4, 20, 52, 104, 204].freeze
    MODULES_ODD_EXP  = [12, 10, 8, 6, 4].freeze
    MODULES_EVEN_EXP = [5, 7, 9, 11, 13].freeze
    WIDEST_ODD_EXP   = [7, 5, 4, 3, 1].freeze
    WIDEST_EVEN_EXP  = [2, 4, 5, 6, 8].freeze

    CHECKSUM_WEIGHT_EXP = [
      1, 3, 9, 27, 81, 32, 96, 77, 20, 60, 180, 118, 143, 7, 21, 63, 189,
      145, 13, 39, 117, 140, 209, 205, 193, 157, 49, 147, 19, 57, 171, 91,
      62, 186, 136, 197, 169, 85, 44, 132, 185, 133, 188, 142, 4, 12, 36,
      108, 113, 128, 173, 97, 80, 29, 87, 50, 150, 28, 84, 41, 123, 158, 52,
      156, 46, 138, 203, 187, 139, 206, 196, 166, 76, 17, 51, 153, 37, 111,
      122, 155, 43, 129, 176, 106, 107, 110, 119, 146, 16, 48, 144, 10, 30,
      90, 59, 177, 109, 116, 137, 200, 178, 112, 125, 164, 70, 210, 208, 202,
      184, 130, 179, 115, 134, 191, 151, 31, 93, 68, 204, 190, 148, 22, 66,
      198, 172, 94, 71, 2, 6, 18, 54, 162, 64, 192, 154, 40, 120, 149, 25,
      75, 14, 42, 126, 167, 79, 26, 78, 23, 69, 207, 199, 175, 103, 98, 83,
      38, 114, 131, 182, 124, 161, 61, 183, 127, 170, 88, 53, 159, 55, 165,
      73, 8, 24, 72, 5, 15, 45, 135, 194, 160, 58, 174, 100, 89
    ].freeze

    FINDER_PATTERN_EXP = [
      1, 8, 4, 1, 1, 1, 1, 4, 8, 1, 3, 6, 4, 1, 1, 1, 1, 4, 6, 3, 3, 4, 6, 1,
      1, 1, 1, 6, 4, 3, 3, 2, 8, 1, 1, 1, 1, 8, 2, 3, 2, 6, 5, 1, 1, 1, 1, 5,
      6, 2, 2, 2, 9, 1, 1, 1, 1, 9, 2, 2
    ].freeze

    FINDER_SEQUENCE = [
      1, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 4, 3, 0, 0, 0, 0, 0, 0, 0, 0, 1, 6,
      3, 8, 0, 0, 0, 0, 0, 0, 0, 1, 10, 3, 8, 5, 0, 0, 0, 0, 0, 0, 1, 10, 3,
      8, 7, 12, 0, 0, 0, 0, 0, 1, 10, 3, 8, 9, 12, 11, 0, 0, 0, 0, 1, 2, 3,
      4, 5, 6, 7, 8, 0, 0, 0, 1, 2, 3, 4, 5, 6, 7, 10, 9, 0, 0, 1, 2, 3, 4,
      5, 6, 7, 10, 11, 12, 0, 1, 2, 3, 4, 5, 8, 7, 10, 9, 12, 11
    ].freeze

    WEIGHT_ROWS = [
      0, 1, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 6,
      3, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 9, 10, 3, 4,
      13, 14, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 17, 18, 3, 4, 13,
      14, 7, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 17, 18, 3, 4, 13, 14,
      11, 12, 21, 22, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 17, 18, 3, 4, 13, 14,
      15, 16, 21, 22, 19, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 5, 6, 7,
      8, 9, 10, 11, 12, 13, 14, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 5, 6, 7, 8,
      9, 10, 11, 12, 17, 18, 15, 16, 0, 0, 0, 0, 0, 1, 2, 3, 4, 5, 6, 7, 8,
      9, 10, 11, 12, 17, 18, 19, 20, 21, 22, 0, 0, 0, 1, 2, 3, 4, 5, 6, 7, 8,
      13, 14, 11, 12, 17, 18, 15, 16, 21, 22, 19, 20
    ].freeze

    # ISO/IEC 646 special character encoding table
    ISO_CHARS = {
      "!" => "11101000", "\"" => "11101001", "%" => "11101010",
      "&" => "11101011", "'" => "11101100", "(" => "11101101",
      ")" => "11101110", "*" => "11101111", "+" => "11110000",
      "," => "11110001", "-" => "11110010", "." => "11110011",
      "/" => "11110100", ":" => "11110101", ";" => "11110110",
      "<" => "11110111", "=" => "11111000", ">" => "11111001",
      "?" => "11111010", "_" => "11111011", " " => "11111100"
    }.freeze

    attr_accessor :symbol_type, :num_columns
    attr_reader :human_readable, :patterns, :row_heights, :row_count, :elements

    def initialize(output_format = FORMAT_PNG, symbol_type = UNSTACKED, num_columns = 2)
      super(output_format)
      @symbol_type = symbol_type
      @num_columns = num_columns
      @show_text = false
      @human_readable = ""
      @elements = []
      @patterns = []
      @row_heights = []
      @row_count = 0
      @binary_string = String.new
      @general_field = String.new
      @general_field_type = []
      @enc_method_result = 0
    end

    # Encode GS1 DataBar Expanded.
    def encode(code)
      content = code.gsub("{AI}", "")
      raise "input data is empty" if content.empty?

      @binary_string = "0".dup # Linkage flag
      calculate_binary_string(content, code)

      # Data characters from binary string
      data_chars = @binary_string.length / 12

      # Calculate vs values (12-bit segments)
      vs = Array.new(data_chars, 0)
      data_chars.times do |i|
        v = 0
        12.times do |j|
          v += (2048 >> j) if @binary_string[i * 12 + j] == "1"
        end
        vs[i] = v
      end

      # Determine groups and calculate widths
      group = Array.new(data_chars, 0)
      char_widths = Array.new(data_chars) { Array.new(8, 0) }

      data_chars.times do |i|
        if vs[i] <= 347
          group[i] = 1
        elsif vs[i] <= 1387
          group[i] = 2
        elsif vs[i] <= 2947
          group[i] = 3
        elsif vs[i] <= 3987
          group[i] = 4
        else
          group[i] = 5
        end

        gi = group[i] - 1
        vo = (vs[i] - G_SUM_EXP[gi]) / T_EVEN_EXP[gi]
        ve = (vs[i] - G_SUM_EXP[gi]) % T_EVEN_EXP[gi]

        w = RSSUtils.get_widths(vo, MODULES_ODD_EXP[gi], 4, WIDEST_ODD_EXP[gi], 0)
        char_widths[i][0] = w[0]
        char_widths[i][2] = w[1]
        char_widths[i][4] = w[2]
        char_widths[i][6] = w[3]

        w = RSSUtils.get_widths(ve, MODULES_EVEN_EXP[gi], 4, WIDEST_EVEN_EXP[gi], 1)
        char_widths[i][1] = w[0]
        char_widths[i][3] = w[1]
        char_widths[i][5] = w[2]
        char_widths[i][7] = w[3]
      end

      # Checksum (modulo 211)
      checksum = 0
      data_chars.times do |i|
        row = WEIGHT_ROWS[(((data_chars - 2) / 2) * 21) + i]
        8.times do |j|
          checksum += char_widths[i][j] * CHECKSUM_WEIGHT_EXP[row * 8 + j]
        end
      end

      check_char = 211 * ((data_chars + 1) - 4) + (checksum % 211)

      # Check character widths
      if check_char <= 347
        cg = 1
      elsif check_char <= 1387
        cg = 2
      elsif check_char <= 2947
        cg = 3
      elsif check_char <= 3987
        cg = 4
      else
        cg = 5
      end

      c_odd = (check_char - G_SUM_EXP[cg - 1]) / T_EVEN_EXP[cg - 1]
      c_even = (check_char - G_SUM_EXP[cg - 1]) % T_EVEN_EXP[cg - 1]

      check_widths = Array.new(8, 0)
      w = RSSUtils.get_widths(c_odd, MODULES_ODD_EXP[cg - 1], 4, WIDEST_ODD_EXP[cg - 1], 0)
      check_widths[0] = w[0]; check_widths[2] = w[1]
      check_widths[4] = w[2]; check_widths[6] = w[3]
      w = RSSUtils.get_widths(c_even, MODULES_EVEN_EXP[cg - 1], 4, WIDEST_EVEN_EXP[cg - 1], 1)
      check_widths[1] = w[0]; check_widths[3] = w[1]
      check_widths[5] = w[2]; check_widths[7] = w[3]

      # Build elements array
      pattern_width = ((((data_chars + 1) / 2) + ((data_chars + 1) & 1)) * 5 +
        (data_chars + 1) * 8 + 4)
      elems = Array.new(pattern_width, 0)
      elems[0] = 1
      elems[1] = 1
      elems[pattern_width - 2] = 1
      elems[pattern_width - 1] = 1

      # Finder patterns
      finder_count = (data_chars + 1) / 2 + ((data_chars + 1) & 1)
      finder_count.times do |i|
        k = ((((data_chars + 1) - 2) / 2 + ((data_chars + 1) & 1)) - 1) * 11
        k += i
        5.times do |j|
          elems[21 * i + j + 10] = FINDER_PATTERN_EXP[(FINDER_SEQUENCE[k] - 1) * 5 + j]
        end
      end

      # Check character at position 2
      8.times do |i|
        elems[i + 2] = check_widths[i]
      end

      # Odd-index data characters (forward)
      i = 1
      while i < data_chars
        8.times do |j|
          elems[((i - 1) / 2) * 21 + 23 + j] = char_widths[i][j]
        end
        i += 2
      end

      # Even-index data characters (reversed)
      i = 0
      while i < data_chars
        8.times do |j|
          elems[(i / 2) * 21 + 15 + j] = char_widths[i][7 - j]
        end
        i += 2
      end

      @elements = elems

      # Generate patterns for rendering
      generate_patterns(elems, pattern_width, data_chars)

      elems
    end

    # Draw renders the GS1 DataBar Expanded barcode.
    def draw(code, width, height)
      encode(code)
      if svg_output?
        draw_multi_row_svg(@patterns, @row_heights, width, height)
      else
        draw_multi_row_png(@patterns, @row_heights, width, height)
      end
    end

    private

    def int_to_bits(value, num_bits)
      result = String.new
      num_bits.times do |j|
        if (value & (1 << (num_bits - 1 - j))) != 0
          result << "1"
        else
          result << "0"
        end
      end
      result
    end

    def calculate_remainder(binary_string_length)
      remainder = 12 - (binary_string_length % 12)
      remainder = 0 if remainder == 12
      remainder = 36 - binary_string_length if binary_string_length < 36
      remainder
    end

    def bin2pat_from_black(binary)
      return "" if binary.nil? || binary.empty?
      pattern = RSSUtils.bin2pat(binary)
      pattern = "0" + pattern if binary[0] == "0"
      pattern
    end

    # Apply ISO 24724 general field encoding rules.
    # Returns true if last block is numeric and odd size.
    def apply_general_field_rules
      gft = @general_field_type
      n = @general_field.length
      return false if n == 0

      # Build blocks
      block_len = [1]
      block_type = [gft[0]]
      (1...n).each do |i|
        if gft[i] == gft[i - 1]
          block_len[-1] += 1
        else
          block_len << 1
          block_type << gft[i]
        end
      end
      block_count = block_len.length

      # Apply rules
      block_count.times do |i|
        cur = block_type[i]
        nxt = i < block_count - 1 ? block_type[i + 1] : -1

        if cur == ENC_ISO && i != block_count - 1
          if nxt == ENC_ANY_ENC && block_len[i + 1] >= 4
            block_type[i + 1] = ENC_NUMERIC
          end
          if nxt == ENC_ANY_ENC && block_len[i + 1] < 4
            block_type[i + 1] = ENC_ISO
          end
          if nxt == ENC_ALPHA_OR_ISO && block_len[i + 1] >= 5
            block_type[i + 1] = ENC_ALPHA
          end
          if nxt == ENC_ALPHA_OR_ISO && block_len[i + 1] < 5
            block_type[i + 1] = ENC_ISO
          end
        end

        block_type[i] = ENC_ALPHA if cur == ENC_ALPHA_OR_ISO

        if cur == ENC_ALPHA && i != block_count - 1
          if nxt == ENC_ANY_ENC && block_len[i + 1] >= 6
            block_type[i + 1] = ENC_NUMERIC
          end
          if nxt == ENC_ANY_ENC && block_len[i + 1] < 6
            if i == block_count - 2 && block_len[i + 1] >= 4
              block_type[i + 1] = ENC_NUMERIC
            else
              block_type[i + 1] = ENC_ALPHA
            end
          end
        end

        block_type[i] = ENC_NUMERIC if cur == ENC_ANY_ENC
      end

      # Merge adjacent same-type blocks
      if block_count > 1
        i = 1
        while i < block_len.length
          if block_type[i - 1] == block_type[i]
            block_len[i - 1] += block_len[i]
            block_len.delete_at(i)
            block_type.delete_at(i)
            i -= 1 if i > 1
          else
            i += 1
          end
        end
        block_count = block_len.length
      end

      # Odd-size numeric blocks (not the last)
      (0...block_count - 1).each do |i|
        if block_type[i] == ENC_NUMERIC && (block_len[i] & 1) != 0
          block_len[i] -= 1
          block_len[i + 1] += 1
        end
      end

      # Expand back to per-character types
      j = 0
      block_count.times do |i|
        block_len[i].times do
          @general_field_type[j] = block_type[i]
          j += 1
        end
      end

      block_type[block_count - 1] == ENC_NUMERIC && (block_len[block_count - 1] & 1) != 0
    end

    # Build the binary data string.
    def calculate_binary_string(source, original_code)
      ai_tag = "{AI}"
      last_mode = ENC_NUMERIC

      # Determine encoding method
      encoding_method = 2
      if source.length >= 16 && source[0] == "0" && source[1] == "1"
        encoding_method = 1
        @human_readable = "(01)" + original_code[2..]
        loop do
          p_idx = @human_readable.index(ai_tag)
          break if p_idx.nil?
          ai_start = p_idx + ai_tag.length
          @human_readable = @human_readable[0...p_idx] + "(" +
            @human_readable[ai_start, 2] + ")" +
            @human_readable[ai_start + 2..]
        end
      else
        encoding_method = 2
        @human_readable = original_code.dup
        loop do
          p_idx = @human_readable.index(ai_tag)
          break if p_idx.nil?
          ai_start = p_idx + ai_tag.length
          @human_readable = @human_readable[0...p_idx] + "(" +
            @human_readable[ai_start, 2] + ")" +
            @human_readable[ai_start + 2..]
        end
      end

      # Check for compressed methods > 2
      if source.length >= 20 && encoding_method == 1 &&
         source[2] == "9" && source[16] == "3"
        check_compressed_methods(source, encoding_method)
        encoding_method = @enc_method_result
      end

      # Method header and read position
      read_posn = apply_method_header(encoding_method, source.length)

      # Verify compressed data is numeric
      read_posn.times do |i|
        c = source[i]
        unless (c >= "0" && c <= "9") || c == "[" || c == "]"
          raise "invalid characters in input data"
        end
      end

      # Encode compressed data field
      encode_compressed(encoding_method, source)

      # General purpose data field
      @general_field = source[read_posn..]
      @general_field_type = Array.new(@general_field.length, ENC_ISO)

      if @general_field.length > 0
        last_mode = encode_general_field(last_mode)
      end

      # Length check
      raise "input too long" if @binary_string.length > 252

      # Padding
      remainder = calculate_remainder(@binary_string.length)
      padstring = String.new
      pad_i = remainder
      if @general_field.length > 0 && last_mode == ENC_NUMERIC
        padstring = "0000".dup
        pad_i = remainder - 4
      end
      while pad_i > 0
        padstring << "00100"
        pad_i -= 5
      end
      if remainder > 0 && remainder <= padstring.length
        @binary_string << padstring[0, remainder]
      elsif remainder > 0
        @binary_string << padstring
      end

      # Patch variable length symbol bit field
      bs = @binary_string
      data_char_count = bs.length / 12 + 1
      patch0 = (data_char_count & 1) != 0 ? "1" : "0"
      patch1 = bs.length > 156 ? "1" : "0"
      patch = patch0 + patch1

      case encoding_method
      when 1
        @binary_string = bs[0, 2] + patch + bs[4..]
      when 2
        @binary_string = bs[0, 3] + patch + bs[5..]
      when 5, 6
        @binary_string = bs[0, 6] + patch + bs[8..]
      end
    end

    def check_compressed_methods(source, base_method)
      @enc_method_result = base_method

      if source.length >= 26 && source[17] == "1" && source[18] == "0"
        weight = 0.0
        6.times { |i| weight = weight * 10 + (source[20 + i].ord - 48) }
        if weight < 99999.0
          if source[19] == "3" && source.length == 26
            if weight / 1000.0 <= 32.767
              @enc_method_result = 3
              @human_readable = "(01)" + source[2, 14] + "(3103)" + source[20..]
            end
          end
          if source.length == 34
            set_weight_date_method(source, weight, "310", [7, 9, 11, 13])
          end
        end
      end

      if source.length >= 26 && source[17] == "2" && source[18] == "0"
        weight = 0.0
        6.times { |i| weight = weight * 10 + (source[20 + i].ord - 48) }
        if weight < 99999.0
          if (source[19] == "2" || source[19] == "3") && source.length == 26
            if source[19] == "3"
              if weight / 1000.0 <= 22.767
                @enc_method_result = 4
                @human_readable = "(01)" + source[2, 14] + "(3203)" + source[20..]
              end
            else
              if weight / 100.0 <= 99.99
                @enc_method_result = 4
                @human_readable = "(01)" + source[2, 14] + "(3202)" + source[20..]
              end
            end
          end
          if source.length == 34
            set_weight_date_method(source, weight, "320", [8, 10, 12, 14])
          end
        end
      end

      if source[17] == "9"
        if source[18] == "2" && source[19] >= "0" && source[19] <= "3"
          @enc_method_result = 5
          @human_readable = "(01)" + source[2, 14] +
            "(392" + source[19] + ")" + source[20..]
        end
        if source[18] == "3" && source[19] >= "0" && source[19] <= "3"
          @enc_method_result = 6
          @human_readable = "(01)" + source[2, 14] +
            "(393" + source[19] + ")" + source[20..]
        end
      end
    end

    def set_weight_date_method(source, _weight, ai_prefix, methods)
      return if source.length != 34
      date_ai_map = { "1" => methods[0], "3" => methods[1], "5" => methods[2], "7" => methods[3] }
      if source[26] == "1"
        m = date_ai_map[source[27]]
        if m
          date_ais = { methods[0] => "11", methods[1] => "13", methods[2] => "15", methods[3] => "17" }
          @enc_method_result = m
          @human_readable = "(01)" + source[2, 14] +
            "(" + ai_prefix + source[19] + ")" + source[20, 6] +
            "(" + date_ais[m] + ")" + source[28..]
        end
      end
    end

    def apply_method_header(encoding_method, source_len)
      headers = {
        1 => ["1XX", 16], 2 => ["00XX", 0],
        3 => ["0100", -1], 4 => ["0101", -1],
        5 => ["01100XX", 20], 6 => ["01101XX", 23],
        7 => ["0111000", -1], 8 => ["0111001", -1],
        9 => ["0111010", -1], 10 => ["0111011", -1],
        11 => ["0111100", -1], 12 => ["0111101", -1],
        13 => ["0111110", -1], 14 => ["0111111", -1]
      }
      h = headers[encoding_method]
      @binary_string << h[0]
      h[1] < 0 ? source_len : h[1]
    end

    def encode_compressed(encoding_method, source)
      case encoding_method
      when 1
        group_val = source[2].ord - 48
        @binary_string << int_to_bits(group_val, 4)
        (1..4).each do |i|
          gv = 100 * (source[i * 3].ord - 48) + 10 * (source[i * 3 + 1].ord - 48) + (source[i * 3 + 2].ord - 48)
          @binary_string << int_to_bits(gv, 10)
        end

      when 3, 4
        (1..4).each do |i|
          gv = 100 * (source[i * 3].ord - 48) + 10 * (source[i * 3 + 1].ord - 48) + (source[i * 3 + 2].ord - 48)
          @binary_string << int_to_bits(gv, 10)
        end
        gv = 0
        6.times { |i| gv = gv * 10 + (source[20 + i].ord - 48) }
        gv += 10000 if encoding_method == 4 && source[19] == "3"
        @binary_string << int_to_bits(gv, 15)

      when 7, 8, 9, 10, 11, 12, 13, 14
        (1..4).each do |i|
          gv = 100 * (source[i * 3].ord - 48) + 10 * (source[i * 3 + 1].ord - 48) + (source[i * 3 + 2].ord - 48)
          @binary_string << int_to_bits(gv, 10)
        end
        gv = source[19].ord - 48
        5.times { |i| gv = gv * 10 + (source[21 + i].ord - 48) }
        @binary_string << int_to_bits(gv, 20)
        if source.length == 34
          gv = (((10 * (source[28].ord - 48) + (source[29].ord - 48)) * 384) +
            ((10 * (source[30].ord - 48) + (source[31].ord - 48)) - 1) * 32 +
            10 * (source[32].ord - 48) + (source[33].ord - 48))
        else
          gv = 38400
        end
        @binary_string << int_to_bits(gv, 16)

      when 5, 6
        (1..4).each do |i|
          gv = 100 * (source[i * 3].ord - 48) + 10 * (source[i * 3 + 1].ord - 48) + (source[i * 3 + 2].ord - 48)
          @binary_string << int_to_bits(gv, 10)
        end
        dec_map = { "0" => "00", "1" => "01", "2" => "10", "3" => "11" }
        @binary_string << dec_map[source[19]]
        if encoding_method == 6
          gv = 0
          3.times { |i| gv = gv * 10 + (source[20 + i].ord - 48) }
          @binary_string << int_to_bits(gv, 10)
        end
      end
    end

    def encode_general_field(last_mode)
      gf = @general_field
      gft = @general_field_type
      n = gf.length

      # Classify each character
      has_invalid = false
      n.times do |i|
        c = gf[i]
        if c.ord < 32 || c.ord > 122
          gft[i] = ENC_INVALID
          has_invalid = true
        else
          gft[i] = ENC_ISO
        end
        if c == "#" || c == "$" || c == "@" || c == "\\" || c == "^" || c == "`"
          gft[i] = ENC_INVALID
          has_invalid = true
        end
        if (c >= "A" && c <= "Z") || c == "*" || c == "," || c == "-" || c == "." || c == "/"
          gft[i] = ENC_ALPHA_OR_ISO
        end
        if (c >= "0" && c <= "9") || c == "["
          gft[i] = ENC_ANY_ENC
        end
      end

      return last_mode if has_invalid

      # Propagate ISOIEC/ALPHA_OR_ISO to following FNC1
      (0...n - 1).each do |i|
        gft[i + 1] = ENC_ISO if gft[i] == ENC_ISO && gf[i + 1] == "["
      end
      (0...n - 1).each do |i|
        gft[i + 1] = ENC_ALPHA_OR_ISO if gft[i] == ENC_ALPHA_OR_ISO && gf[i + 1] == "["
      end

      # Apply rules
      latch = apply_general_field_rules

      # Set initial mode
      if gft[0] == ENC_ALPHA
        @binary_string << "0000"
        last_mode = ENC_ALPHA
      end
      if gft[0] == ENC_ISO
        @binary_string << "0000"
        @binary_string << "00100"
        last_mode = ENC_ISO
      end

      # Encode characters
      i = 0
      loop do
        mode = gft[i]

        if mode == ENC_NUMERIC
          @binary_string << "000" if last_mode != ENC_NUMERIC
          d1 = gf[i] != "[" ? (gf[i].ord - 48) : 10
          d2 = gf[i + 1] != "[" ? (gf[i + 1].ord - 48) : 10
          value = 11 * d1 + d2 + 8
          @binary_string << int_to_bits(value, 7)
          i += 2
          last_mode = ENC_NUMERIC

        elsif mode == ENC_ALPHA
          if i != 0
            @binary_string << "0000" if last_mode == ENC_NUMERIC
            @binary_string << "00100" if last_mode == ENC_ISO
          end
          c = gf[i]
          if c >= "0" && c <= "9"
            @binary_string << int_to_bits(c.ord - 43, 5)
          end
          if c >= "A" && c <= "Z"
            @binary_string << int_to_bits(c.ord - 33, 6)
          end
          last_mode = ENC_ALPHA
          if c == "["
            @binary_string << "01111"
            last_mode = ENC_NUMERIC
          end
          @binary_string << "111010" if c == "*"
          @binary_string << "111011" if c == ","
          @binary_string << "111100" if c == "-"
          @binary_string << "111101" if c == "."
          @binary_string << "111110" if c == "/"
          i += 1

        elsif mode == ENC_ISO
          if i != 0
            if last_mode == ENC_NUMERIC
              @binary_string << "0000"
              @binary_string << "00100"
            end
            @binary_string << "00100" if last_mode == ENC_ALPHA
          end
          c = gf[i]
          if c >= "0" && c <= "9"
            @binary_string << int_to_bits(c.ord - 43, 5)
          end
          if c >= "A" && c <= "Z"
            @binary_string << int_to_bits(c.ord - 1, 7)
          end
          if c >= "a" && c <= "z"
            @binary_string << int_to_bits(c.ord - 7, 7)
          end
          last_mode = ENC_ISO
          if c == "["
            @binary_string << "01111"
            last_mode = ENC_NUMERIC
          end
          @binary_string << ISO_CHARS[c] if ISO_CHARS.key?(c)
          i += 1

        else
          i += 1
        end

        current_length = i
        current_length += 1 if latch
        break if current_length >= n
      end

      # Handle single remaining digit (latch case)
      remainder = calculate_remainder(@binary_string.length)
      if latch
        if last_mode == ENC_NUMERIC
          if remainder >= 4 && remainder <= 6
            value = (gf[i].ord - 48) + 1
            @binary_string << int_to_bits(value, 4)
          else
            d1 = gf[i].ord - 48
            d2 = 10
            value = 11 * d1 + d2 + 8
            @binary_string << int_to_bits(value, 7)
          end
        else
          value = gf[i].ord - 43
          @binary_string << int_to_bits(value, 5)
        end
      end

      last_mode
    end

    def generate_patterns(elements, pattern_width, data_chars)
      if @symbol_type == UNSTACKED
        @row_count = 1
        @row_heights = [-1]
        pattern = "0".dup
        pattern_width.times do |i|
          pattern << (elements[i] + 48).chr
        end
        @patterns = [pattern]
      else
        generate_stacked(elements, pattern_width, data_chars)
      end
    end

    def generate_stacked(elements, pattern_width, data_chars)
      codeblocks = (data_chars + 1) / 2 + ((data_chars + 1) % 2)
      blocks_per_row = @num_columns
      blocks_per_row = 2 if blocks_per_row < 1 || blocks_per_row > 10

      stack_rows = codeblocks / blocks_per_row
      stack_rows += 1 if codeblocks % blocks_per_row > 0

      row_count = stack_rows * 4 - 3
      @row_heights = Array.new(row_count, 0)
      @patterns = Array.new(row_count, "")
      symbol_row = 0
      current_block = 0

      (1..stack_rows).each do |current_row|
        sub_elements = Array.new(235, 0)
        special_case_row = false

        sub_elements[0] = 1
        sub_elements[1] = 1
        elements_in_sub = 2

        reader = 0
        left_to_right = false

        loop do
          if (blocks_per_row & 1) != 0 ||
             (current_row & 1) != 0 ||
             (current_row == stack_rows &&
               codeblocks != current_row * blocks_per_row &&
               (((current_row * blocks_per_row) - codeblocks) & 1) != 0)
            left_to_right = true
            base_i = 2 + current_block * 21
            21.times do |j|
              if base_i + j < pattern_width
                sub_elements[j + reader * 21 + 2] = elements[base_i + j]
                elements_in_sub += 1
              end
            end
          else
            left_to_right = false
            if current_row * blocks_per_row < codeblocks
              base_i = 2 + ((current_row * blocks_per_row) - reader - 1) * 21
              21.times do |j|
                if base_i + j < pattern_width
                  sub_elements[(20 - j) + reader * 21 + 2] = elements[base_i + j]
                  elements_in_sub += 1
                end
              end
            else
              k_off = (current_row * blocks_per_row) - codeblocks
              l_off = (current_row * blocks_per_row) - reader - 1
              base_i = 2 + (l_off - k_off) * 21
              21.times do |j|
                if base_i + j < pattern_width
                  sub_elements[(20 - j) + reader * 21 + 2] = elements[base_i + j]
                  elements_in_sub += 1
                end
              end
            end
          end

          reader += 1
          current_block += 1
          break if reader >= blocks_per_row || current_block >= codeblocks
        end

        # Row stop
        sub_elements[elements_in_sub] = 1
        sub_elements[elements_in_sub + 1] = 1
        elements_in_sub += 2

        # Build pattern
        @row_heights[symbol_row] = -1
        black = true

        if (current_row & 1) != 0
          @patterns[symbol_row] = "0".dup
          black = false
        else
          if current_row == stack_rows &&
             codeblocks != current_row * blocks_per_row &&
             (((current_row * blocks_per_row) - codeblocks) & 1) != 0
            special_case_row = true
            sub_elements[0] = 2
            @patterns[symbol_row] = "0".dup
            black = false
          else
            @patterns[symbol_row] = String.new
          end
        end

        # Build separator binary and pattern string
        writer = 0
        sep_bin = String.new
        elements_in_sub.times do |ei|
          @patterns[symbol_row] << (sub_elements[ei] + 48).chr
          sub_elements[ei].times do
            sep_bin << (black ? "0" : "1")
          end
          black = !black
          writer += sub_elements[ei]
        end

        # Fix separator binary
        if writer > 4
          sep_bin = "0000" + sep_bin[4..]
          sep_bin = sep_bin[0, writer] if sep_bin.length > writer
        end

        reader.times do |j|
          k = 49 * j + 18
          k = 49 * j + 19 if special_case_row
          if left_to_right
            15.times do |ii|
              idx = ii + k
              if idx - 1 >= 0 && idx < sep_bin.length &&
                 sep_bin[idx - 1] == "1" && sep_bin[idx] == "1"
                sep_bin[idx] = "0"
              end
            end
          else
            14.downto(0) do |ii|
              idx = ii + k
              if idx >= 0 && idx + 1 < sep_bin.length &&
                 sep_bin[idx + 1] == "1" && sep_bin[idx] == "1"
                sep_bin[idx] = "0"
              end
            end
          end
        end

        sep_pattern = bin2pat_from_black(sep_bin)

        # Separator rows
        if current_row != 1
          mid_sep = "05".dup
          cnt = 5
          while cnt < 49 * blocks_per_row
            mid_sep << "11"
            cnt += 2
          end
          @patterns[symbol_row - 2] = mid_sep
          @row_heights[symbol_row - 2] = 1
          @patterns[symbol_row - 1] = sep_pattern
          @row_heights[symbol_row - 1] = 1
        end

        if current_row != stack_rows
          @patterns[symbol_row + 1] = sep_pattern
          @row_heights[symbol_row + 1] = 1
        end

        symbol_row += 4
      end

      @row_count = row_count
    end
  end
end
