# frozen_string_literal: true

module BarcodePao
  # SymbolType14 specifies the GS1 DataBar 14 variant.
  OMNIDIRECTIONAL        = 0
  STACKED                = 1
  STACKED_OMNIDIRECTIONAL = 2

  class GS1DataBar14 < BarcodeBase1D
    # ISO 24724 Table 3, 4
    G_SUM_TABLE_14   = [0, 161, 961, 2015, 2715, 0, 336, 1036, 1516].freeze
    T_TABLE_14       = [1, 10, 34, 70, 126, 4, 20, 48, 81].freeze
    MODULES_ODD_14   = [12, 10, 8, 6, 4, 5, 7, 9, 11].freeze
    MODULES_EVEN_14  = [4, 6, 8, 10, 12, 10, 8, 6, 4].freeze
    WIDEST_ODD_14    = [8, 6, 4, 3, 1, 2, 4, 6, 8].freeze
    WIDEST_EVEN_14   = [1, 3, 5, 6, 8, 7, 5, 3, 1].freeze

    CHECKSUM_WEIGHT_14 = [
      1, 3, 9, 27, 2, 6, 18, 54,
      4, 12, 36, 29, 8, 24, 72, 58,
      16, 48, 65, 37, 32, 17, 51, 74,
      64, 34, 23, 69, 49, 68, 46, 59
    ].freeze

    FINDER_PATTERN_14 = [
      3, 8, 2, 1, 1,
      3, 5, 5, 1, 1,
      3, 3, 7, 1, 1,
      3, 1, 9, 1, 1,
      2, 7, 4, 1, 1,
      2, 5, 6, 1, 1,
      2, 3, 8, 1, 1,
      1, 5, 7, 1, 1,
      1, 3, 9, 1, 1
    ].freeze

    attr_accessor :symbol_type
    attr_reader :gtin14, :patterns, :row_heights, :row_count

    def initialize(output_format = FORMAT_PNG, symbol_type = OMNIDIRECTIONAL)
      super(output_format)
      @symbol_type = symbol_type
      @show_text = false
      @gtin14 = ""
      @total_widths = []
      @patterns = []
      @row_heights = []
      @row_count = 0
    end

    # Returns "(01)GTIN-14".
    def human_readable
      "(01)#{@gtin14}"
    end

    # Encode GS1 DataBar 14.
    def encode(code)
      code.each_char do |c|
        raise "GS1 DataBar requires numeric digits only" unless c >= "0" && c <= "9"
      end
      raise "input too long (max 13 digits)" if code.length > 13

      # Left-pad to 13 digits
      padded = DataBarUtils.pad_left(code, 13)

      # Convert to numeric value
      accum = DataBarUtils.parse_int64(padded)

      # Split into left_reg and right_reg
      left_reg = accum / 4537077
      right_reg = accum % 4537077

      # 4 data characters
      data_chars = [
        left_reg / 1597,
        left_reg % 1597,
        right_reg / 1597,
        right_reg % 1597
      ]

      # Data groups
      data_group = [0, 0, 0, 0]
      [0, 2].each do |idx|
        data_group[idx] = 0
        data_group[idx] = 1 if data_chars[idx] >= 161
        data_group[idx] = 2 if data_chars[idx] >= 961
        data_group[idx] = 3 if data_chars[idx] >= 2015
        data_group[idx] = 4 if data_chars[idx] >= 2715
      end
      [1, 3].each do |idx|
        data_group[idx] = 5
        data_group[idx] = 6 if data_chars[idx] >= 336
        data_group[idx] = 7 if data_chars[idx] >= 1036
        data_group[idx] = 8 if data_chars[idx] >= 1516
      end

      # Calculate odd/even values
      v_odd = [0, 0, 0, 0]
      v_even = [0, 0, 0, 0]
      v_odd[0]  = (data_chars[0] - G_SUM_TABLE_14[data_group[0]]) / T_TABLE_14[data_group[0]]
      v_even[0] = (data_chars[0] - G_SUM_TABLE_14[data_group[0]]) % T_TABLE_14[data_group[0]]
      v_odd[1]  = (data_chars[1] - G_SUM_TABLE_14[data_group[1]]) % T_TABLE_14[data_group[1]]
      v_even[1] = (data_chars[1] - G_SUM_TABLE_14[data_group[1]]) / T_TABLE_14[data_group[1]]
      v_odd[3]  = (data_chars[3] - G_SUM_TABLE_14[data_group[3]]) % T_TABLE_14[data_group[3]]
      v_even[3] = (data_chars[3] - G_SUM_TABLE_14[data_group[3]]) / T_TABLE_14[data_group[3]]
      v_odd[2]  = (data_chars[2] - G_SUM_TABLE_14[data_group[2]]) / T_TABLE_14[data_group[2]]
      v_even[2] = (data_chars[2] - G_SUM_TABLE_14[data_group[2]]) % T_TABLE_14[data_group[2]]

      # Width patterns
      data_widths = Array.new(8) { Array.new(4, 0) }
      4.times do |i|
        if i == 0 || i == 2
          w = RSSUtils.get_widths(v_odd[i], MODULES_ODD_14[data_group[i]], 4, WIDEST_ODD_14[data_group[i]], 1)
          data_widths[0][i] = w[0]
          data_widths[2][i] = w[1]
          data_widths[4][i] = w[2]
          data_widths[6][i] = w[3]
          w = RSSUtils.get_widths(v_even[i], MODULES_EVEN_14[data_group[i]], 4, WIDEST_EVEN_14[data_group[i]], 0)
          data_widths[1][i] = w[0]
          data_widths[3][i] = w[1]
          data_widths[5][i] = w[2]
          data_widths[7][i] = w[3]
        else
          w = RSSUtils.get_widths(v_odd[i], MODULES_ODD_14[data_group[i]], 4, WIDEST_ODD_14[data_group[i]], 0)
          data_widths[0][i] = w[0]
          data_widths[2][i] = w[1]
          data_widths[4][i] = w[2]
          data_widths[6][i] = w[3]
          w = RSSUtils.get_widths(v_even[i], MODULES_EVEN_14[data_group[i]], 4, WIDEST_EVEN_14[data_group[i]], 1)
          data_widths[1][i] = w[0]
          data_widths[3][i] = w[1]
          data_widths[5][i] = w[2]
          data_widths[7][i] = w[3]
        end
      end

      # Checksum (modulo 79)
      checksum = 0
      8.times do |i|
        checksum += CHECKSUM_WEIGHT_14[i] * data_widths[i][0]
        checksum += CHECKSUM_WEIGHT_14[i + 8] * data_widths[i][1]
        checksum += CHECKSUM_WEIGHT_14[i + 16] * data_widths[i][2]
        checksum += CHECKSUM_WEIGHT_14[i + 24] * data_widths[i][3]
      end
      checksum %= 79

      # Finder pattern selection
      checksum += 1 if checksum >= 8
      checksum += 1 if checksum >= 72
      c_left = checksum / 9
      c_right = checksum % 9

      # Build total widths (46 elements)
      total_widths = Array.new(46, 0)
      total_widths[0] = 1
      total_widths[1] = 1
      total_widths[44] = 1
      total_widths[45] = 1

      8.times do |i|
        total_widths[i + 2] = data_widths[i][0]
        total_widths[i + 15] = data_widths[7 - i][1]
        total_widths[i + 23] = data_widths[i][3]
        total_widths[i + 36] = data_widths[7 - i][2]
      end

      5.times do |i|
        total_widths[i + 10] = FINDER_PATTERN_14[i + 5 * c_left]
        total_widths[i + 31] = FINDER_PATTERN_14[(4 - i) + 5 * c_right]
      end

      # GTIN-14 check digit
      cd = RSSUtils.calculate_gtin_check_digit(padded)
      @gtin14 = padded + cd.to_s
      @total_widths = total_widths

      # Generate grid and patterns
      generate_grid(total_widths)

      total_widths
    end

    # Draw renders the GS1 DataBar 14 barcode.
    def draw(code, width, height)
      encode(code)
      if svg_output?
        draw_multi_row_svg(@patterns, @row_heights, width, height)
      else
        draw_multi_row_png(@patterns, @row_heights, width, height)
      end
    end

    private

    def generate_grid(total_widths)
      grid = Array.new(5) { Array.new(100, false) }
      row_count = 0
      symbol_width = 0

      case @symbol_type
      when OMNIDIRECTIONAL
        writer = 0
        latch = false
        46.times do |i|
          total_widths[i].times do
            grid[row_count][writer] = true if latch
            writer += 1
          end
          latch = !latch
        end
        symbol_width = writer if writer > symbol_width
        row_count += 1

      when STACKED
        # Upper row (row 0)
        writer = 0
        latch = false
        23.times do |i|
          total_widths[i].times do
            grid[row_count][writer] = latch
            writer += 1
          end
          latch = !latch
        end
        grid[row_count][writer] = true
        grid[row_count][writer + 1] = false

        # Lower row (row 2)
        row_count += 2
        grid[row_count][0] = true
        grid[row_count][1] = false
        writer = 0
        latch = true
        (23...46).each do |i|
          total_widths[i].times do
            grid[row_count][writer + 2] = latch
            writer += 1
          end
          latch = !latch
        end

        # Separator (row 1)
        (4...46).each do |i|
          if grid[row_count - 2][i] == grid[row_count][i]
            grid[row_count - 1][i] = true unless grid[row_count - 2][i]
          else
            grid[row_count - 1][i] = true unless grid[row_count - 1][i - 1]
          end
        end

        row_count += 1
        symbol_width = 50 if 50 > symbol_width

      when STACKED_OMNIDIRECTIONAL
        # Upper row (row 0)
        writer = 0
        latch = false
        23.times do |i|
          total_widths[i].times do
            grid[row_count][writer] = latch
            writer += 1
          end
          latch = !latch
        end
        grid[row_count][writer] = true
        grid[row_count][writer + 1] = false

        # Lower row (row 4)
        row_count += 4
        grid[row_count][0] = true
        grid[row_count][1] = false
        writer = 0
        latch = true
        (23...46).each do |i|
          total_widths[i].times do
            grid[row_count][writer + 2] = latch
            writer += 1
          end
          latch = !latch
        end

        # Middle separator (row 2)
        i = 5
        while i < 46
          grid[row_count - 2][i] = true
          i += 2
        end

        # Upper separator (row 1)
        (4...46).each do |i|
          grid[row_count - 3][i] = true unless grid[row_count - 4][i]
        end
        latch_val = true
        (17...33).each do |i|
          if !grid[row_count - 4][i]
            if latch_val
              grid[row_count - 3][i] = true
              latch_val = false
            else
              grid[row_count - 3][i] = false
              latch_val = true
            end
          else
            grid[row_count - 3][i] = false
            latch_val = true
          end
        end

        # Lower separator (row 3)
        (4...46).each do |i|
          grid[row_count - 1][i] = true unless grid[row_count][i]
        end
        latch_val = true
        (16...32).each do |i|
          if !grid[row_count][i]
            if latch_val
              grid[row_count - 1][i] = true
              latch_val = false
            else
              grid[row_count - 1][i] = false
              latch_val = true
            end
          else
            grid[row_count - 1][i] = false
            latch_val = true
          end
        end

        symbol_width = 50 if 50 > symbol_width
        row_count += 1
      end

      # Convert grid to pattern strings
      @patterns = []
      @row_heights = []

      row_count.times do |i|
        binary = String.new
        symbol_width.times do |j|
          binary << (grid[i][j] ? "1" : "0")
        end

        pattern = RSSUtils.bin2pat(binary)
        pattern = "0" + pattern if !binary.empty? && binary[0] == "0"

        @patterns << pattern
      end

      # Row heights
      case @symbol_type
      when OMNIDIRECTIONAL
        @row_heights = [-1]
      when STACKED
        @row_heights = [5, 1, 7]
      when STACKED_OMNIDIRECTIONAL
        @row_heights = [-1, 1, 1, 1, -1]
      end

      @row_count = row_count
    end
  end
end
