# frozen_string_literal: true

module BarcodePao
  # GS1-128 character sets (initialized at load time)
  GS1_128_CODE_A = (32..95).map(&:chr).join + (0..31).map(&:chr).join
  GS1_128_CODE_B = (32..126).map(&:chr).join + 127.chr
  GS1_128_CODE_C = (0..99).map { |i| format("%02d", i) }.freeze
  GS1_128_ONLY_B = "`abcdefghijklmnopqrstuvwxyz{|}~"
  GS1_128_ESC_CHARS = (0..31).map(&:chr).join
  GS1_128_START = ["211412", "211214", "211232"].freeze
  GS1_128_STOP = "2331112"

  class GS1128 < BarcodeBase1D
    def initialize(output_format = FORMAT_PNG)
      super(output_format)
      @fit_width = true
    end

    def encode(code)
      raise "empty string" if code.nil? || code.empty?

      sw_abc = determine_start_set(code)
      all_pattern = []

      # Start code
      GS1_128_START[sw_abc].each_char { |ch| all_pattern << ch.to_i }

      # Data encoding
      sw_abc2 = sw_abc
      i = 0
      while i < code.length
        chg_flg = false
        if sw_abc2 == 0 # CODE-A
          if gs1_128_is_int4(code, i)
            sw_abc2 = 2; chg_flg = true
          end
          if i < code.length && GS1_128_ONLY_B.include?(code[i])
            sw_abc2 = 1; chg_flg = true
          end
        elsif sw_abc2 == 1 # CODE-B
          if gs1_128_is_int4(code, i)
            sw_abc2 = 2; chg_flg = true
          end
          if i < code.length && GS1_128_ESC_CHARS.include?(code[i])
            sw_abc2 = 0; chg_flg = true
          end
        else # CODE-C
          c2 = safe_sub(code, i, 2)
          if c2.length < 2
            sw_abc2 = gs1_128_get_abc(code[i..]); chg_flg = true
          elsif c2 != "{F" && c2 != "{A"
            unless GS1_128_CODE_C.include?(c2)
              sw_abc2 = gs1_128_get_abc(code[i..]); chg_flg = true
            end
          end
        end

        if chg_flg
          sw_code = case sw_abc2
                    when 0 then 101
                    when 1 then 100
                    else 99
                    end
          CODE128_PATTERNS[sw_code].each_char { |ch| all_pattern << ch.to_i }
        end

        # Encode character
        idx = 0
        if i + 6 <= code.length && code[i, 6] == "{FNC1}"
          idx = 102; i += 6
        elsif i + 4 <= code.length && code[i, 4] == "{AI}"
          i += 4; next
        elsif sw_abc2 == 0
          pos = GS1_128_CODE_A.index(code[i])
          idx = pos if pos
          i += 1
        elsif sw_abc2 == 1
          pos = GS1_128_CODE_B.index(code[i])
          idx = pos if pos
          i += 1
        else
          c2 = safe_sub(code, i, 2)
          if c2.length < 2
            pos = GS1_128_CODE_B.index(code[i])
            idx = pos if pos
            i += 1
          else
            GS1_128_CODE_C.each_with_index do |cc, ci|
              if c2 == cc
                idx = ci; break
              end
            end
            i += 2
          end
        end

        CODE128_PATTERNS[idx].each_char { |ch| all_pattern << ch.to_i }
      end

      # Check digit
      cd_idx = calc_check_digit(code)
      CODE128_PATTERNS[cd_idx].each_char { |ch| all_pattern << ch.to_i }

      # Stop code
      GS1_128_STOP.each_char { |ch| all_pattern << ch.to_i }

      all_pattern
    end

    def draw(code, width, height)
      pattern = encode(code)
      display = ""
      if @show_text
        display = code.gsub("{FNC1}", "").gsub("{AI}", "")
      end
      if svg_output?
        draw_svg_bars(pattern, width, height, display)
      else
        render_bars_to_png(pattern, width, height, display)
      end
    end

    def self.calc_check_digit_mod10_w3(digits)
      total = 0
      weight3 = true
      (digits.length - 1).downto(0) do |i|
        d = digits[i].to_i
        total += weight3 ? d * 3 : d
        weight3 = !weight3
      end
      mod_val = total % 10
      mod_val == 0 ? 0 : 10 - mod_val
    end

    private

    def gs1_128_is_int4(s, pos)
      return false if pos + 4 > s.length
      (pos...pos + 4).all? { |j| s[j] >= "0" && s[j] <= "9" }
    end

    def gs1_128_get_abc(code)
      code.each_char do |c|
        return 0 if c.ord < 32
        return 1 if GS1_128_ONLY_B.include?(c)
      end
      1
    end

    def safe_sub(s, pos, length)
      return "" if pos >= s.length
      end_pos = [pos + length, s.length].min
      s[pos...end_pos]
    end

    def determine_start_set(code)
      cnt_num = 0; cnt_f = 0; cnt_ai = 0

      while cnt_num + cnt_f + cnt_ai < code.length - 1
        pos = cnt_num + cnt_f + cnt_ai
        if pos + 6 <= code.length && code[pos, 6] == "{FNC1}"
          cnt_f += 6; next
        end
        if pos + 4 <= code.length && code[pos, 4] == "{AI}"
          cnt_ai += 4; next
        end
        break if pos >= code.length
        break if code[pos] < "0" || code[pos] > "9"
        cnt_num += 1
      end

      cnt_num < 3 ? gs1_128_get_abc(code) : 2
    end

    def calc_check_digit(code)
      cnt_num = 0; cnt_f = 0; cnt_ai = 0

      while cnt_num + cnt_f + cnt_ai < code.length - 1
        pos = cnt_num + cnt_f + cnt_ai
        if pos + 6 <= code.length && code[pos, 6] == "{FNC1}"
          cnt_f += 6; next
        end
        if pos + 4 <= code.length && code[pos, 4] == "{AI}"
          cnt_ai += 4; next
        end
        break if pos >= code.length
        break if code[pos] < "0" || code[pos] > "9"
        cnt_num += 1
      end

      if cnt_num < 3
        sw_abc = gs1_128_get_abc(code)
        char_sum = sw_abc == 1 ? 104 : 103
      else
        sw_abc = 2
        char_sum = 105
      end

      cnt = 0
      i = 0
      while i < code.length
        if sw_abc == 0
          if gs1_128_is_int4(code, i)
            sw_abc = 2; cnt += 1; char_sum += 99 * cnt
          end
          if i < code.length && GS1_128_ONLY_B.include?(code[i])
            sw_abc = 1; cnt += 1; char_sum += 100 * cnt
          end
        elsif sw_abc == 1
          if gs1_128_is_int4(code, i)
            sw_abc = 2; cnt += 1; char_sum += 99 * cnt
          end
          if i < code.length && GS1_128_ESC_CHARS.include?(code[i])
            sw_abc = 0; cnt += 1; char_sum += 101 * cnt
          end
        else
          c2 = safe_sub(code, i, 2)
          if c2.length < 2
            sw_abc = gs1_128_get_abc(code[i..])
            cnt += 1; char_sum += (sw_abc == 1 ? 100 : 101) * cnt
          elsif c2 != "{F" && c2 != "{A"
            unless GS1_128_CODE_C.include?(c2)
              sw_abc = gs1_128_get_abc(code[i..])
              cnt += 1; char_sum += (sw_abc == 1 ? 100 : 101) * cnt
            end
          end
        end

        idx = -1
        if i + 6 <= code.length && code[i, 6] == "{FNC1}"
          idx = 102; i += 6
        elsif i + 4 <= code.length && code[i, 4] == "{AI}"
          idx = -1; i += 4
        elsif sw_abc == 0
          c = safe_sub(code, i, 1)
          if c != ""
            pos = GS1_128_CODE_A.index(c[0])
            idx = pos if pos
          end
          i += 1
        elsif sw_abc == 1
          c = safe_sub(code, i, 1)
          if c != ""
            pos = GS1_128_CODE_B.index(c[0])
            idx = pos if pos
          end
          i += 1
        else
          c2 = safe_sub(code, i, 2)
          if c2.length < 2
            pos = GS1_128_CODE_B.index(code[i])
            idx = pos if pos
            i += 1
          else
            GS1_128_CODE_C.each_with_index do |cc, ci|
              if c2 == cc
                idx = ci; break
              end
            end
            i += 2
          end
        end

        if idx != -1
          cnt += 1
          char_sum += idx * cnt
        end
      end

      char_sum % 103
    end
  end
end
