# frozen_string_literal: true

module BarcodePao
  class DataMatrix < BarcodeBase2D
    # --- DataMatrix code size constants ---
    DX_SZ_RECT_AUTO  = -3
    DX_SZ_AUTO       = -2
    DX_SZ_SHAPE_AUTO = -1
    DX_SZ_10x10      = 0
    DX_SZ_12x12      = 1
    DX_SZ_14x14      = 2
    DX_SZ_16x16      = 3
    DX_SZ_18x18      = 4
    DX_SZ_20x20      = 5
    DX_SZ_22x22      = 6
    DX_SZ_24x24      = 7
    DX_SZ_26x26      = 8
    DX_SZ_32x32      = 9
    DX_SZ_36x36      = 10
    DX_SZ_40x40      = 11
    DX_SZ_44x44      = 12
    DX_SZ_48x48      = 13
    DX_SZ_52x52      = 14
    DX_SZ_64x64      = 15
    DX_SZ_72x72      = 16
    DX_SZ_80x80      = 17
    DX_SZ_88x88      = 18
    DX_SZ_96x96      = 19
    DX_SZ_104x104    = 20
    DX_SZ_120x120    = 21
    DX_SZ_132x132    = 22
    DX_SZ_144x144    = 23
    DX_SZ_8x18       = 24
    DX_SZ_8x32       = 25
    DX_SZ_12x26      = 26
    DX_SZ_12x36      = 27
    DX_SZ_16x36      = 28
    DX_SZ_16x48      = 29

    # --- Encoding scheme constants ---
    DX_SCHEME_AUTO_FAST = -2
    DX_SCHEME_AUTO_BEST = -1
    DX_SCHEME_ASCII     = 0
    DX_SCHEME_C40       = 1
    DX_SCHEME_TEXT      = 2
    DX_SCHEME_X12       = 3
    DX_SCHEME_EDIFACT   = 4
    DX_SCHEME_BASE256   = 5
    DX_SCHEME_ASCII_GS1 = 6

    attr_accessor :code_size, :encode_scheme

    def initialize(output_format = FORMAT_PNG)
      super(output_format)
      @code_size = DX_SZ_AUTO
      @encode_scheme = DX_SCHEME_ASCII
    end

    def set_code_size(size)
      @code_size = size
    end

    def get_code_size
      @code_size
    end

    def set_encode_scheme(scheme)
      @encode_scheme = scheme
    end

    def get_encode_scheme
      @encode_scheme
    end

    # Draw renders the DataMatrix barcode at the given size.
    def draw(code, size)
      data = string_to_bytes(code)
      draw_bytes(data, size, size)
    end

    # Get the raw pattern as column-major bool matrix [col][row].
    def get_pattern(code)
      data = string_to_bytes(code)
      cal_data_matrix_bytes(data)
    end

    private

    # --- Internal symbol attribute indices ---
    DX_ATTR_SYMBOL_ROWS           = 0
    DX_ATTR_SYMBOL_COLS           = 1
    DX_ATTR_DATA_REGION_ROWS      = 2
    DX_ATTR_DATA_REGION_COLS      = 3
    DX_ATTR_HORIZ_DATA_REGIONS    = 4
    DX_ATTR_VERT_DATA_REGIONS     = 5
    DX_ATTR_MAPPING_MATRIX_ROWS   = 6
    DX_ATTR_MAPPING_MATRIX_COLS   = 7
    DX_ATTR_INTERLEAVED_BLOCKS    = 8
    DX_ATTR_BLOCK_ERROR_WORDS     = 9
    DX_ATTR_BLOCK_MAX_CORRECTABLE = 10
    DX_ATTR_SYMBOL_DATA_WORDS     = 11
    DX_ATTR_SYMBOL_ERROR_WORDS    = 12
    DX_ATTR_SYMBOL_MAX_CORRECTABLE = 13

    # --- Mask bits ---
    DX_MASK_BIT1 = 0x80
    DX_MASK_BIT2 = 0x40
    DX_MASK_BIT3 = 0x20
    DX_MASK_BIT4 = 0x10
    DX_MASK_BIT5 = 0x08
    DX_MASK_BIT6 = 0x04
    DX_MASK_BIT7 = 0x02
    DX_MASK_BIT8 = 0x01

    # --- Channel status constants ---
    DX_CHANNEL_VALID             = 0x00
    DX_CHANNEL_UNSUPPORTED_CHAR  = 0x01
    DX_CHANNEL_CANNOT_UNLATCH    = 0x02

    # --- Unlatch type constants ---
    DX_UNLATCH_EXPLICIT = 0
    DX_UNLATCH_IMPLICIT = 1

    # --- Module state flags ---
    DX_MODULE_OFF       = 0x00
    DX_MODULE_ON_RED    = 0x01
    DX_MODULE_ON_GREEN  = 0x02
    DX_MODULE_ON_BLUE   = 0x04
    DX_MODULE_ON_RGB    = 0x07
    DX_MODULE_ASSIGNED  = 0x10
    DX_MODULE_VISITED   = 0x20
    DX_MODULE_DATA      = 0x40

    # --- Encoding character constants ---
    DX_CHAR_ASCII_PAD         = 129
    DX_CHAR_ASCII_UPPER_SHIFT = 235
    DX_CHAR_TRIPLET_SHIFT1    = 0
    DX_CHAR_TRIPLET_SHIFT2    = 1
    DX_CHAR_TRIPLET_SHIFT3    = 2
    DX_CHAR_FNC1              = 232
    DX_CHAR_TRIPLET_UNLATCH   = 254
    DX_CHAR_EDIFACT_UNLATCH   = 31
    DX_CHAR_C40_LATCH         = 230
    DX_CHAR_TEXT_LATCH         = 239
    DX_CHAR_X12_LATCH         = 238
    DX_CHAR_EDIFACT_LATCH     = 240
    DX_CHAR_BASE256_LATCH     = 231

    DX_SZ_SQUARE_COUNT = 24
    DX_SZ_RECT_COUNT   = 6
    DX_UNDEFINED        = -1

    # --- Symbol attribute tables (30 entries: 24 square + 6 rectangular) ---
    DX_SYMBOL_ROWS = [
      10, 12, 14, 16, 18, 20, 22, 24, 26,
      32, 36, 40, 44, 48, 52,
      64, 72, 80, 88, 96, 104,
      120, 132, 144,
      8, 8, 12, 12, 16, 16
    ].freeze

    DX_SYMBOL_COLS = [
      10, 12, 14, 16, 18, 20, 22, 24, 26,
      32, 36, 40, 44, 48, 52,
      64, 72, 80, 88, 96, 104,
      120, 132, 144,
      18, 32, 26, 36, 36, 48
    ].freeze

    DX_DATA_REGION_ROWS = [
      8, 10, 12, 14, 16, 18, 20, 22, 24,
      14, 16, 18, 20, 22, 24,
      14, 16, 18, 20, 22, 24,
      18, 20, 22,
      6, 6, 10, 10, 14, 14
    ].freeze

    DX_DATA_REGION_COLS = [
      8, 10, 12, 14, 16, 18, 20, 22, 24,
      14, 16, 18, 20, 22, 24,
      14, 16, 18, 20, 22, 24,
      18, 20, 22,
      16, 14, 24, 16, 16, 22
    ].freeze

    DX_HORIZ_DATA_REGIONS = [
      1, 1, 1, 1, 1, 1, 1, 1, 1,
      2, 2, 2, 2, 2, 2,
      4, 4, 4, 4, 4, 4,
      6, 6, 6,
      1, 2, 1, 2, 2, 2
    ].freeze

    DX_INTERLEAVED_BLOCKS = [
      1, 1, 1, 1, 1, 1, 1, 1, 1,
      1, 1, 1, 1, 1, 2,
      2, 4, 4, 4, 4, 6,
      6, 8, 10,
      1, 1, 1, 1, 1, 1
    ].freeze

    DX_SYMBOL_DATA_WORDS = [
      3, 5, 8, 12, 18, 22, 30, 36, 44,
      62, 86, 114, 144, 174, 204,
      280, 368, 456, 576, 696, 816,
      1050, 1304, 1558,
      5, 10, 16, 22, 32, 49
    ].freeze

    DX_BLOCK_ERROR_WORDS = [
      5, 7, 10, 12, 14, 18, 20, 24, 28,
      36, 42, 48, 56, 68, 42,
      56, 36, 48, 56, 68, 56,
      68, 62, 62,
      7, 11, 14, 18, 24, 28
    ].freeze

    DX_BLOCK_MAX_CORRECTABLE = [
      2, 3, 5, 6, 7, 9, 10, 12, 14,
      18, 21, 24, 28, 34, 21,
      28, 18, 24, 28, 34, 28,
      34, 31, 31,
      3, 5, 7, 9, 12, 14
    ].freeze

    # --- GF(256) antilog table ---
    DX_ALOG_VAL = [
      1, 2, 4, 8, 16, 32, 64, 128, 45, 90, 180, 69, 138, 57, 114, 228,
      229, 231, 227, 235, 251, 219, 155, 27, 54, 108, 216, 157, 23, 46, 92, 184,
      93, 186, 89, 178, 73, 146, 9, 18, 36, 72, 144, 13, 26, 52, 104, 208,
      141, 55, 110, 220, 149, 7, 14, 28, 56, 112, 224, 237, 247, 195, 171, 123,
      246, 193, 175, 115, 230, 225, 239, 243, 203, 187, 91, 182, 65, 130, 41, 82,
      164, 101, 202, 185, 95, 190, 81, 162, 105, 210, 137, 63, 126, 252, 213, 135,
      35, 70, 140, 53, 106, 212, 133, 39, 78, 156, 21, 42, 84, 168, 125, 250,
      217, 159, 19, 38, 76, 152, 29, 58, 116, 232, 253, 215, 131, 43, 86, 172,
      117, 234, 249, 223, 147, 11, 22, 44, 88, 176, 77, 154, 25, 50, 100, 200,
      189, 87, 174, 113, 226, 233, 255, 211, 139, 59, 118, 236, 245, 199, 163, 107,
      214, 129, 47, 94, 188, 85, 170, 121, 242, 201, 191, 83, 166, 97, 194, 169,
      127, 254, 209, 143, 51, 102, 204, 181, 71, 142, 49, 98, 196, 165, 103, 206,
      177, 79, 158, 17, 34, 68, 136, 61, 122, 244, 197, 167, 99, 198, 161, 111,
      222, 145, 15, 30, 60, 120, 240, 205, 183, 67, 134, 33, 66, 132, 37, 74,
      148, 5, 10, 20, 40, 80, 160, 109, 218, 153, 31, 62, 124, 248, 221, 151,
      3, 6, 12, 24, 48, 96, 192, 173, 119, 238, 241, 207, 179, 75, 150, 1
    ].freeze

    # --- GF(256) log table ---
    DX_LOG_VAL = [
      -255, 255, 1, 240, 2, 225, 241, 53, 3, 38, 226, 133, 242, 43, 54, 210,
      4, 195, 39, 114, 227, 106, 134, 28, 243, 140, 44, 23, 55, 118, 211, 234,
      5, 219, 196, 96, 40, 222, 115, 103, 228, 78, 107, 125, 135, 8, 29, 162,
      244, 186, 141, 180, 45, 99, 24, 49, 56, 13, 119, 153, 212, 199, 235, 91,
      6, 76, 220, 217, 197, 11, 97, 184, 41, 36, 223, 253, 116, 138, 104, 193,
      229, 86, 79, 171, 108, 165, 126, 145, 136, 34, 9, 74, 30, 32, 163, 84,
      245, 173, 187, 204, 142, 81, 181, 190, 46, 88, 100, 159, 25, 231, 50, 207,
      57, 147, 14, 67, 120, 128, 154, 248, 213, 167, 200, 63, 236, 110, 92, 176,
      7, 161, 77, 124, 221, 102, 218, 95, 198, 90, 12, 152, 98, 48, 185, 179,
      42, 209, 37, 132, 224, 52, 254, 239, 117, 233, 139, 22, 105, 27, 194, 113,
      230, 206, 87, 158, 80, 189, 172, 203, 109, 175, 166, 62, 127, 247, 146, 66,
      137, 192, 35, 252, 10, 183, 75, 216, 31, 83, 33, 73, 164, 144, 85, 170,
      246, 65, 174, 61, 188, 202, 205, 157, 143, 169, 82, 72, 182, 215, 191, 251,
      47, 178, 89, 151, 101, 94, 160, 123, 26, 112, 232, 21, 51, 238, 208, 131,
      58, 69, 148, 18, 15, 16, 68, 17, 121, 149, 129, 19, 155, 59, 249, 70,
      214, 250, 168, 71, 201, 156, 64, 60, 237, 130, 111, 20, 93, 122, 177, 150
    ].freeze

    # ================================================================
    # GF(256) helper methods
    # ================================================================

    def dx_gf_product(a, b)
      return 0 if a == 0 || b == 0
      DX_ALOG_VAL[(DX_LOG_VAL[a] + DX_LOG_VAL[b]) % 255]
    end

    def dx_gf_sum(a, b)
      a ^ b
    end

    def dx_gf_doublify(a, b)
      return 0 if a == 0
      return a if b == 0
      DX_ALOG_VAL[(DX_LOG_VAL[a] + b) % 255]
    end

    # ================================================================
    # Symbol attribute lookup
    # ================================================================

    def dx_get_symbol_attribute(attribute, size_idx)
      return DX_UNDEFINED if size_idx < 0 || size_idx >= DX_SZ_SQUARE_COUNT + DX_SZ_RECT_COUNT

      case attribute
      when DX_ATTR_SYMBOL_ROWS
        DX_SYMBOL_ROWS[size_idx]
      when DX_ATTR_SYMBOL_COLS
        DX_SYMBOL_COLS[size_idx]
      when DX_ATTR_DATA_REGION_ROWS
        DX_DATA_REGION_ROWS[size_idx]
      when DX_ATTR_DATA_REGION_COLS
        DX_DATA_REGION_COLS[size_idx]
      when DX_ATTR_HORIZ_DATA_REGIONS
        DX_HORIZ_DATA_REGIONS[size_idx]
      when DX_ATTR_VERT_DATA_REGIONS
        if size_idx < DX_SZ_SQUARE_COUNT
          DX_HORIZ_DATA_REGIONS[size_idx]
        else
          1
        end
      when DX_ATTR_MAPPING_MATRIX_ROWS
        DX_DATA_REGION_ROWS[size_idx] * dx_get_symbol_attribute(DX_ATTR_VERT_DATA_REGIONS, size_idx)
      when DX_ATTR_MAPPING_MATRIX_COLS
        DX_DATA_REGION_COLS[size_idx] * DX_HORIZ_DATA_REGIONS[size_idx]
      when DX_ATTR_INTERLEAVED_BLOCKS
        DX_INTERLEAVED_BLOCKS[size_idx]
      when DX_ATTR_BLOCK_ERROR_WORDS
        DX_BLOCK_ERROR_WORDS[size_idx]
      when DX_ATTR_BLOCK_MAX_CORRECTABLE
        DX_BLOCK_MAX_CORRECTABLE[size_idx]
      when DX_ATTR_SYMBOL_DATA_WORDS
        DX_SYMBOL_DATA_WORDS[size_idx]
      when DX_ATTR_SYMBOL_ERROR_WORDS
        DX_BLOCK_ERROR_WORDS[size_idx] * DX_INTERLEAVED_BLOCKS[size_idx]
      when DX_ATTR_SYMBOL_MAX_CORRECTABLE
        DX_BLOCK_MAX_CORRECTABLE[size_idx] * DX_INTERLEAVED_BLOCKS[size_idx]
      else
        DX_UNDEFINED
      end
    end

    def dx_get_block_data_size(size_idx, block_idx)
      sdw = dx_get_symbol_attribute(DX_ATTR_SYMBOL_DATA_WORDS, size_idx)
      ilb = dx_get_symbol_attribute(DX_ATTR_INTERLEAVED_BLOCKS, size_idx)
      return DX_UNDEFINED if sdw < 1 || ilb < 1
      count = sdw / ilb
      if size_idx == DX_SZ_144x144 && block_idx < 8
        return count + 1
      end
      count
    end

    # ================================================================
    # Reed-Solomon ECC generation
    # ================================================================

    def dx_gen_reed_sol_ecc(code, size_idx)
      sdw = dx_get_symbol_attribute(DX_ATTR_SYMBOL_DATA_WORDS, size_idx)
      sew = dx_get_symbol_attribute(DX_ATTR_SYMBOL_ERROR_WORDS, size_idx)
      stw = sdw + sew
      bew = dx_get_symbol_attribute(DX_ATTR_BLOCK_ERROR_WORDS, size_idx)
      step = dx_get_symbol_attribute(DX_ATTR_INTERLEAVED_BLOCKS, size_idx)

      return if bew != sew / step

      g = Array.new(69, 1)

      1.upto(bew) do |i|
        (i - 1).downto(0) do |j|
          g[j] = dx_gf_doublify(g[j], i)
          g[j] = dx_gf_sum(g[j], g[j - 1]) if j > 0
        end
      end

      step.times do |block|
        b = Array.new(68, 0)

        i = block
        while i < sdw
          val = dx_gf_sum(b[bew - 1], code[i])
          (bew - 1).downto(1) do |j|
            b[j] = dx_gf_sum(b[j - 1], dx_gf_product(g[j], val))
          end
          b[0] = dx_gf_product(g[0], val)
          i += step
        end

        block_data_words = dx_get_block_data_size(size_idx, block)
        b_index = bew
        i = block + (step * block_data_words)
        while i < stw
          b_index -= 1
          code[i] = b[b_index]
          i += step
        end
      end
    end

    # ================================================================
    # Randomize helpers
    # ================================================================

    def dx_randomize_253(codeword_value, codeword_position)
      pseudo_random = ((149 * codeword_position) % 253) + 1
      tmp = codeword_value + pseudo_random
      tmp -= 254 if tmp > 254
      tmp & 0xFF
    end

    def dx_randomize_255(codeword_value, codeword_position)
      pseudo_random = ((149 * codeword_position) % 255) + 1
      tmp = codeword_value + pseudo_random
      if tmp <= 255
        tmp & 0xFF
      else
        (tmp - 256) & 0xFF
      end
    end

    def dx_un_randomize_255(value, idx)
      pseudo_random = ((149 * idx) % 255) + 1
      tmp = value - pseudo_random
      tmp += 256 if tmp < 0
      tmp & 0xFF
    end

    # ================================================================
    # Triplet/Quadruplet helpers
    # ================================================================

    def dx_get_triplet_values(cw1, cw2)
      compact = (cw1 << 8) | cw2
      [
        (compact - 1) / 1600,
        ((compact - 1) / 40) % 40,
        (compact - 1) % 40
      ]
    end

    def dx_get_quadruplet_values(cw1, cw2, cw3)
      [
        cw1 >> 2,
        ((cw1 & 0x03) << 4) | ((cw2 & 0xf0) >> 4),
        ((cw2 & 0x0f) << 2) | ((cw3 & 0xc0) >> 6),
        cw3 & 0x3f
      ]
    end

    # ================================================================
    # DxMessage - internal message structure
    # ================================================================

    class DxMessage
      attr_accessor :array, :code, :pad_count

      def initialize(dm_instance, size_idx)
        mr = dm_instance.send(:dx_get_symbol_attribute, DX_ATTR_MAPPING_MATRIX_ROWS, size_idx)
        mc = dm_instance.send(:dx_get_symbol_attribute, DX_ATTR_MAPPING_MATRIX_COLS, size_idx)
        cs = dm_instance.send(:dx_get_symbol_attribute, DX_ATTR_SYMBOL_DATA_WORDS, size_idx) +
             dm_instance.send(:dx_get_symbol_attribute, DX_ATTR_SYMBOL_ERROR_WORDS, size_idx)
        @array = Array.new(mc * mr, 0)
        @code = Array.new(cs, 0)
        @pad_count = 0
      end

      def symbol_module_status(dm_instance, size_idx, symbol_row, symbol_col)
        drr = dm_instance.send(:dx_get_symbol_attribute, DX_ATTR_DATA_REGION_ROWS, size_idx)
        drc = dm_instance.send(:dx_get_symbol_attribute, DX_ATTR_DATA_REGION_COLS, size_idx)
        sr = dm_instance.send(:dx_get_symbol_attribute, DX_ATTR_SYMBOL_ROWS, size_idx)
        mc = dm_instance.send(:dx_get_symbol_attribute, DX_ATTR_MAPPING_MATRIX_COLS, size_idx)

        srr = sr - symbol_row - 1
        mapping_row = srr - 1 - 2 * (srr / (drr + 2))
        mapping_col = symbol_col - 1 - 2 * (symbol_col / (drc + 2))

        # Solid finder pattern (L-shape: bottom and left edges)
        if (symbol_row % (drr + 2)) == 0 || (symbol_col % (drc + 2)) == 0
          return DX_MODULE_ON_RGB
        end

        # Alternating clock pattern (top edge)
        if ((symbol_row + 1) % (drr + 2)) == 0
          return (symbol_col & 0x01) != 0 ? 0 : DX_MODULE_ON_RGB
        end

        # Alternating clock pattern (right edge)
        if ((symbol_col + 1) % (drc + 2)) == 0
          return (symbol_row & 0x01) != 0 ? 0 : DX_MODULE_ON_RGB
        end

        # Data module
        @array[mapping_row * mc + mapping_col] | DX_MODULE_DATA
      end
    end

    # ================================================================
    # DxChannel - encoding channel
    # ================================================================

    class DxChannel
      attr_accessor :enc_scheme, :invalid, :input_index, :encoded_length,
                    :current_length, :first_code_word, :input, :encoded_words

      def initialize
        @enc_scheme = DX_SCHEME_ASCII
        @invalid = DX_CHANNEL_VALID
        @input_index = 0
        @encoded_length = 0
        @current_length = 0
        @first_code_word = 0
        @input = nil
        @encoded_words = Array.new(1558, 0)
      end

      def deep_copy
        nc = DxChannel.new
        nc.enc_scheme = @enc_scheme
        nc.invalid = @invalid
        nc.input_index = @input_index
        nc.encoded_length = @encoded_length
        nc.current_length = @current_length
        nc.first_code_word = @first_code_word
        nc.input = @input  # shared read-only
        nc.encoded_words = @encoded_words.dup
        nc
      end
    end

    # ================================================================
    # DxEncode - main encoding engine
    # ================================================================

    class DxEncode
      attr_accessor :scheme, :size_idx_req, :margin_size, :module_size,
                    :flg_gs1, :message, :raw_data, :size_idx,
                    :symbol_rows, :symbol_cols, :mapping_rows, :mapping_cols

      def initialize(dm_instance)
        @dm = dm_instance
        @scheme = DX_SCHEME_ASCII
        @size_idx_req = DX_SZ_AUTO
        @margin_size = 10
        @module_size = 5
        @flg_gs1 = false
        @message = nil
        @raw_data = nil
        @size_idx = 0
        @symbol_rows = 0
        @symbol_cols = 0
        @mapping_rows = 0
        @mapping_cols = 0
      end

      def encode_data_matrix_raw(input_bytes)
        buf = Array.new(4096, 0)
        size_idx = @size_idx_req

        data_word_count, size_idx = encode_data_codewords(buf, input_bytes, size_idx)
        if data_word_count <= 0
          raise "failed to encode data: data may be empty or contain invalid characters"
        end
        if size_idx == DX_SZ_AUTO || size_idx == DX_SZ_RECT_AUTO
          raise "data too long for specified symbol size"
        end

        padded_size = @dm.send(:dx_get_symbol_attribute, DX_ATTR_SYMBOL_DATA_WORDS, size_idx)
        pad_count = add_pad_chars(buf, data_word_count, padded_size)

        @size_idx = size_idx
        @symbol_rows = @dm.send(:dx_get_symbol_attribute, DX_ATTR_SYMBOL_ROWS, size_idx)
        @symbol_cols = @dm.send(:dx_get_symbol_attribute, DX_ATTR_SYMBOL_COLS, size_idx)
        @mapping_rows = @dm.send(:dx_get_symbol_attribute, DX_ATTR_MAPPING_MATRIX_ROWS, size_idx)
        @mapping_cols = @dm.send(:dx_get_symbol_attribute, DX_ATTR_MAPPING_MATRIX_COLS, size_idx)

        @message = DxMessage.new(@dm, size_idx)
        @message.pad_count = pad_count

        padded_size.times do |i|
          @message.code[i] = buf[i]
        end

        @dm.send(:dx_gen_reed_sol_ecc, @message.code, size_idx)
        module_placement_ecc200(@message.array, @message.code, size_idx, DX_MODULE_ON_RGB)
        print_pattern_raw
      end

      private

      # --- Encoding schemes ---

      def encode_data_codewords(buf, input_bytes, size_idx)
        case @scheme
        when DX_SCHEME_AUTO_BEST
          dwc = encode_auto_best(buf, input_bytes)
        when DX_SCHEME_AUTO_FAST
          dwc = 0
        else
          dwc = encode_single_scheme(buf, input_bytes, @scheme)
        end

        size_idx = find_correct_symbol_size(dwc, size_idx)
        return [0, size_idx] if size_idx == DX_SZ_SHAPE_AUTO
        [dwc, size_idx]
      end

      def find_correct_symbol_size(data_words, size_idx_request)
        return DX_SZ_SHAPE_AUTO if data_words <= 0

        size_idx = size_idx_request

        if size_idx_request == DX_SZ_AUTO || size_idx_request == DX_SZ_RECT_AUTO
          if size_idx_request == DX_SZ_AUTO
            idx_beg = 0
            idx_end = DX_SZ_SQUARE_COUNT
          else
            idx_beg = DX_SZ_SQUARE_COUNT
            idx_end = DX_SZ_SQUARE_COUNT + DX_SZ_RECT_COUNT
          end

          size_idx = idx_beg
          (idx_beg...idx_end).each do |i|
            size_idx = i
            break if @dm.send(:dx_get_symbol_attribute, DX_ATTR_SYMBOL_DATA_WORDS, i) >= data_words
          end

          return DX_SZ_SHAPE_AUTO if size_idx == idx_end
        end

        return DX_SZ_SHAPE_AUTO if data_words > @dm.send(:dx_get_symbol_attribute, DX_ATTR_SYMBOL_DATA_WORDS, size_idx)

        size_idx
      end

      def encode_single_scheme(buf, codewords, scheme)
        channel = DxChannel.new
        init_channel(channel, codewords)

        while channel.input_index < channel.input.length
          ok = encode_next_word(channel, scheme)
          return 0 unless ok
          return 0 if channel.invalid != DX_CHANNEL_VALID
        end

        size = channel.encoded_length / 12
        size.times do |i|
          buf[i] = channel.encoded_words[i]
        end
        size
      end

      def encode_auto_best(buf, codewords)
        channels = Array.new(6)

        (DX_SCHEME_ASCII..DX_SCHEME_BASE256).each do |ts|
          channels[ts] = DxChannel.new
          init_channel(channels[ts], codewords)
          ok = encode_next_word(channels[ts], ts)
          # C++ bug: "if (err) return 0;" - returns 0 when encoding succeeds
          return 0 if ok
        end

        # Encode remaining (unreachable due to above bug, ported for faithfulness)
        while channels[0].input_index < channels[0].input.length
          best = Array.new(6)
          (DX_SCHEME_ASCII..DX_SCHEME_BASE256).each do |ts|
            best[ts] = find_best_channel(channels, ts)
          end
          channels = best
        end

        winner = channels[DX_SCHEME_ASCII]
        (DX_SCHEME_ASCII + 1..DX_SCHEME_BASE256).each do |ts|
          next if channels[ts].nil?
          next if channels[ts].invalid != DX_CHANNEL_VALID
          winner = channels[ts] if channels[ts].encoded_length < winner.encoded_length
        end

        ws = winner.encoded_length / 12
        ws.times do |i|
          buf[i] = winner.encoded_words[i]
        end
        ws
      end

      def find_best_channel(channels, target_scheme)
        winner = nil
        (DX_SCHEME_ASCII..DX_SCHEME_BASE256).each do |s|
          channel = channels[s]
          next if channel.nil? || channel.invalid != DX_CHANNEL_VALID
          next if channel.input_index == channel.input.length
          encode_next_word(channel, target_scheme)
          if (channel.invalid & DX_CHANNEL_UNSUPPORTED_CHAR) != 0
            return channel.deep_copy
          end
          next if (channel.invalid & DX_CHANNEL_CANNOT_UNLATCH) != 0
          if winner.nil? || channel.current_length < winner.current_length
            winner = channel.deep_copy
          end
        end
        return winner if winner
        DxChannel.new
      end

      def encode_next_word(channel, target_scheme)
        if channel.enc_scheme != target_scheme
          change_enc_scheme(channel, target_scheme, DX_UNLATCH_EXPLICIT)
          return false if channel.invalid != DX_CHANNEL_VALID
        end

        return false if channel.enc_scheme != target_scheme

        case channel.enc_scheme
        when DX_SCHEME_ASCII
          encode_ascii_codeword(channel)
        when DX_SCHEME_C40, DX_SCHEME_TEXT, DX_SCHEME_X12
          encode_triplet_codeword(channel)
        when DX_SCHEME_EDIFACT
          encode_edifact_codeword(channel)
        when DX_SCHEME_BASE256
          encode_base256_codeword(channel)
        else
          false
        end
      end

      # --- ASCII encoding ---

      def encode_ascii_codeword(channel)
        input_value = channel.input[channel.input_index]

        # Digit pair compression
        if digit_byte?(input_value) && channel.current_length >= channel.first_code_word + 12
          prev_index = (channel.current_length - 12) / 12
          prev_value = (channel.encoded_words[prev_index] - 1) & 0xFF
          prev_prev_value = 0
          if prev_index > channel.first_code_word / 12
            prev_prev_value = channel.encoded_words[prev_index - 1]
          end

          if prev_prev_value != 235 && digit_byte?(prev_value)
            channel.encoded_words[prev_index] = (10 * (prev_value - 48) + (input_value - 48) + 130) & 0xFF
            channel.input_index += 1
            return true
          end
        end

        # FNC1 in GS1 mode
        if @flg_gs1 && input_value == DX_CHAR_FNC1
          push_input_word(channel, DX_CHAR_FNC1)
          increment_progress(channel, 12)
          channel.input_index += 1
          return true
        end

        # Upper shift for values >= 128
        if input_value >= 128
          push_input_word(channel, DX_CHAR_ASCII_UPPER_SHIFT)
          increment_progress(channel, 12)
          input_value -= 128
        end

        push_input_word(channel, (input_value + 1) & 0xFF)
        increment_progress(channel, 12)
        channel.input_index += 1
        true
      end

      def digit_byte?(v)
        v >= 48 && v <= 57
      end

      # --- Triplet encoding (C40/Text/X12) ---

      def encode_triplet_codeword(channel)
        unless [DX_SCHEME_C40, DX_SCHEME_TEXT, DX_SCHEME_X12].include?(channel.enc_scheme)
          return false
        end

        return false if channel.current_length > channel.encoded_length

        if channel.current_length == channel.encoded_length
          return false if (channel.current_length % 12) != 0

          ptr_index = channel.input_index
          triplet_count = 0
          buffer = Array.new(6, 0)

          loop do
            while triplet_count < 3 && ptr_index < channel.input.length
              input_word = channel.input[ptr_index]
              ptr_index += 1
              output_words = get_c40_text_x12_words(input_word, channel.enc_scheme)
              if output_words.nil? || output_words.empty?
                channel.invalid = DX_CHANNEL_UNSUPPORTED_CHAR
                return false
              end
              output_words.each do |w|
                buffer[triplet_count] = w & 0xFF
                triplet_count += 1
              end
            end

            triplet = [buffer[0], buffer[1], buffer[2]]

            if triplet_count >= 3
              push_triplet(channel, triplet)
              buffer[0] = buffer[3]
              buffer[1] = buffer[4]
              buffer[2] = buffer[5]
              triplet_count -= 3
            end

            if ptr_index == channel.input.length
              while channel.current_length < channel.encoded_length
                increment_progress(channel, 8)
                channel.input_index += 1
              end

              if channel.current_length == channel.encoded_length + 8
                channel.current_length = channel.encoded_length
                channel.input_index -= 1
              end

              input_count = channel.input.length - channel.input_index
              ok = process_end_of_symbol_triplet(channel, triplet, triplet_count, input_count)
              return false unless ok
              break
            end

            break if triplet_count == 0
          end
        end

        if channel.current_length < channel.encoded_length
          increment_progress(channel, 8)
          channel.input_index += 1
        end

        true
      end

      def get_c40_text_x12_words(input_word, enc_scheme)
        result = []

        if input_word > 127
          return nil if enc_scheme == DX_SCHEME_X12
          result << DX_CHAR_TRIPLET_SHIFT2
          result << 30
          input_word -= 128
        end

        if enc_scheme == DX_SCHEME_X12
          if input_word == 13
            result << 0
          elsif input_word == 42
            result << 1
          elsif input_word == 62
            result << 2
          elsif input_word == 32
            result << 3
          elsif input_word >= 45 && input_word <= 57
            result << (input_word - 44)
          elsif input_word >= 65 && input_word <= 90
            result << (input_word - 51)
          end
        else
          if input_word <= 31
            result << DX_CHAR_TRIPLET_SHIFT1
            result << input_word
          elsif input_word == 32
            result << 3
          elsif input_word <= 47
            result << DX_CHAR_TRIPLET_SHIFT2
            result << (input_word - 33)
          elsif input_word <= 57
            result << (input_word - 44)
          elsif input_word <= 64
            result << DX_CHAR_TRIPLET_SHIFT2
            result << (input_word - 43)
          elsif input_word <= 90 && enc_scheme == DX_SCHEME_C40
            result << (input_word - 51)
          elsif input_word <= 90 && enc_scheme == DX_SCHEME_TEXT
            result << DX_CHAR_TRIPLET_SHIFT3
            result << (input_word - 64)
          elsif input_word <= 95
            result << DX_CHAR_TRIPLET_SHIFT2
            result << (input_word - 69)
          elsif input_word == 96 && enc_scheme == DX_SCHEME_TEXT
            result << DX_CHAR_TRIPLET_SHIFT3
            result << 0
          elsif input_word <= 122 && enc_scheme == DX_SCHEME_TEXT
            result << (input_word - 83)
          elsif input_word <= 127
            result << DX_CHAR_TRIPLET_SHIFT3
            result << (input_word - 96)
          end
        end

        result
      end

      def process_end_of_symbol_triplet(channel, triplet, triplet_count, input_count)
        return false if (channel.current_length % 12) != 0

        input_adjust = triplet_count - input_count
        current_byte = channel.current_length / 12
        needed = current_byte + input_count
        needed = current_byte + 2 if input_count == 3
        size_idx = find_correct_symbol_size(needed, @size_idx_req)
        return false if size_idx == DX_SZ_SHAPE_AUTO

        remaining = @dm.send(:dx_get_symbol_attribute, DX_ATTR_SYMBOL_DATA_WORDS, size_idx) - current_byte

        if input_count == 1 && remaining == 1
          change_enc_scheme(channel, DX_SCHEME_ASCII, DX_UNLATCH_IMPLICIT)
          ok = encode_next_word(channel, DX_SCHEME_ASCII)
          return false unless ok
          return false if channel.invalid != DX_CHANNEL_VALID || channel.input_index != channel.input.length
        elsif remaining == 2
          if triplet_count == 3
            push_triplet(channel, triplet)
            increment_progress(channel, 24)
            channel.enc_scheme = DX_SCHEME_ASCII
            channel.input_index += 3
            channel.input_index -= input_adjust
          elsif triplet_count == 2
            t = [triplet[0], triplet[1], 0]
            push_triplet(channel, t)
            increment_progress(channel, 24)
            channel.enc_scheme = DX_SCHEME_ASCII
            channel.input_index += 2
            channel.input_index -= input_adjust
          elsif triplet_count == 1
            change_enc_scheme(channel, DX_SCHEME_ASCII, DX_UNLATCH_EXPLICIT)
            ok = encode_next_word(channel, DX_SCHEME_ASCII)
            return false unless ok
            return false if channel.invalid != DX_CHANNEL_VALID
          end
        else
          current_byte = channel.current_length / 12
          remaining = @dm.send(:dx_get_symbol_attribute, DX_ATTR_SYMBOL_DATA_WORDS, size_idx) - current_byte
          if remaining > 0
            change_enc_scheme(channel, DX_SCHEME_ASCII, DX_UNLATCH_EXPLICIT)
            while channel.input_index < channel.input.length
              ok = encode_next_word(channel, DX_SCHEME_ASCII)
              return false unless ok
              return false if channel.invalid != DX_CHANNEL_VALID
            end
          end
        end

        channel.input_index == channel.input.length
      end

      def push_triplet(channel, triplet)
        tv = 1600 * triplet[0] + 40 * triplet[1] + triplet[2] + 1
        push_input_word(channel, (tv >> 8) & 0xFF)
        push_input_word(channel, tv & 0xFF)
      end

      # --- Edifact encoding ---

      def encode_edifact_codeword(channel)
        return false if channel.enc_scheme != DX_SCHEME_EDIFACT

        input_value = channel.input[channel.input_index]
        if input_value < 32 || input_value > 94
          channel.invalid = DX_CHANNEL_UNSUPPORTED_CHAR
          return false
        end

        push_input_word(channel, input_value & 0x3f)
        increment_progress(channel, 9)
        channel.input_index += 1

        check_end_of_symbol_edifact(channel)
        true
      end

      def check_end_of_symbol_edifact(channel)
        edifact_values = channel.input.length - channel.input_index
        return if edifact_values > 4

        current_byte = channel.current_length / 12
        size_idx = find_correct_symbol_size(current_byte, DX_SZ_AUTO)
        symbol_codewords = @dm.send(:dx_get_symbol_attribute, DX_ATTR_SYMBOL_DATA_WORDS, size_idx) - current_byte

        if (channel.current_length % 12) == 0 && (symbol_codewords == 1 || symbol_codewords == 2)
          ascii_codewords = edifact_values
          if ascii_codewords <= symbol_codewords
            change_enc_scheme(channel, DX_SCHEME_ASCII, DX_UNLATCH_IMPLICIT)
            edifact_values.times do
              ok = encode_next_word(channel, DX_SCHEME_ASCII)
              return unless ok
              return if channel.invalid != DX_CHANNEL_VALID
            end
          end
        elsif edifact_values == 0
          change_enc_scheme(channel, DX_SCHEME_ASCII, DX_UNLATCH_EXPLICIT)
        end
      end

      # --- Base256 encoding ---

      def encode_base256_codeword(channel)
        return false if channel.enc_scheme != DX_SCHEME_BASE256

        first_byte_ptr_index = channel.first_code_word / 12
        header_byte0 = @dm.send(:dx_un_randomize_255,
          channel.encoded_words[first_byte_ptr_index],
          channel.first_code_word / 12 + 1)

        if header_byte0 <= 249
          new_data_length = header_byte0
        else
          new_data_length = 250 * (header_byte0 - 249)
          new_data_length += @dm.send(:dx_un_randomize_255,
            channel.encoded_words[first_byte_ptr_index + 1],
            channel.first_code_word / 12 + 2)
        end

        new_data_length += 1

        if new_data_length <= 249
          header_byte_count = 1
          hb = [new_data_length & 0xFF, 0]
        else
          header_byte_count = 2
          hb = [(new_data_length / 250 + 249) & 0xFF, (new_data_length % 250) & 0xFF]
        end

        return false if new_data_length <= 0 || new_data_length > 1555

        if new_data_length == 250
          i = channel.current_length / 12 - 1
          while i > channel.first_code_word / 12
            val_tmp = @dm.send(:dx_un_randomize_255, channel.encoded_words[i], i + 1)
            channel.encoded_words[i + 1] = @dm.send(:dx_randomize_255, val_tmp, i + 2)
            i -= 1
          end
          increment_progress(channel, 12)
          channel.encoded_length += 12
        end

        header_byte_count.times do |i|
          channel.encoded_words[first_byte_ptr_index + i] = @dm.send(:dx_randomize_255,
            hb[i], channel.first_code_word / 12 + i + 1)
        end

        push_input_word(channel, @dm.send(:dx_randomize_255,
          channel.input[channel.input_index], channel.current_length / 12 + 1))
        increment_progress(channel, 12)
        channel.input_index += 1
        true
      end

      # --- Channel utilities ---

      def push_input_word(channel, codeword)
        return if channel.encoded_length / 12 > 3 * 1558

        case channel.enc_scheme
        when DX_SCHEME_ASCII, DX_SCHEME_C40, DX_SCHEME_TEXT, DX_SCHEME_X12, DX_SCHEME_BASE256
          channel.encoded_words[channel.current_length / 12] = codeword & 0xFF
          channel.encoded_length += 12

        when DX_SCHEME_EDIFACT
          pos = channel.current_length % 4
          start_byte = ((channel.current_length + 9) / 12) - pos

          q = @dm.send(:dx_get_quadruplet_values,
            channel.encoded_words[start_byte],
            channel.encoded_words[start_byte + 1],
            channel.encoded_words[start_byte + 2])
          q[pos] = codeword & 0xFF
          ((pos + 1)...4).each do |i|
            q[i] = 0
          end

          if pos >= 2
            channel.encoded_words[start_byte + 2] = ((q[2] & 0x03) << 6) | q[3]
          end
          if pos >= 1
            channel.encoded_words[start_byte + 1] = ((q[1] & 0x0f) << 4) | (q[2] >> 2)
          end
          channel.encoded_words[start_byte] = (q[0] << 2) | (q[1] >> 4)

          channel.encoded_length += 9
        end
      end

      def increment_progress(channel, encoded_units)
        if channel.enc_scheme == DX_SCHEME_C40 || channel.enc_scheme == DX_SCHEME_TEXT
          pos = (channel.current_length % 6) / 2
          start_byte = (channel.current_length / 12) - (pos >> 1)
          triplet = @dm.send(:dx_get_triplet_values,
            channel.encoded_words[start_byte],
            channel.encoded_words[start_byte + 1])
          if triplet[pos] <= 2
            channel.current_length += 8
          end
        end

        channel.current_length += encoded_units
      end

      def change_enc_scheme(channel, target_scheme, unlatch_type)
        return if channel.enc_scheme == target_scheme

        case channel.enc_scheme
        when DX_SCHEME_ASCII
          if (channel.current_length % 12) != 0 && (channel.current_length % 20) != 0
            return
          end

        when DX_SCHEME_C40, DX_SCHEME_TEXT, DX_SCHEME_X12
          if (channel.current_length % 12) != 0 && (channel.current_length % 20) != 0
            channel.invalid = DX_CHANNEL_CANNOT_UNLATCH
            return
          end
          if channel.current_length != channel.encoded_length
            channel.invalid = DX_CHANNEL_CANNOT_UNLATCH
            return
          end
          if unlatch_type == DX_UNLATCH_EXPLICIT
            push_input_word(channel, DX_CHAR_TRIPLET_UNLATCH)
            increment_progress(channel, 12)
          end

        when DX_SCHEME_EDIFACT
          if (channel.current_length % 3) != 0
            return
          end
          if unlatch_type == DX_UNLATCH_EXPLICIT
            push_input_word(channel, DX_CHAR_EDIFACT_UNLATCH)
            increment_progress(channel, 9)
          end
          advance = (channel.current_length % 4) * 3
          channel.current_length += advance
          channel.encoded_length += advance

        # Base256: nothing on unlatch
        end

        channel.enc_scheme = DX_SCHEME_ASCII

        case target_scheme
        when DX_SCHEME_ASCII
          # no latch needed
        when DX_SCHEME_C40
          push_input_word(channel, DX_CHAR_C40_LATCH)
          increment_progress(channel, 12)
        when DX_SCHEME_TEXT
          push_input_word(channel, DX_CHAR_TEXT_LATCH)
          increment_progress(channel, 12)
        when DX_SCHEME_X12
          push_input_word(channel, DX_CHAR_X12_LATCH)
          increment_progress(channel, 12)
        when DX_SCHEME_EDIFACT
          push_input_word(channel, DX_CHAR_EDIFACT_LATCH)
          increment_progress(channel, 12)
        when DX_SCHEME_BASE256
          push_input_word(channel, DX_CHAR_BASE256_LATCH)
          increment_progress(channel, 12)
          push_input_word(channel, @dm.send(:dx_randomize_255, 0, 2))
          increment_progress(channel, 12)
        end

        channel.enc_scheme = target_scheme
        channel.first_code_word = channel.current_length - 12
      end

      def init_channel(channel, codewords)
        channel.enc_scheme = DX_SCHEME_ASCII
        channel.invalid = DX_CHANNEL_VALID
        channel.input_index = 0
        channel.input = codewords.dup
      end

      # --- Padding ---

      def add_pad_chars(buf, data_word_count, padded_size)
        pad_count = 0
        if data_word_count < padded_size
          pad_count += 1
          buf[data_word_count] = DX_CHAR_ASCII_PAD
          data_word_count += 1
        end

        while data_word_count < padded_size
          pad_count += 1
          buf[data_word_count] = @dm.send(:dx_randomize_253, DX_CHAR_ASCII_PAD, data_word_count + 1)
          data_word_count += 1
        end

        pad_count
      end

      # --- Module placement ECC200 ---

      def module_placement_ecc200(modules, codewords, size_idx, module_on_color)
        mr = @dm.send(:dx_get_symbol_attribute, DX_ATTR_MAPPING_MATRIX_ROWS, size_idx)
        mc = @dm.send(:dx_get_symbol_attribute, DX_ATTR_MAPPING_MATRIX_COLS, size_idx)

        ch = 0
        row = 4
        col = 0

        loop do
          if row == mr && col == 0
            pattern_shape_special1(modules, mr, mc, codewords, ch, module_on_color)
            ch += 1
          elsif row == mr - 2 && col == 0 && (mc % 4) != 0
            pattern_shape_special2(modules, mr, mc, codewords, ch, module_on_color)
            ch += 1
          elsif row == mr - 2 && col == 0 && (mc % 8) == 4
            pattern_shape_special3(modules, mr, mc, codewords, ch, module_on_color)
            ch += 1
          elsif row == mr + 4 && col == 2 && (mc % 8) == 0
            pattern_shape_special4(modules, mr, mc, codewords, ch, module_on_color)
            ch += 1
          end

          # Upward diagonal
          loop do
            if row < mr && col >= 0 && (modules[row * mc + col] & DX_MODULE_VISITED) == 0
              pattern_shape_standard(modules, mr, mc, row, col, codewords, ch, module_on_color)
              ch += 1
            end
            row -= 2
            col += 2
            break unless row >= 0 && col < mc
          end
          row += 1
          col += 3

          # Downward diagonal
          loop do
            if row >= 0 && col < mc && (modules[row * mc + col] & DX_MODULE_VISITED) == 0
              pattern_shape_standard(modules, mr, mc, row, col, codewords, ch, module_on_color)
              ch += 1
            end
            row += 2
            col -= 2
            break unless row < mr && col >= 0
          end
          row += 3
          col += 1

          break unless row < mr || col < mc
        end

        # Fill unvisited corner
        if (modules[mr * mc - 1] & DX_MODULE_VISITED) == 0
          modules[mr * mc - 1] |= module_on_color
          modules[mr * mc - mc - 2] |= module_on_color
        end
      end

      def pattern_shape_standard(modules, mr, mc, row, col, cw, ci, color)
        place_module(modules, mr, mc, row - 2, col - 2, cw, ci, DX_MASK_BIT1, color)
        place_module(modules, mr, mc, row - 2, col - 1, cw, ci, DX_MASK_BIT2, color)
        place_module(modules, mr, mc, row - 1, col - 2, cw, ci, DX_MASK_BIT3, color)
        place_module(modules, mr, mc, row - 1, col - 1, cw, ci, DX_MASK_BIT4, color)
        place_module(modules, mr, mc, row - 1, col,     cw, ci, DX_MASK_BIT5, color)
        place_module(modules, mr, mc, row,     col - 2, cw, ci, DX_MASK_BIT6, color)
        place_module(modules, mr, mc, row,     col - 1, cw, ci, DX_MASK_BIT7, color)
        place_module(modules, mr, mc, row,     col,     cw, ci, DX_MASK_BIT8, color)
      end

      def pattern_shape_special1(modules, mr, mc, cw, ci, color)
        place_module(modules, mr, mc, mr - 1, 0,      cw, ci, DX_MASK_BIT1, color)
        place_module(modules, mr, mc, mr - 1, 1,      cw, ci, DX_MASK_BIT2, color)
        place_module(modules, mr, mc, mr - 1, 2,      cw, ci, DX_MASK_BIT3, color)
        place_module(modules, mr, mc, 0,      mc - 2, cw, ci, DX_MASK_BIT4, color)
        place_module(modules, mr, mc, 0,      mc - 1, cw, ci, DX_MASK_BIT5, color)
        place_module(modules, mr, mc, 1,      mc - 1, cw, ci, DX_MASK_BIT6, color)
        place_module(modules, mr, mc, 2,      mc - 1, cw, ci, DX_MASK_BIT7, color)
        place_module(modules, mr, mc, 3,      mc - 1, cw, ci, DX_MASK_BIT8, color)
      end

      def pattern_shape_special2(modules, mr, mc, cw, ci, color)
        place_module(modules, mr, mc, mr - 3, 0,      cw, ci, DX_MASK_BIT1, color)
        place_module(modules, mr, mc, mr - 2, 0,      cw, ci, DX_MASK_BIT2, color)
        place_module(modules, mr, mc, mr - 1, 0,      cw, ci, DX_MASK_BIT3, color)
        place_module(modules, mr, mc, 0,      mc - 4, cw, ci, DX_MASK_BIT4, color)
        place_module(modules, mr, mc, 0,      mc - 3, cw, ci, DX_MASK_BIT5, color)
        place_module(modules, mr, mc, 0,      mc - 2, cw, ci, DX_MASK_BIT6, color)
        place_module(modules, mr, mc, 0,      mc - 1, cw, ci, DX_MASK_BIT7, color)
        place_module(modules, mr, mc, 1,      mc - 1, cw, ci, DX_MASK_BIT8, color)
      end

      def pattern_shape_special3(modules, mr, mc, cw, ci, color)
        place_module(modules, mr, mc, mr - 3, 0,      cw, ci, DX_MASK_BIT1, color)
        place_module(modules, mr, mc, mr - 2, 0,      cw, ci, DX_MASK_BIT2, color)
        place_module(modules, mr, mc, mr - 1, 0,      cw, ci, DX_MASK_BIT3, color)
        place_module(modules, mr, mc, 0,      mc - 2, cw, ci, DX_MASK_BIT4, color)
        place_module(modules, mr, mc, 0,      mc - 1, cw, ci, DX_MASK_BIT5, color)
        place_module(modules, mr, mc, 1,      mc - 1, cw, ci, DX_MASK_BIT6, color)
        place_module(modules, mr, mc, 2,      mc - 1, cw, ci, DX_MASK_BIT7, color)
        place_module(modules, mr, mc, 3,      mc - 1, cw, ci, DX_MASK_BIT8, color)
      end

      def pattern_shape_special4(modules, mr, mc, cw, ci, color)
        place_module(modules, mr, mc, mr - 1, 0,      cw, ci, DX_MASK_BIT1, color)
        place_module(modules, mr, mc, mr - 1, mc - 1, cw, ci, DX_MASK_BIT2, color)
        place_module(modules, mr, mc, 0,      mc - 3, cw, ci, DX_MASK_BIT3, color)
        place_module(modules, mr, mc, 0,      mc - 2, cw, ci, DX_MASK_BIT4, color)
        place_module(modules, mr, mc, 0,      mc - 1, cw, ci, DX_MASK_BIT5, color)
        place_module(modules, mr, mc, 1,      mc - 3, cw, ci, DX_MASK_BIT6, color)
        place_module(modules, mr, mc, 1,      mc - 2, cw, ci, DX_MASK_BIT7, color)
        place_module(modules, mr, mc, 1,      mc - 1, cw, ci, DX_MASK_BIT8, color)
      end

      def place_module(modules, mr, mc, row, col, codeword, ci, mask, color)
        if row < 0
          row += mr
          col += 4 - ((mr + 4) % 8)
        end
        if col < 0
          col += mc
          row += 4 - ((mc + 4) % 8)
        end

        index = row * mc + col
        if (modules[index] & DX_MODULE_ASSIGNED) != 0
          if (modules[index] & color) != 0
            modules[index] = modules[index] | mask
          else
            modules[index] = modules[index] & (~mask & 0xFF)
          end
        else
          if (codeword[ci] & mask) != 0
            modules[index] |= color
          end
          modules[index] |= DX_MODULE_ASSIGNED
        end

        modules[index] |= DX_MODULE_VISITED
      end

      # --- Pattern output ---

      def print_pattern_raw
        sc = @symbol_cols
        sr = @symbol_rows
        si = @size_idx

        @raw_data = Array.new(sc) { Array.new(sr, false) }
        sr.times do |symbol_row|
          sc.times do |symbol_col|
            status = @message.symbol_module_status(@dm, si, symbol_row, symbol_col)
            @raw_data[symbol_col][sr - symbol_row - 1] = (status & DX_MODULE_ON_BLUE) != 0
          end
        end
      end
    end

    # ================================================================
    # Private instance methods for DataMatrix
    # ================================================================

    def string_to_bytes(code)
      code.bytes.map(&:to_i)
    end

    def draw_bytes(data, width, height)
      patt = cal_data_matrix_bytes(data)

      pattern_cols = patt.length
      raise "empty pattern" if pattern_cols == 0
      pattern_rows = patt[0].length
      raise "empty pattern" if pattern_rows == 0

      if svg_output?
        draw_svg_2d(patt, width, height)
      else
        draw_png_2d(patt, width, height)
      end
    end

    def cal_data_matrix_bytes(byte_data)
      enc = DxEncode.new(self)
      enc.module_size = 1
      enc.margin_size = 0
      enc.size_idx_req = @code_size
      enc.scheme = @encode_scheme

      enc.encode_data_matrix_raw(byte_data)
      raise "encoding produced no pattern data" if enc.raw_data.nil?
      enc.raw_data
    end
  end
end
