# frozen_string_literal: true

module BarcodePao
  module DataBarRenderer
    # Render multi-row pattern-based barcode as SVG
    # patterns: run-length encoded strings for each row
    # row_heights: height for each row (-1 = variable/fills remaining)
    def draw_multi_row_svg(patterns, row_heights, width, height)
      raise "no patterns to render" if patterns.empty?

      # Calculate total fixed height and variable rows
      total_fixed = 0
      variable_count = 0
      row_heights.each do |h|
        if h > 0
          total_fixed += h
        else
          variable_count += 1
        end
      end

      # Calculate row pixel heights
      row_px_heights = Array.new(row_heights.length, 0)
      if variable_count > 0
        variable_height = [(height - total_fixed) / variable_count, 1].max
        row_heights.each_with_index do |h, i|
          row_px_heights[i] = h > 0 ? h : variable_height
        end
      else
        scale = height.to_f / total_fixed
        row_heights.each_with_index do |h, i|
          row_px_heights[i] = (h * scale).round
        end
      end

      svg_begin(width, height)
      svg_rect(0, 0, width.to_f, height.to_f, @background_color)

      y = 0
      patterns.each_with_index do |pat, ri|
        break if ri >= row_px_heights.length
        rh = row_px_heights[ri]
        _draw_pattern_row_svg(pat, 0, y, width, rh)
        y += rh
      end

      if BarcodePao.trial_mode?
        svg_sample_overlay(0, 0, width, height)
      end
      @svg_stream << "</svg>\n"
    end

    # Render multi-row pattern-based barcode as PNG
    def draw_multi_row_png(patterns, row_heights, width, height)
      raise "no patterns to render" if patterns.empty?

      total_fixed = 0
      variable_count = 0
      row_heights.each do |h|
        if h > 0
          total_fixed += h
        else
          variable_count += 1
        end
      end

      row_px_heights = Array.new(row_heights.length, 0)
      if variable_count > 0
        variable_height = [(height - total_fixed) / variable_count, 1].max
        row_heights.each_with_index do |h, i|
          row_px_heights[i] = h > 0 ? h : variable_height
        end
      else
        scale = height.to_f / total_fixed
        row_heights.each_with_index do |h, i|
          row_px_heights[i] = (h * scale).round
        end
      end

      img = PNGImage.new(width, height, @background_color)

      y = 0
      patterns.each_with_index do |pat, ri|
        break if ri >= row_px_heights.length
        rh = row_px_heights[ri]
        _draw_pattern_row_png(img, pat, 0, y, width, rh)
        y += rh
      end

      if BarcodePao.trial_mode?
        img.draw_sample_overlay(0, 0, width, height)
      end

      @image_buffer = img.to_png
    end

    private

    def _draw_pattern_row_svg(pattern, x, y, width, height)
      return if pattern.nil? || pattern.empty?

      start_idx = 0
      is_bar = true
      if pattern[0] == "0"
        is_bar = false
        start_idx = 1
      end

      total_modules = 0
      (start_idx...pattern.length).each { |i| total_modules += pattern[i].to_i }
      return if total_modules <= 0

      unit_w = width.to_f / total_modules
      accum = 0.0
      bar = is_bar

      (start_idx...pattern.length).each do |i|
        modules = pattern[i].to_i
        x1 = (x + accum * unit_w).round.to_f
        accum += modules
        x2 = (x + accum * unit_w).round.to_f
        if bar && x2 > x1
          svg_rect(x1, y.to_f, x2 - x1, height.to_f, @foreground_color)
        end
        bar = !bar
      end
    end

    def _draw_pattern_row_png(img, pattern, x, y, width, height)
      return if pattern.nil? || pattern.empty?

      start_idx = 0
      is_bar = true
      if pattern[0] == "0"
        is_bar = false
        start_idx = 1
      end

      total_modules = 0
      (start_idx...pattern.length).each { |i| total_modules += pattern[i].to_i }
      return if total_modules <= 0

      unit_w = width.to_f / total_modules
      accum = 0.0
      bar = is_bar

      (start_idx...pattern.length).each do |i|
        modules = pattern[i].to_i
        x1 = (x + accum * unit_w).round
        accum += modules
        x2 = (x + accum * unit_w).round
        if bar && x2 > x1
          img.fill_rect(x1, y, x2, y + height, @foreground_color)
        end
        bar = !bar
      end
    end
  end

  module DataBarUtils
    def self.pad_left(s, length)
      s.rjust(length, "0")
    end

    def self.parse_int64(s)
      s.to_i
    end
  end

  class BarcodeBase1D
    include DataBarRenderer
  end
end
