# frozen_string_literal: true

module BarcodePao
  CODE93_CHARS = "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ-. $/+%<>?!"
  CODE93_PATTERNS = [
    "131112", "111213", "111312", "111411", "121113", "121212", "121311",
    "111114", "131211", "141111", "211113", "211212", "211311", "221112",
    "221211", "231111", "112113", "112212", "112311", "122112", "132111",
    "111123", "111222", "111321", "121122", "131121", "212112", "212211",
    "211122", "211221", "221121", "222111", "112122", "112221", "122121",
    "123111", "121131", "311112", "311211", "321111", "112131", "113121",
    "211131", "121221", "312111", "311121", "122211",
  ].freeze
  CODE93_START = "111141"
  CODE93_STOP  = "1111411"

  class Code93 < BarcodeBase1D
    def initialize(output_format = FORMAT_PNG)
      super(output_format)
    end

    def encode(code)
      raise "empty string" if code.nil? || code.empty?

      code = code.upcase
      code.each_char do |ch|
        raise "invalid character in CODE93: #{ch}" unless CODE93_CHARS.include?(ch)
      end

      check_c = code93_modulus47(code, 20)
      code_with_c = code + CODE93_CHARS[check_c]
      check_k = code93_modulus47(code_with_c, 15)

      result = []
      CODE93_START.each_char { |ch| result << ch.to_i }

      code.each_char do |ch|
        index = CODE93_CHARS.index(ch)
        CODE93_PATTERNS[index].each_char { |p| result << p.to_i }
      end

      CODE93_PATTERNS[check_c].each_char { |p| result << p.to_i }
      CODE93_PATTERNS[check_k].each_char { |p| result << p.to_i }

      CODE93_STOP.each_char { |ch| result << ch.to_i }
      result
    end

    def draw(code, width, height)
      draw_1d(code, width, height) { |c| encode(c) }
    end

    private

    def code93_modulus47(code, weight)
      total = 0
      code.length.times do |i|
        value = CODE93_CHARS.index(code[i])
        raise "invalid character in CODE93: #{code[i]}" if value.nil?
        pos_weight = code.length - i
        pos_weight = ((pos_weight - 1) % weight) + 1 if pos_weight > weight
        total += value * pos_weight
      end
      total % 47
    end
  end
end
