# frozen_string_literal: true

module BarcodePao
  CODE39_CHARS = "1234567890ABCDEFGHIJKLMNOPQRSTUVWXYZ-. *$/+%"
  CODE39_PTN = [
    0x121, 0x061, 0x160, 0x031, 0x130, 0x070, 0x025, 0x124, 0x064, 0x034,
    0x109, 0x049, 0x148, 0x019, 0x118, 0x058, 0x00d, 0x10c, 0x04c, 0x01c,
    0x103, 0x043, 0x142, 0x013, 0x112, 0x052, 0x007, 0x106, 0x046, 0x016,
    0x181, 0x0c1, 0x1c0, 0x091, 0x190, 0x0d0, 0x085, 0x184, 0x0c4, 0x094,
    0x0a8, 0x0a2, 0x08a, 0x02a,
  ].freeze

  class Code39 < BarcodeBase1D
    attr_accessor :show_start_stop

    def initialize(output_format = FORMAT_PNG)
      super(output_format)
      @show_start_stop = true
    end

    def encode(code)
      raise "empty string" if code.nil? || code.empty?

      valid_code = String.new
      code.each_char do |ch|
        upper = ch.upcase
        raise "invalid character in CODE39: '*' is reserved for start/stop code" if upper == "*"
        pos = CODE39_CHARS.index(upper)
        raise "invalid character in CODE39: #{ch}" if pos.nil? || pos == 39
        valid_code << upper
      end

      full_code = "*#{valid_code}*"

      result = []
      full_code.each_char.with_index do |ch, idx|
        result << 1 if idx > 0 # inter-character gap
        char_index = CODE39_CHARS.index(ch)
        bit_pattern = CODE39_PTN[char_index]

        # 9-bit pattern: MSB first, 1=wide(3), 0=narrow(1)
        mask = 0x100
        9.times do |i|
          result << ((bit_pattern & (mask >> i)) != 0 ? 3 : 1)
        end
      end
      result
    end

    def draw(code, width, height)
      draw_1d(code, width, height) { |c| encode(c) }
    end
  end
end
