# frozen_string_literal: true

module BarcodePao
  CODE128_AUTO  = 0
  CODE128_CODE_A = 1
  CODE128_CODE_B = 2
  CODE128_CODE_C = 3

  CODE128_PATTERNS = [
    "212222", "222122", "222221", "121223", "121322", "131222", "122213",
    "122312", "132212", "221213", "221312", "231212", "112232", "122132",
    "122231", "113222", "123122", "123221", "223211", "221132", "221231",
    "213212", "223112", "312131", "311222", "321122", "321221", "312212",
    "322112", "322211", "212123", "212321", "232121", "111323", "131123",
    "131321", "112313", "132113", "132311", "211313", "231113", "231311",
    "112133", "112331", "132131", "113123", "113321", "133121", "313121",
    "211331", "231131", "213113", "213311", "213131", "311123", "311321",
    "331121", "312113", "312311", "332111", "314111", "221411", "431111",
    "111224", "111422", "121124", "121421", "141122", "141221", "112214",
    "112412", "122114", "122411", "142112", "142211", "241211", "221114",
    "413111", "241112", "134111", "111242", "121142", "121241", "114212",
    "124112", "124211", "411212", "421112", "421211", "212141", "214121",
    "412121", "111143", "111341", "131141", "114113", "114311", "411113",
    "411311", "113141", "114131", "311141", "411131",
    "211412", "211214", "211232", "2331112", # 103-106: Start A/B/C, Stop
  ].freeze

  DP_INF = 999_999_999
  DP_SET_A = 0
  DP_SET_B = 1
  DP_SET_C = 2

  class Code128 < BarcodeBase1D
    attr_accessor :code_mode

    def initialize(output_format = FORMAT_PNG)
      super(output_format)
      @code_mode = CODE128_AUTO
    end

    def encode(code)
      raise "empty string" if code.nil? || code.empty?
      if @code_mode == CODE128_AUTO
        encode_auto(code)
      else
        encode_single_set(code, @code_mode)
      end
    end

    def draw(code, width, height)
      draw_1d(code, width, height) { |c| encode(c) }
    end

    private

    def code128_a_index(c)
      c.ord <= 95 ? c.ord : -1
    end

    def code128_b_index(c)
      o = c.ord
      (o < 32 || o > 126) ? -1 : o - 32
    end

    def code128_c_index(s, i)
      return -1 if i + 1 >= s.length
      c1, c2 = s[i].ord, s[i + 1].ord
      return -1 if c1 < 48 || c1 > 57 || c2 < 48 || c2 > 57
      (c1 - 48) * 10 + (c2 - 48)
    end

    def solve_dp(s, i, iset, memo)
      return { cost: 0, enc: [] } if i >= s.length

      key = (i << 2) + iset
      return memo[key] if memo.key?(key)

      best_cost = DP_INF
      best_enc = nil
      c = s[i]

      # Stay in current set
      case iset
      when DP_SET_A
        idx = code128_a_index(c)
        if idx >= 0
          nxt = solve_dp(s, i + 1, iset, memo)
          cost = 1 + nxt[:cost]
          if cost < best_cost
            best_cost = cost
            best_enc = [idx] + nxt[:enc]
          end
        end
      when DP_SET_B
        idx = code128_b_index(c)
        if idx >= 0
          nxt = solve_dp(s, i + 1, iset, memo)
          cost = 1 + nxt[:cost]
          if cost < best_cost
            best_cost = cost
            best_enc = [idx] + nxt[:enc]
          end
        end
      when DP_SET_C
        idx = code128_c_index(s, i)
        if idx >= 0
          nxt = solve_dp(s, i + 2, iset, memo)
          cost = 1 + nxt[:cost]
          if cost < best_cost
            best_cost = cost
            best_enc = [idx] + nxt[:enc]
          end
        end
      end

      # Switch to different set
      switch_vals = [
        [0, 100, 99],   # from A
        [101, 0, 99],   # from B
        [101, 100, 0],  # from C
      ]

      3.times do |st|
        next if st == iset
        sv = switch_vals[iset][st]

        case st
        when DP_SET_A
          idx = code128_a_index(c)
          if idx >= 0
            nxt = solve_dp(s, i + 1, st, memo)
            cost = 2 + nxt[:cost]
            if cost < best_cost
              best_cost = cost
              best_enc = [sv, idx] + nxt[:enc]
            end
          end
        when DP_SET_B
          idx = code128_b_index(c)
          if idx >= 0
            nxt = solve_dp(s, i + 1, st, memo)
            cost = 2 + nxt[:cost]
            if cost < best_cost
              best_cost = cost
              best_enc = [sv, idx] + nxt[:enc]
            end
          end
        when DP_SET_C
          idx = code128_c_index(s, i)
          if idx >= 0
            nxt = solve_dp(s, i + 2, st, memo)
            cost = 2 + nxt[:cost]
            if cost < best_cost
              best_cost = cost
              best_enc = [sv, idx] + nxt[:enc]
            end
          end
        end
      end

      raise "DP encode error at pos=#{i}" if best_cost == DP_INF

      memo[key] = { cost: best_cost, enc: best_enc }
    end

    def encode_single_set(s, fixed_set)
      start_val = 102 + fixed_set
      vals = [start_val]

      i = 0
      while i < s.length
        ch = s[i]
        case fixed_set
        when CODE128_CODE_C
          cc = code128_c_index(s, i)
          raise "not 2-digit at pos=#{i}" if cc < 0
          vals << cc
          i += 2
        when CODE128_CODE_A
          idx = code128_a_index(ch)
          raise "invalid char for CODE_A" if idx < 0
          vals << idx
          i += 1
        else
          idx = code128_b_index(ch)
          raise "invalid char for CODE_B" if idx < 0
          vals << idx
          i += 1
        end
      end

      check_sum = vals[0]
      (1...vals.length).each { |j| check_sum += vals[j] * j }
      vals << (check_sum % 103)
      vals << 106
      expand_code128_pattern(vals)
    end

    def encode_auto(s)
      candidates = []
      3.times do |iset|
        start_val = 103 + iset
        memo = {}
        begin
          r = solve_dp(s, 0, iset, memo)
          candidates << { start_val: start_val, cost: r[:cost], enc: r[:enc] }
        rescue
          # skip invalid start set
        end
      end

      raise "no valid start code" if candidates.empty?

      best = candidates.min_by { |c| c[:cost] }

      enc_vals = [best[:start_val]] + best[:enc]
      check_sum = enc_vals[0]
      (1...enc_vals.length).each { |j| check_sum += enc_vals[j] * j }
      enc_vals << (check_sum % 103)
      enc_vals << 106
      expand_code128_pattern(enc_vals)
    end

    def expand_code128_pattern(vals)
      result = []
      vals.each do |v|
        raise "invalid code value #{v}" if v < 0 || v > 106
        ptn = CODE128_PATTERNS[v]
        ptn.each_char { |ch| result << ch.to_i }
      end
      result
    end
  end
end
