# frozen_string_literal: true

module BarcodePao
  class BarcodeBase2D < BarcodeBase
    attr_accessor :string_encoding

    def initialize(output_format)
      super(output_format)
      @string_encoding = "utf-8"
    end

    # Draw 2D pattern (column-major bool matrix) as SVG
    def draw_svg_2d(pattern, width, height)
      cols = pattern.length
      rows = cols > 0 ? pattern[0].length : 0
      return if cols == 0 || rows == 0

      svg_begin(width, height)
      svg_rect(0, 0, width.to_f, height.to_f, @background_color)

      cell_w = width.to_f / cols
      cell_h = height.to_f / rows

      cols.times do |c|
        rows.times do |r|
          if pattern[c][r]
            x = (c * cell_w).round(2)
            y = (r * cell_h).round(2)
            w = ((c + 1) * cell_w).round(2) - x
            h = ((r + 1) * cell_h).round(2) - y
            svg_rect(x, y, w, h, @foreground_color)
          end
        end
      end

      svg_end
    end

    # Draw 2D pattern as PNG
    def draw_png_2d(pattern, width, height)
      cols = pattern.length
      rows = cols > 0 ? pattern[0].length : 0
      return if cols == 0 || rows == 0

      require_relative "render_png"
      img = PNGImage.new(width, height, @background_color)

      cell_w = width.to_f / cols
      cell_h = height.to_f / rows

      cols.times do |c|
        rows.times do |r|
          if pattern[c][r]
            x1 = (c * cell_w).round
            y1 = (r * cell_h).round
            x2 = ((c + 1) * cell_w).round
            y2 = ((r + 1) * cell_h).round
            img.fill_rect(x1, y1, x2, y2, @foreground_color)
          end
        end
      end

      if BarcodePao.trial_mode?
        img.draw_sample_overlay(0, 0, width, height)
      end

      @image_buffer = img.to_png
    end

    # Draw row-major pattern (for PDF417)
    def draw_svg_2d_rows(pattern, width, height)
      rows = pattern.length
      cols = rows > 0 ? pattern[0].length : 0
      return if rows == 0 || cols == 0

      svg_begin(width, height)
      svg_rect(0, 0, width.to_f, height.to_f, @background_color)

      cell_w = width.to_f / cols
      cell_h = height.to_f / rows

      rows.times do |r|
        cols.times do |c|
          if pattern[r][c]
            x = (c * cell_w).round(2)
            y = (r * cell_h).round(2)
            w = ((c + 1) * cell_w).round(2) - x
            h = ((r + 1) * cell_h).round(2) - y
            svg_rect(x, y, w, h, @foreground_color)
          end
        end
      end

      svg_end
    end

    def draw_png_2d_rows(pattern, width, height)
      rows = pattern.length
      cols = rows > 0 ? pattern[0].length : 0
      return if rows == 0 || cols == 0

      require_relative "render_png"
      img = PNGImage.new(width, height, @background_color)

      cell_w = width.to_f / cols
      cell_h = height.to_f / rows

      rows.times do |r|
        cols.times do |c|
          if pattern[r][c]
            x1 = (c * cell_w).round
            y1 = (r * cell_h).round
            x2 = ((c + 1) * cell_w).round
            y2 = ((r + 1) * cell_h).round
            img.fill_rect(x1, y1, x2, y2, @foreground_color)
          end
        end
      end

      if BarcodePao.trial_mode?
        img.draw_sample_overlay(0, 0, width, height)
      end

      @image_buffer = img.to_png
    end
  end
end
