# frozen_string_literal: true

module BarcodePao
  class BarcodeBase1D < BarcodeBase
    attr_accessor :show_text, :text_even_spacing, :text_font_scale,
                  :text_horizontal_spacing_scale, :text_vertical_offset_scale,
                  :min_line_width

    def initialize(output_format)
      super(output_format)
      @show_text = true
      @text_even_spacing = true
      @text_font_scale = 1.0
      @text_horizontal_spacing_scale = 1.0
      @text_vertical_offset_scale = 1.0
      @min_line_width = 1
    end

    # Generic 1D draw: encode → render
    def draw_1d(code, width, height, &encoder)
      pattern = encoder.call(code)
      if svg_output?
        draw_svg_1d(pattern, code, width, height)
      else
        draw_png_1d(pattern, code, width, height)
      end
    end

    # --- SVG rendering for generic 1D ---

    def draw_svg_1d(pattern, code, width, height)
      display = @show_text ? code : ""
      draw_svg_bars(pattern, width, height, display)
    end

    def draw_svg_bars(pattern, width, height, text)
      total_units = pattern.sum
      raise "pattern has zero total units" if total_units <= 0

      font_size = [8, (height * 0.15 * @text_font_scale).to_i].max

      text_h = 0
      if text && !text.empty?
        text_h = (font_size * 1.4).to_i
      end
      bar_h = height - text_h
      if bar_h <= 0
        bar_h = height
        text = ""
      end

      svg_begin(width, height)
      svg_rect(0, 0, width.to_f, height.to_f, @background_color)

      unit_w = width.to_f / total_units
      accum = 0.0
      is_bar = true
      pattern.each do |units|
        x1 = (accum * unit_w).round
        accum += units
        x2 = (accum * unit_w).round
        if is_bar && x2 > x1
          svg_rect(x1.to_f, 0, (x2 - x1).to_f, bar_h.to_f, @foreground_color)
        end
        is_bar = !is_bar
      end

      if text && !text.empty?
        text_y = bar_h + 2.0
        if @text_even_spacing
          svg_text_even_distribution(text, 0, text_y, width.to_f, font_size, @text_horizontal_spacing_scale)
        else
          svg_text(width / 2.0, text_y, text, font_size, @foreground_color, "middle")
        end
      end

      svg_end
    end

    def svg_text_even_distribution(text, x, y, width, font_size, h_scale)
      return if text.nil? || text.empty?
      num_chars = text.length

      margin_ratio = if width < 100
                       0.03
                     elsif width > 400
                       0.07
                     else
                       0.05
                     end

      horizontal_margin = width * margin_ratio * h_scale
      text_width = width - 2 * horizontal_margin

      char_spacing = if num_chars > 1
                       text_width / (num_chars - 1).to_f
                     else
                       text_width
                     end

      min_spacing = font_size / 2.0
      char_spacing = min_spacing if char_spacing < min_spacing

      total_text_width = num_chars > 1 ? char_spacing * (num_chars - 1) : 0.0
      start_x = x + (width - total_text_width) / 2.0

      num_chars.times do |i|
        char_x = start_x + i * char_spacing
        svg_text(char_x, y, text[i], font_size, @foreground_color, "middle")
      end
    end
  end
end
