# frozen_string_literal: true

require "base64"

module BarcodePao
  FORMAT_PNG  = "png"
  FORMAT_JPEG = "jpeg"
  FORMAT_SVG  = "svg"

  class BarcodeBase
    attr_accessor :foreground_color, :background_color, :fit_width,
                  :px_adjust_black, :px_adjust_white
    attr_reader :output_format

    def initialize(output_format)
      @foreground_color = [0, 0, 0, 255]
      @background_color = [255, 255, 255, 255]
      @output_format = output_format
      @fit_width = false
      @px_adjust_black = 0
      @px_adjust_white = 0
      @svg_stream = String.new
      @svg_width = 0
      @svg_height = 0
      @image_buffer = nil
    end

    def set_output_format(fmt)
      @output_format = fmt
    end

    def set_foreground_color(r, g, b, a = 255)
      @foreground_color = [r, g, b, a]
    end

    def set_background_color(r, g, b, a = 255)
      @background_color = [r, g, b, a]
    end

    def svg_output?
      @output_format == FORMAT_SVG
    end

    def get_image_memory
      @image_buffer
    end

    def get_image_base64
      raise "GetImageBase64() is not available in SVG mode; use get_svg()" if svg_output?
      return "" if @image_buffer.nil? || @image_buffer.empty?
      encoded = Base64.strict_encode64(@image_buffer)
      mime = @output_format == FORMAT_JPEG ? "image/jpeg" : "image/png"
      "data:#{mime};base64,#{encoded}"
    end

    def get_svg
      raise "get_svg() requires SVG output mode" unless svg_output?
      @svg_stream
    end

    # --- SVG helpers ---

    def svg_begin(width, height)
      @svg_width = width
      @svg_height = height
      @svg_stream = String.new
      @svg_stream << "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n"
      @svg_stream << "<svg xmlns=\"http://www.w3.org/2000/svg\" version=\"1.1\" " \
                     "width=\"#{width}\" height=\"#{height}\" " \
                     "viewBox=\"0 0 #{width} #{height}\">\n"
    end

    def svg_end
      if BarcodePao.trial_mode?
        svg_sample_overlay(0, 0, @svg_width, @svg_height)
      end
      @svg_stream << "</svg>\n"
    end

    def svg_sample_overlay(x, y, width, height)
      font_size = (height * 0.12).to_i
      font_size = 8 if font_size < 8
      font_size = 40 if font_size > 40
      margin = (height * 0.01).to_i
      margin = 2 if margin < 2
      red = [255, 0, 0, 255]
      text = BarcodePao.trial_text
      @svg_stream << "  <text x=\"#{x + margin}\" y=\"#{y + margin}\" " \
                     "font-family=\"Arial, sans-serif\" font-size=\"#{font_size}\" " \
                     "font-weight=\"bold\" font-style=\"italic\" " \
                     "fill=\"#{color_to_rgb(red)}\" " \
                     "dominant-baseline=\"hanging\">#{text}</text>\n"
    end

    def svg_rect(x, y, w, h, color)
      opacity = ""
      if color[3] < 255
        opacity = "\" fill-opacity=\"#{'%.2f' % (color[3] / 255.0)}"
      end
      @svg_stream << "  <rect x=\"#{'%.2f' % x}\" y=\"#{'%.2f' % y}\" " \
                     "width=\"#{'%.2f' % w}\" height=\"#{'%.2f' % h}\" " \
                     "fill=\"#{color_to_rgb(color)}#{opacity}\"/>\n"
    end

    def svg_text(x, y, text, font_size, color, anchor = "middle")
      @svg_stream << "  <text x=\"#{'%.2f' % x}\" y=\"#{'%.2f' % y}\" " \
                     "font-family=\"Arial, sans-serif\" font-size=\"#{font_size}\" " \
                     "fill=\"#{color_to_rgb(color)}\" text-anchor=\"#{anchor}\" " \
                     "dominant-baseline=\"hanging\">#{text}</text>\n"
    end

    def color_to_rgb(c)
      "rgb(#{c[0]},#{c[1]},#{c[2]})"
    end
  end
end
