# Barcode.Ruby - All-in-One
#
# Full 18-type barcode demo app using Sinatra.
# 4 output modes: PNG, SVG, PDF (Prawn), Canvas (ChunkyPNG)
#
#   bundle exec ruby app.rb
#   -> http://localhost:5741

require "sinatra"
require "json"
require "base64"
require "date"
require "stringio"

# Load Pure Ruby barcode library
$LOAD_PATH.unshift(File.join(__dir__, "..", "barcode_pao", "lib"))
require "barcode_pao"

set :port, 5741
set :bind, "0.0.0.0"
set :host_authorization, permitted_hosts: [".pao.ac", "localhost"]

BARCODE_TYPES = [
  # 2D
  { id: "QR", label: "QR Code", group: "2D Barcode", dim: "2d", default: "https://www.pao.ac/", w: 200, h: 200 },
  { id: "DataMatrix", label: "DataMatrix", group: "2D Barcode", dim: "2d", default: "Hello DataMatrix", w: 200, h: 200 },
  { id: "PDF417", label: "PDF417", group: "2D Barcode", dim: "2d", default: "Hello PDF417", w: 200, h: 100 },
  # Special
  { id: "GS1_128", label: "GS1-128", group: "Special", dim: "1d", default: "[01]04912345123459", w: 400, h: 100 },
  { id: "YubinCustomer", label: "Yubin Customer", group: "Special", dim: "postal", default: "1060032", w: 400, h: 60 },
  # 1D
  { id: "Code128", label: "Code 128", group: "1D Barcode", dim: "1d", default: "Hello-2026", w: 400, h: 100 },
  { id: "Code39", label: "Code 39", group: "1D Barcode", dim: "1d", default: "HELLO-123", w: 400, h: 100 },
  { id: "Code93", label: "Code 93", group: "1D Barcode", dim: "1d", default: "CODE93", w: 400, h: 100 },
  { id: "NW7", label: "NW-7 / Codabar", group: "1D Barcode", dim: "1d", default: "A123456B", w: 400, h: 100 },
  { id: "ITF", label: "ITF", group: "1D Barcode", dim: "1d", default: "123456", w: 400, h: 100 },
  { id: "Matrix2of5", label: "Matrix 2 of 5", group: "1D Barcode", dim: "1d", default: "1234", w: 400, h: 100 },
  { id: "NEC2of5", label: "NEC 2 of 5", group: "1D Barcode", dim: "1d", default: "1234", w: 400, h: 100 },
  # GS1 DataBar
  { id: "GS1DataBar14", label: "GS1 DataBar 14", group: "GS1 DataBar", dim: "1d", default: "0123456789012", w: 300, h: 80 },
  { id: "GS1DataBarLimited", label: "GS1 DataBar Limited", group: "GS1 DataBar", dim: "1d", default: "0123456789012", w: 300, h: 80 },
  { id: "GS1DataBarExpanded", label: "GS1 DataBar Expanded", group: "GS1 DataBar", dim: "1d", default: "[01]90012345678908", w: 400, h: 80 },
  # JAN / UPC
  { id: "JAN13", label: "JAN-13 / EAN-13", group: "JAN / UPC", dim: "1d", default: "490123456789", w: 300, h: 100 },
  { id: "JAN8", label: "JAN-8 / EAN-8", group: "JAN / UPC", dim: "1d", default: "1234567", w: 250, h: 100 },
  { id: "UPCA", label: "UPC-A", group: "JAN / UPC", dim: "1d", default: "01234567890", w: 300, h: 100 },
  { id: "UPCE", label: "UPC-E", group: "JAN / UPC", dim: "1d", default: "0123456", w: 250, h: 100 },
].freeze

BARCODE_MAP = BARCODE_TYPES.each_with_object({}) { |bt, h| h[bt[:id]] = bt }.freeze

def parse_hex_color(hex)
  hex = hex.sub(/^#/, "")
  return [0, 0, 0] unless hex.length == 6
  [hex[0..1].to_i(16), hex[2..3].to_i(16), hex[4..5].to_i(16)]
end

def apply_colors(bc, prms)
  fore = prms[:fore_color] || ""
  back = prms[:back_color] || ""
  transparent = prms[:transparent_bg] == "1"

  if fore != "" && fore != "000000" && fore != "#000000"
    r, g, b = parse_hex_color(fore)
    bc.set_foreground_color(r, g, b, 255)
  end
  if transparent
    bc.set_background_color(0, 0, 0, 0)
  elsif back != "" && back != "FFFFFF" && back != "#FFFFFF" && back != "ffffff" && back != "#ffffff"
    r, g, b = parse_hex_color(back)
    bc.set_background_color(r, g, b, 255)
  end
end

def create_and_draw(type_id, code, format, width, height, prms)
  case type_id
  when "QR"
    bc = BarcodePao::QRCode.new(format)
    ecc = (prms[:qr_error_level] || "1").to_i
    bc.error_correction_level = ecc
    ver = (prms[:qr_version] || "0").to_i
    bc.version = ver if ver > 0
    apply_colors(bc, prms)
    bc.draw(code, width)
    bc

  when "DataMatrix"
    bc = BarcodePao::DataMatrix.new(format)
    sz = (prms[:datamatrix_size] || "-1").to_i
    bc.set_code_size(sz) if sz >= 0
    apply_colors(bc, prms)
    bc.draw(code, width)
    bc

  when "PDF417"
    bc = BarcodePao::PDF417.new(format)
    el = (prms[:pdf417_error_level] || "-1").to_i
    bc.set_error_level(el) if el >= 0 && el <= 8
    cols = (prms[:pdf417_cols] || "0").to_i
    bc.set_columns(cols) if cols > 0
    rows = (prms[:pdf417_rows] || "0").to_i
    bc.set_rows(rows) if rows > 0
    ar = (prms[:pdf417_aspect_ratio] || "0.5").to_f
    bc.aspect_ratio = ar if ar > 0
    apply_colors(bc, prms)
    bc.draw(code, width, height)
    bc

  when "YubinCustomer"
    bc = BarcodePao::YubinCustomer.new(format)
    apply_colors(bc, prms)
    bc.draw(code, height)
    bc

  when "GS1DataBar14"
    st_str = prms[:databar14_type] || "Omni"
    sym_type = case st_str
               when "Stacked"     then BarcodePao::STACKED
               when "StackedOmni" then BarcodePao::STACKED_OMNIDIRECTIONAL
               else                    BarcodePao::OMNIDIRECTIONAL
               end
    bc = BarcodePao::GS1DataBar14.new(format, sym_type)
    apply_colors(bc, prms)
    bc.draw(code, width, height)
    bc

  when "GS1DataBarLimited"
    bc = BarcodePao::GS1DataBarLimited.new(format)
    apply_colors(bc, prms)
    bc.draw(code, width, height)
    bc

  when "GS1DataBarExpanded"
    st_str = prms[:databar_expanded_type] || "Unstacked"
    sym_type = st_str == "Stacked" ? BarcodePao::STACKED_EXP : BarcodePao::UNSTACKED
    num_cols = (prms[:databar_expanded_cols] || "2").to_i
    bc = BarcodePao::GS1DataBarExpanded.new(format, sym_type, num_cols)
    apply_colors(bc, prms)
    bc.draw(code, width, height)
    bc

  else
    # Standard 1D types
    bc = case type_id
         when "Code128"   then BarcodePao::Code128.new(format)
         when "Code39"    then BarcodePao::Code39.new(format)
         when "Code93"    then BarcodePao::Code93.new(format)
         when "NW7"       then BarcodePao::NW7.new(format)
         when "ITF"       then BarcodePao::ITF.new(format)
         when "Matrix2of5" then BarcodePao::Matrix2of5.new(format)
         when "NEC2of5"   then BarcodePao::NEC2of5.new(format)
         when "JAN13"     then BarcodePao::JAN13.new(format)
         when "JAN8"      then BarcodePao::JAN8.new(format)
         when "UPCA"      then BarcodePao::UPCA.new(format)
         when "UPCE"      then BarcodePao::UPCE.new(format)
         when "GS1_128"   then BarcodePao::GS1128.new(format)
         else raise "Unknown barcode type: #{type_id}"
         end
    bc.show_text = (prms[:show_text] != "0")
    bc.text_even_spacing = (prms[:even_spacing] != "0")
    # Extended guard for JAN/UPC
    if %w[JAN13 JAN8 UPCA UPCE].include?(type_id)
      bc.extended_guard = (prms[:extended_guard] == "1")
    end
    # Show start/stop for Code39/NW7
    if %w[Code39 NW7].include?(type_id) && prms.key?(:disp_start_stop)
      bc.show_start_stop = (prms[:disp_start_stop] != "0")
    end
    apply_colors(bc, prms)
    bc.draw(code, width, height)
    bc
  end
end

def create_barcode_for_canvas(type_id, format, show_text)
  case type_id
  when "QR"            then BarcodePao::QRCode.new(format)
  when "DataMatrix"    then BarcodePao::DataMatrix.new(format)
  when "PDF417"        then BarcodePao::PDF417.new(format)
  when "YubinCustomer" then BarcodePao::YubinCustomer.new(format)
  when "GS1DataBar14"  then BarcodePao::GS1DataBar14.new(format)
  when "GS1DataBarLimited" then BarcodePao::GS1DataBarLimited.new(format)
  when "GS1DataBarExpanded" then BarcodePao::GS1DataBarExpanded.new(format)
  else
    bc = case type_id
         when "Code128"   then BarcodePao::Code128.new(format)
         when "Code39"    then BarcodePao::Code39.new(format)
         when "Code93"    then BarcodePao::Code93.new(format)
         when "NW7"       then BarcodePao::NW7.new(format)
         when "ITF"       then BarcodePao::ITF.new(format)
         when "Matrix2of5" then BarcodePao::Matrix2of5.new(format)
         when "NEC2of5"   then BarcodePao::NEC2of5.new(format)
         when "JAN13"     then BarcodePao::JAN13.new(format)
         when "JAN8"      then BarcodePao::JAN8.new(format)
         when "UPCA"      then BarcodePao::UPCA.new(format)
         when "UPCE"      then BarcodePao::UPCE.new(format)
         when "GS1_128"   then BarcodePao::GS1128.new(format)
         else BarcodePao::QRCode.new(format)
         end
    bc.show_text = show_text
    bc
  end
end

def draw_barcode_for_canvas(type_id, code, width, height)
  bc = create_barcode_for_canvas(type_id, BarcodePao::FORMAT_PNG, true)
  info = BARCODE_MAP[type_id] || BARCODE_MAP["QR"]
  case info[:dim]
  when "2d"
    if type_id == "PDF417"
      bc.draw(code, width, height)
    else
      bc.draw(code, width)
    end
  when "postal"
    bc.draw(code, height)
  else
    bc.draw(code, width, height)
  end
  bc.get_image_memory
end

def get_sample_codes(type_id)
  case type_id
  when "QR"            then ["Hello World", "https://www.pao.ac", "SAMPLE-001", "TEST DATA"]
  when "DataMatrix"    then ["DataMatrix-1", "ABCDE12345", "Test-DM", "PAO"]
  when "PDF417"        then ["PDF417-Test", "ABCDEFGHIJ", "12345", "PAO"]
  when "Code128"       then ["Hello-123", "ABC-2026", "Test-Code128", "BARCODE"]
  when "Code39"        then ["HELLO-123", "ABC-2026", "TEST", "CODE39"]
  when "Code93"        then ["CODE93", "TEST-93", "ABC123", "SAMPLE"]
  when "NW7"           then ["A123456B", "A999999A", "B12345B", "C00001D"]
  when "ITF"           then ["123456", "000000", "111111", "999999"]
  when "Matrix2of5"    then ["1234", "5678", "0000", "9999"]
  when "NEC2of5"       then ["1234", "5678", "0000", "9999"]
  when "JAN13"         then ["4912345123459", "4901234567890", "4567890123456", "1234567890128"]
  when "JAN8"          then ["1234567", "0000000", "9999999", "4567890"]
  when "UPCA"          then ["01234567890", "12345678901", "99999999999", "00000000000"]
  when "UPCE"          then ["0123456", "0000000", "0999999", "0123450"]
  when "GS1_128"       then ["[01]04912345123459", "[01]09501234567891", "[01]04567890123450", "[01]01234567890128"]
  when "GS1DataBar14"  then ["0123456789012", "9999999999999", "0000000000000", "1234567890123"]
  when "GS1DataBarLimited" then ["0123456789012", "0100000000000", "0999999999999", "0123456789012"]
  when "GS1DataBarExpanded" then ["[01]90012345678908", "[01]95012345678903", "[01]98012345678902", "[01]90000000000003"]
  when "YubinCustomer" then ["1060032", "1000001", "5300001", "6008799"]
  else ["TEST1", "TEST2", "TEST3", "TEST4"]
  end
end

# --- ChunkyPNG helpers ---
def fill_rect(canvas, x, y, w, h, color)
  x1 = [x, 0].max
  y1 = [y, 0].max
  x2 = [x + w - 1, canvas.width - 1].min
  y2 = [y + h - 1, canvas.height - 1].min
  (y1..y2).each do |py|
    (x1..x2).each do |px|
      canvas.compose_pixel(px, py, color)
    end
  end
end

def fill_circle(canvas, cx, cy, r, color)
  (cy - r..cy + r).each do |py|
    next if py < 0 || py >= canvas.height
    (cx - r..cx + r).each do |px|
      next if px < 0 || px >= canvas.width
      canvas.compose_pixel(px, py, color) if (px - cx)**2 + (py - cy)**2 <= r**2
    end
  end
end

def fill_ellipse(canvas, cx, cy, rx, ry, color)
  return if rx <= 0 || ry <= 0
  (cy - ry..cy + ry).each do |py|
    next if py < 0 || py >= canvas.height
    (cx - rx..cx + rx).each do |px|
      next if px < 0 || px >= canvas.width
      dx = (px - cx).to_f / rx
      dy = (py - cy).to_f / ry
      canvas.compose_pixel(px, py, color) if dx * dx + dy * dy <= 1.0
    end
  end
end

def draw_elephant(canvas, ex, ey, s)
  fill_ellipse(canvas, ex, ey, (50*s).to_i, (30*s).to_i, ChunkyPNG::Color.rgba(140, 155, 175, 230))
  fill_ellipse(canvas, (ex+45*s).to_i, (ey-20*s).to_i, (25*s).to_i, (22*s).to_i, ChunkyPNG::Color.rgba(140, 155, 175, 230))
  fill_ellipse(canvas, (ex+60*s).to_i, (ey-30*s).to_i, (18*s).to_i, (20*s).to_i, ChunkyPNG::Color.rgba(120, 135, 160, 200))
  fill_circle(canvas, (ex+45*s).to_i, (ey-28*s).to_i, (4*s).to_i, ChunkyPNG::Color::WHITE)
  fill_circle(canvas, (ex+46*s).to_i, (ey-27*s).to_i, (2*s).to_i, ChunkyPNG::Color.rgba(20, 20, 40, 255))
  [-25, -5, 20, 40].each do |lx|
    fill_rect(canvas, (ex + lx*s).to_i, (ey+25*s).to_i, (14*s).to_i, (28*s).to_i, ChunkyPNG::Color.rgba(130, 145, 170, 220))
  end
end

def draw_flying_elephant(canvas, ex, ey, s)
  fill_ellipse(canvas, ex, ey, (35*s).to_i, (20*s).to_i, ChunkyPNG::Color.rgba(160, 175, 200, 200))
  fill_ellipse(canvas, (ex+30*s).to_i, (ey-10*s).to_i, (18*s).to_i, (16*s).to_i, ChunkyPNG::Color.rgba(160, 175, 200, 200))
  fill_ellipse(canvas, (ex-10*s).to_i, (ey-25*s).to_i, (25*s).to_i, (12*s).to_i, ChunkyPNG::Color.rgba(200, 210, 240, 160))
  fill_ellipse(canvas, (ex+10*s).to_i, (ey-28*s).to_i, (25*s).to_i, (12*s).to_i, ChunkyPNG::Color.rgba(200, 210, 240, 160))
  fill_circle(canvas, (ex+32*s).to_i, (ey-15*s).to_i, (3*s).to_i, ChunkyPNG::Color::WHITE)
  fill_circle(canvas, (ex+33*s).to_i, (ey-14*s).to_i, (1.5*s).to_i, ChunkyPNG::Color.rgba(20, 20, 40, 255))
end

# --- Routes ---

get "/" do
  @barcode_types_json = BARCODE_TYPES.to_json
  erb :index
end

post "/draw-base64" do
  content_type :json
  type_id = params[:type] || "QR"
  info = BARCODE_MAP[type_id] || BARCODE_MAP["QR"]
  type_id = info[:id]
  code = params[:code]
  code = info[:default] if code.nil? || code.empty?
  width = (params[:width] || info[:w]).to_i
  height = (params[:height] || info[:h]).to_i
  begin
    bc = create_and_draw(type_id, code, BarcodePao::FORMAT_PNG, width, height, params)
    { ok: true, base64: bc.get_image_base64 }.to_json
  rescue => e
    { ok: false, error: e.message }.to_json
  end
end

post "/draw-svg" do
  content_type :json
  type_id = params[:type] || "QR"
  info = BARCODE_MAP[type_id] || BARCODE_MAP["QR"]
  type_id = info[:id]
  code = params[:code]
  code = info[:default] if code.nil? || code.empty?
  width = (params[:width] || info[:w]).to_i
  height = (params[:height] || info[:h]).to_i
  begin
    bc = create_and_draw(type_id, code, BarcodePao::FORMAT_SVG, width, height, params)
    { ok: true, svg: bc.get_svg }.to_json
  rescue => e
    { ok: false, error: e.message }.to_json
  end
end

post "/draw-canvas" do
  content_type :json
  require "chunky_png"

  type_id = params[:type] || "QR"
  info = BARCODE_MAP[type_id] || BARCODE_MAP["QR"]
  type_id = info[:id]
  code = params[:code]
  code = info[:default] if code.nil? || code.empty?
  width = (params[:width] || info[:w]).to_i
  height = (params[:height] || info[:h]).to_i
  pos_x = (params[:x] || "0").to_i
  pos_y = (params[:y] || "0").to_i
  show_text = params[:show_text] != "0"

  begin
    cw, ch = 780, 680
    canvas = ChunkyPNG::Image.new(cw, ch, ChunkyPNG::Color::BLACK)

    # Sunset gradient sky
    ch.times do |i|
      t = i.to_f / ch
      r = (180 - 120 * t).to_i
      g = (100 + 60 * t).to_i
      b = (60 + 140 * t).to_i
      cw.times { |x| canvas[x, i] = ChunkyPNG::Color.rgba(r, g, b, 255) }
    end

    # Sun
    fill_circle(canvas, cw - 120, 80, 60, ChunkyPNG::Color.rgba(255, 200, 50, 180))
    fill_circle(canvas, cw - 120, 80, 90, ChunkyPNG::Color.rgba(255, 220, 100, 100))

    # Clouds
    rng = Random.new(42)
    5.times do
      cx = rng.rand(cw - 100) + 50
      cy = rng.rand(150) + 30
      fill_ellipse(canvas, cx, cy, 50 + rng.rand(30), 15 + rng.rand(10), ChunkyPNG::Color.rgba(255, 255, 255, 120 + rng.rand(60)))
      fill_ellipse(canvas, cx + 30, cy - 5, 40, 12, ChunkyPNG::Color.rgba(255, 255, 255, 140))
    end

    # Green grass
    grass_y = (ch * 0.72).to_i
    fill_rect(canvas, 0, grass_y, cw, ch - grass_y, ChunkyPNG::Color.rgba(80, 160, 60, 255))

    # Flowers
    flower_colors = [
      ChunkyPNG::Color.rgba(255, 100, 100, 200),
      ChunkyPNG::Color.rgba(255, 200, 50, 200),
      ChunkyPNG::Color.rgba(200, 100, 255, 200),
      ChunkyPNG::Color.rgba(255, 150, 200, 200),
    ]
    20.times do
      fx = rng.rand(cw)
      fy = grass_y + rng.rand(ch - grass_y)
      sz = 3 + rng.rand(4)
      fill_circle(canvas, fx, fy, sz, flower_colors[rng.rand(flower_colors.length)])
    end

    # Walking elephant
    draw_elephant(canvas, 100, grass_y - 40, 1.0)

    # Flying elephant
    draw_flying_elephant(canvas, cw - 200, 160, 0.6)

    # Barcode
    bx = pos_x
    by = pos_y
    if bx == 0 && by == 0
      bx = cw / 2 - width / 2
      by = grass_y - height - 30
    end

    # White card background
    fill_rect(canvas, bx - 10, by - 10, width + 20, height + 20, ChunkyPNG::Color.rgba(255, 255, 255, 220))

    # Generate and compose barcode
    bc_bytes = draw_barcode_for_canvas(type_id, code, width, height)
    if bc_bytes && !bc_bytes.empty?
      bc_img = ChunkyPNG::Image.from_blob(bc_bytes)
      canvas.compose!(bc_img, bx, by)
    end

    png_blob = canvas.to_blob
    b64 = Base64.strict_encode64(png_blob)
    { ok: true, base64: "data:image/png;base64,#{b64}" }.to_json
  rescue => e
    { ok: false, error: e.message }.to_json
  end
end

get "/pdf" do
  require "prawn"

  type_id = params[:type] || "QR"
  info = BARCODE_MAP[type_id] || BARCODE_MAP["QR"]
  type_id = info[:id]
  code = params[:code]
  code = info[:default] if code.nil? || code.empty?

  begin
    pdf_bytes = generate_report_pdf(type_id, code, info)
    content_type "application/pdf"
    headers "Content-Disposition" => "inline; filename=barcode_report.pdf"
    pdf_bytes
  rescue => e
    content_type :json
    status 500
    { ok: false, error: e.message }.to_json
  end
end

def generate_report_pdf(type_id, code, info)
  require "prawn"

  pdf = Prawn::Document.new(page_size: "A4", margin: 0)
  pw = 595.28
  ph = 841.89
  today = Date.today.iso8601

  # Header
  pdf.fill_color "1E40AF"
  pdf.fill_rectangle [0, ph], pw, 70
  pdf.fill_color "7C3AED"
  pdf.fill_rectangle [0, ph - 70], pw, 4

  pdf.fill_color "FFFFFF"
  pdf.font_size 20
  pdf.draw_text "BARCODE REPORT", at: [40, ph - 45]
  pdf.font_size 9
  pdf.draw_text "#{info[:label]} | #{type_id}", at: [40, ph - 60]
  pdf.draw_text today, at: [pw - 120, ph - 30]
  pdf.draw_text "Pao@Office", at: [pw - 120, ph - 43]
  pdf.draw_text "pao.ac", at: [pw - 120, ph - 56]

  # Data section
  pdf.fill_color "475569"
  pdf.font_size 9
  pdf.draw_text "Data: #{code}", at: [40, ph - 95]

  # Main barcode
  y = ph - 115
  is2d = info[:dim] == "2d"
  if is2d
    bw = type_id == "PDF417" ? 300 : 200
    bh = type_id == "PDF417" ? 120 : 200
  elsif info[:dim] == "postal"
    bw, bh = 300, 40
  else
    bw, bh = 300, 120
  end

  bc_bytes = draw_barcode_for_canvas(type_id, code, bw, bh)
  if bc_bytes && !bc_bytes.empty?
    pdf.image StringIO.new(bc_bytes), at: [40, y], width: bw
  end

  # Sample table
  y -= bh + 30
  pdf.fill_color "1E40AF"
  pdf.fill_rectangle [40, y], pw - 80, 20
  pdf.fill_color "FFFFFF"
  pdf.font_size 8
  pdf.draw_text "Barcode", at: [50, y - 14]
  pdf.draw_text "Data", at: [300, y - 14]
  y -= 20

  samples = get_sample_codes(type_id)
  samples.each_with_index do |samp, i|
    ry = y - i * 55
    if i.even?
      pdf.fill_color "F8FAFC"
      pdf.fill_rectangle [40, ry], pw - 80, 55
    end

    sbc_bytes = draw_barcode_for_canvas(type_id, samp, is2d ? 42 : 190, is2d ? 42 : 32)
    if sbc_bytes && !sbc_bytes.empty?
      pdf.image StringIO.new(sbc_bytes), at: [50, ry - 5], width: (is2d ? 42 : 190)
    end

    pdf.fill_color "000000"
    pdf.font_size 8
    pdf.draw_text samp, at: [300, ry - 30]
  end

  # Footer
  pdf.stroke_color "E2E8F0"
  pdf.line_width 0.5
  pdf.stroke_line [40, 45], [pw - 40, 45]
  pdf.fill_color "94A3B8"
  pdf.font_size 7
  pdf.draw_text "Generated by Barcode.Ruby — barcode_pao + Prawn", at: [40, 32]
  pdf.draw_text "pao.ac", at: [pw - 60, 32]

  pdf.render
end

__END__

@@ index
<!DOCTYPE html>
<html lang="ja">
<head>
<meta charset="UTF-8">
<meta name="viewport" content="width=device-width, initial-scale=1.0">
<title>Barcode.Ruby - All-in-One</title>
<style>
*{margin:0;padding:0;box-sizing:border-box}
body{font-family:'Segoe UI',system-ui,sans-serif;background:#f0f4f8;color:#1e293b}
header{background:linear-gradient(135deg,#CC342D,#8B0000);color:#fff;padding:20px 32px}
header h1{font-size:20px;font-weight:700}
header p{font-size:12px;opacity:.85;margin-top:3px}
.badge{display:inline-block;background:rgba(255,255,255,.2);border-radius:12px;padding:2px 10px;font-size:11px;margin-top:4px}

.container{display:flex;gap:16px;max-width:1100px;margin:16px auto;padding:0 16px}
.left-panel{flex:1;min-width:0}
.right-panel{width:400px;flex-shrink:0}

.card{background:#fff;border-radius:10px;padding:16px 18px;box-shadow:0 1px 3px rgba(0,0,0,.08);margin-bottom:12px}
.section-title{font-size:13px;font-weight:700;color:#CC342D;margin-bottom:10px;padding-left:10px;border-left:3px solid #CC342D}
.section-title.green{color:#065f46;border-color:#10b981}
.section-title.blue{color:#1e40af;border-color:#3b82f6}

label{display:block;font-size:12px;font-weight:600;color:#475569;margin-bottom:4px}
select,input[type=text],input[type=number],input[type=color]{width:100%;padding:8px 10px;border:1.5px solid #e2e8f0;border-radius:6px;font-size:13px}
select:focus,input:focus{outline:none;border-color:#CC342D}
input[type=color]{height:36px;padding:2px 4px;cursor:pointer}

.form-row{display:flex;gap:10px;margin-bottom:10px}
.form-row>div{flex:1}

.mode-row{display:flex;gap:6px;margin:10px 0}
.mode-btn{flex:1;padding:8px 6px;border:1.5px solid #e2e8f0;border-radius:6px;background:#fff;cursor:pointer;font-size:11px;font-weight:600;text-align:center;color:#64748b;transition:all .15s}
.mode-btn.active{border-color:#CC342D;background:#fef2f2;color:#CC342D}
.mode-btn:hover{border-color:#f87171}

.gen-btn{width:100%;padding:10px;border:none;border-radius:8px;background:linear-gradient(135deg,#CC342D,#8B0000);color:#fff;font-size:14px;font-weight:700;cursor:pointer;margin-top:8px;transition:opacity .2s}
.gen-btn:hover{opacity:.9}

.specific-frame{border:1.5px solid #d1fae5;border-radius:8px;padding:12px;margin-top:6px}
.specific-frame[hidden]{display:none}

.preview-card{background:#fff;border-radius:10px;box-shadow:0 1px 3px rgba(0,0,0,.08);position:sticky;top:16px}
.preview-header{padding:12px 14px;border-bottom:1px solid #f1f5f9;display:flex;align-items:center;justify-content:space-between}
.preview-header h3{font-size:13px;color:#CC342D}
.preview-body{padding:16px;display:flex;align-items:center;justify-content:center;min-height:320px;overflow:auto}
.preview-body img{max-width:100%;image-rendering:pixelated}
.preview-body .svg-wrap svg{max-width:100%;height:auto}
.preview-body iframe{width:100%;height:500px;border:1px solid #e2e8f0;border-radius:6px}
.empty-msg{color:#94a3b8;font-size:12px;text-align:center}
.error-msg{color:#ef4444;font-size:12px;text-align:center}

.svg-source-box{margin-top:8px;background:#f8fafc;border:1px solid #e2e8f0;border-radius:6px;padding:8px;max-height:150px;overflow:auto;font-family:monospace;font-size:10px;color:#475569;display:none}

.code-panel{background:#1e293b;border-radius:6px;padding:10px 12px;margin-top:8px}
.code-panel pre{font-family:'Consolas',monospace;font-size:11px;color:#e2e8f0;line-height:1.6;white-space:pre-wrap}

footer{text-align:center;padding:16px;font-size:11px;color:#94a3b8}
@media(max-width:900px){.container{flex-direction:column}.right-panel{width:100%}}
</style>
</head>
<body>
<header>
  <h1>Barcode.Ruby &#x2014; All-in-One</h1>
  <p>Pure Ruby barcode library &#x2014; 18 barcode types</p>
  <span class="badge">Ruby + Sinatra</span>
</header>

<div class="container">
  <div class="left-panel">
    <!-- Main Settings -->
    <div class="card">
      <div class="section-title">Barcode Settings</div>

      <div class="form-row">
        <div style="flex:2">
          <label>Type</label>
          <select id="sel-type" onchange="onTypeChange()"></select>
        </div>
        <div>
          <label>Width</label>
          <input type="number" id="inp-width" value="300" min="10" max="2000">
        </div>
        <div>
          <label>Height</label>
          <input type="number" id="inp-height" value="100" min="10" max="2000">
        </div>
      </div>

      <label>Data</label>
      <input type="text" id="inp-code" placeholder="Enter barcode data...">

      <div class="form-row" style="margin-top:10px">
        <div>
          <label><input type="checkbox" id="chk-show-text" checked> Show Text</label>
        </div>
        <div>
          <label><input type="checkbox" id="chk-even-spacing" checked> Even Spacing</label>
        </div>
      </div>

      <!-- Position -->
      <div class="form-row">
        <div>
          <label>X</label>
          <input type="number" id="inp-x" value="0" min="0">
        </div>
        <div>
          <label>Y</label>
          <input type="number" id="inp-y" value="0" min="0">
        </div>
        <div>
          <label>Font Size</label>
          <input type="number" id="inp-font-size" value="10" min="6" max="24">
        </div>
      </div>

      <!-- Colors -->
      <div class="form-row">
        <div>
          <label>Bar Color</label>
          <input type="color" id="inp-fore-color" value="#000000">
        </div>
        <div>
          <label>Background</label>
          <input type="color" id="inp-back-color" value="#FFFFFF">
        </div>
        <div style="display:flex;align-items:flex-end;padding-bottom:8px">
          <label><input type="checkbox" id="chk-transparent"> Transparent BG</label>
        </div>
      </div>

      <!-- Extended Guard / Start-Stop -->
      <div class="form-row" id="row-extended-guard" style="display:none">
        <div>
          <label><input type="checkbox" id="chk-extended-guard"> Extended Guard Bars</label>
        </div>
      </div>
      <div class="form-row" id="row-start-stop" style="display:none">
        <div>
          <label><input type="checkbox" id="chk-start-stop" checked> Show Start/Stop</label>
        </div>
      </div>

      <div class="mode-row">
        <button class="mode-btn active" data-mode="base64" onclick="setMode('base64')">PNG (Base64)</button>
        <button class="mode-btn" data-mode="svg" onclick="setMode('svg')">SVG</button>
        <button class="mode-btn" data-mode="pdf" onclick="setMode('pdf')">PDF</button>
        <button class="mode-btn" data-mode="canvas" onclick="setMode('canvas')">Canvas</button>
      </div>

      <button class="gen-btn" onclick="generate()">Generate</button>
    </div>

    <!-- Type-Specific Settings -->
    <div class="card">
      <div class="section-title green">Type-Specific Settings</div>

      <div class="specific-frame" id="frame-qr">
        <div class="form-row">
          <div>
            <label>Error Correction</label>
            <select id="qr-ecc">
              <option value="0">L (7%)</option>
              <option value="1" selected>M (15%)</option>
              <option value="2">Q (25%)</option>
              <option value="3">H (30%)</option>
            </select>
          </div>
          <div>
            <label>Version (0=Auto)</label>
            <input type="number" id="qr-version" value="0" min="0" max="40">
          </div>
        </div>
      </div>

      <div class="specific-frame" id="frame-datamatrix" hidden>
        <label>Symbol Size</label>
        <select id="dm-size">
          <option value="-1" selected>Auto</option>
          <option value="0">10x10</option><option value="1">12x12</option>
          <option value="2">14x14</option><option value="3">16x16</option>
          <option value="4">18x18</option><option value="5">20x20</option>
          <option value="6">22x22</option><option value="7">24x24</option>
          <option value="8">26x26</option><option value="9">32x32</option>
          <option value="10">36x36</option><option value="11">40x40</option>
          <option value="12">44x44</option><option value="13">48x48</option>
          <option value="14">52x52</option><option value="15">64x64</option>
          <option value="16">72x72</option><option value="17">80x80</option>
          <option value="18">88x88</option><option value="19">96x96</option>
          <option value="20">104x104</option><option value="21">120x120</option>
          <option value="22">132x132</option><option value="23">144x144</option>
          <option value="24">8x18</option><option value="25">8x32</option>
          <option value="26">12x26</option><option value="27">12x36</option>
          <option value="28">16x36</option><option value="29">16x48</option>
        </select>
      </div>

      <div class="specific-frame" id="frame-pdf417" hidden>
        <div class="form-row">
          <div>
            <label>Error Level (-1=Auto)</label>
            <input type="number" id="pdf417-error" value="-1" min="-1" max="8">
          </div>
          <div>
            <label>Columns (0=Auto)</label>
            <input type="number" id="pdf417-cols" value="0" min="0" max="30">
          </div>
        </div>
        <div class="form-row">
          <div>
            <label>Rows (0=Auto)</label>
            <input type="number" id="pdf417-rows" value="0" min="0" max="90">
          </div>
          <div>
            <label>Aspect Ratio</label>
            <select id="pdf417-aspect">
              <option value="0.5" selected>0.5</option>
              <option value="1.0">1.0</option>
              <option value="2.0">2.0</option>
            </select>
          </div>
        </div>
      </div>

      <div class="specific-frame" id="frame-databar14" hidden>
        <label>Symbol Type</label>
        <select id="databar14-type">
          <option value="Omni">Omnidirectional</option>
          <option value="Stacked">Stacked</option>
          <option value="StackedOmni">Stacked Omnidirectional</option>
        </select>
      </div>

      <div class="specific-frame" id="frame-databar-exp" hidden>
        <div class="form-row">
          <div>
            <label>Symbol Type</label>
            <select id="databar-exp-type">
              <option value="Unstacked">Unstacked</option>
              <option value="Stacked">Stacked</option>
            </select>
          </div>
          <div>
            <label>Columns</label>
            <input type="number" id="databar-exp-cols" value="2" min="1" max="11">
          </div>
        </div>
      </div>

      <div id="no-specific" style="color:#94a3b8;font-size:12px;padding:8px 0">
        No type-specific settings for this barcode type.
      </div>
    </div>

    <!-- Code Panel -->
    <div class="card">
      <div class="section-title blue">Ruby Code</div>
      <div class="code-panel">
        <pre id="code-display"></pre>
      </div>
    </div>
  </div>

  <div class="right-panel">
    <div class="preview-card">
      <div class="preview-header">
        <h3>Preview</h3>
        <span id="mode-badge" style="font-size:11px;color:#64748b">PNG</span>
      </div>
      <div class="preview-body" id="preview-body">
        <div class="empty-msg">Select a barcode type and click Generate</div>
      </div>
      <div class="svg-source-box" id="svg-source-box"></div>
    </div>
  </div>
</div>

<footer>Barcode.Ruby &#x2014; Pure Ruby barcode library &#x2014; pao.ac</footer>

<script>
const TYPES = <%= @barcode_types_json %>;
const typeMap = {};
TYPES.forEach(t => typeMap[t.id] = t);

let currentMode = 'base64';

// Populate select
const selType = document.getElementById('sel-type');
let lastGroup = '';
let optgroup;
TYPES.forEach(t => {
  if (t.group !== lastGroup) {
    optgroup = document.createElement('optgroup');
    optgroup.label = t.group;
    selType.appendChild(optgroup);
    lastGroup = t.group;
  }
  const opt = document.createElement('option');
  opt.value = t.id;
  opt.textContent = t.label;
  optgroup.appendChild(opt);
});

function onTypeChange() {
  const t = typeMap[selType.value];
  if (!t) return;
  document.getElementById('inp-code').value = t.default;
  document.getElementById('inp-width').value = t.w;
  document.getElementById('inp-height').value = t.h;

  // Show/hide specific frames
  const frames = ['frame-qr', 'frame-datamatrix', 'frame-pdf417', 'frame-databar14', 'frame-databar-exp'];
  frames.forEach(f => document.getElementById(f).hidden = true);
  const noSpecific = document.getElementById('no-specific');
  noSpecific.style.display = '';

  if (t.id === 'QR') { document.getElementById('frame-qr').hidden = false; noSpecific.style.display = 'none'; }
  else if (t.id === 'DataMatrix') { document.getElementById('frame-datamatrix').hidden = false; noSpecific.style.display = 'none'; }
  else if (t.id === 'PDF417') { document.getElementById('frame-pdf417').hidden = false; noSpecific.style.display = 'none'; }
  else if (t.id === 'GS1DataBar14') { document.getElementById('frame-databar14').hidden = false; noSpecific.style.display = 'none'; }
  else if (t.id === 'GS1DataBarExpanded') { document.getElementById('frame-databar-exp').hidden = false; noSpecific.style.display = 'none'; }

  // Show/hide 1D settings
  const is1d = t.dim === '1d';
  document.getElementById('chk-show-text').parentElement.parentElement.parentElement.style.display = is1d ? '' : 'none';
  document.getElementById('chk-even-spacing').parentElement.parentElement.parentElement.style.display = is1d ? '' : 'none';

  // Extended guard for JAN/UPC
  const hasGuard = ['JAN13', 'JAN8', 'UPCA', 'UPCE'].includes(t.id);
  document.getElementById('row-extended-guard').style.display = hasGuard ? '' : 'none';

  // Start/stop for Code39/NW7
  const hasStartStop = ['Code39', 'NW7'].includes(t.id);
  document.getElementById('row-start-stop').style.display = hasStartStop ? '' : 'none';

  updateCodePanel();
}

function setMode(mode) {
  currentMode = mode;
  document.querySelectorAll('.mode-btn').forEach(b => b.classList.remove('active'));
  document.querySelector(`[data-mode="${mode}"]`).classList.add('active');
  const labels = { base64: 'PNG', svg: 'SVG', pdf: 'PDF', canvas: 'Canvas' };
  document.getElementById('mode-badge').textContent = labels[mode];
  document.getElementById('svg-source-box').style.display = 'none';
  updateCodePanel();
}

function getRubyClass(typeID) {
  const map = {
    'QR': 'QRCode', 'DataMatrix': 'DataMatrix', 'PDF417': 'PDF417',
    'Code128': 'Code128', 'Code39': 'Code39', 'Code93': 'Code93',
    'NW7': 'NW7', 'ITF': 'ITF', 'Matrix2of5': 'Matrix2of5',
    'NEC2of5': 'NEC2of5', 'JAN13': 'JAN13', 'JAN8': 'JAN8',
    'UPCA': 'UPCA', 'UPCE': 'UPCE', 'GS1_128': 'GS1128',
    'GS1DataBar14': 'GS1DataBar14', 'GS1DataBarLimited': 'GS1DataBarLimited',
    'GS1DataBarExpanded': 'GS1DataBarExpanded', 'YubinCustomer': 'YubinCustomer',
  };
  return map[typeID] || typeID;
}

function updateCodePanel() {
  const typeID = selType.value;
  const t = typeMap[typeID];
  const cls = getRubyClass(typeID);
  const is2d = t.dim === '2d';

  let code = '';
  if (currentMode === 'canvas') {
    code = `require "chunky_png"\n\ncanvas = ChunkyPNG::Image.new(780, 680)\n# ... draw background ...\n`;
    const fmt = '"png"';
    code += `bc = BarcodePao::${cls}.new(${fmt})\n`;
    if (is2d && typeID !== 'PDF417') code += `bc.draw(code, ${t.w})\n`;
    else if (typeID === 'YubinCustomer') code += `bc.draw(code, ${t.h})\n`;
    else code += `bc.draw(code, ${t.w}, ${t.h})\n`;
    code += `bc_img = ChunkyPNG::Image.from_blob(bc.get_image_memory)\ncanvas.compose!(bc_img, x, y)`;
  } else if (currentMode === 'pdf') {
    code = `require "prawn"\n\npdf = Prawn::Document.new\n`;
    code += `bc = BarcodePao::${cls}.new("png")\n`;
    if (is2d && typeID !== 'PDF417') code += `bc.draw(code, ${t.w})\n`;
    else if (typeID === 'YubinCustomer') code += `bc.draw(code, ${t.h})\n`;
    else code += `bc.draw(code, ${t.w}, ${t.h})\n`;
    code += `pdf.image StringIO.new(bc.get_image_memory), width: ${t.w}\npdf.render_file("output.pdf")`;
  } else {
    const fmt = currentMode === 'svg' ? '"svg"' : '"png"';
    code = `bc = BarcodePao::${cls}.new(${fmt})\n`;
    if (is2d && typeID !== 'PDF417') code += `bc.draw(code, ${t.w})\n`;
    else if (typeID === 'YubinCustomer') code += `bc.draw(code, ${t.h})\n`;
    else code += `bc.draw(code, ${t.w}, ${t.h})\n`;
    if (currentMode === 'svg') code += `svg = bc.get_svg`;
    else code += `b64 = bc.get_image_base64`;
  }
  document.getElementById('code-display').textContent = code;
}

function buildParams() {
  const p = new URLSearchParams();
  p.set('type', selType.value);
  p.set('code', document.getElementById('inp-code').value.trim());
  p.set('width', document.getElementById('inp-width').value);
  p.set('height', document.getElementById('inp-height').value);
  p.set('show_text', document.getElementById('chk-show-text').checked ? '1' : '0');
  p.set('even_spacing', document.getElementById('chk-even-spacing').checked ? '1' : '0');
  p.set('x', document.getElementById('inp-x').value);
  p.set('y', document.getElementById('inp-y').value);
  p.set('font_size', document.getElementById('inp-font-size').value);

  const foreColor = document.getElementById('inp-fore-color').value.replace('#', '');
  const backColor = document.getElementById('inp-back-color').value.replace('#', '');
  p.set('fore_color', foreColor);
  p.set('back_color', backColor);
  p.set('transparent_bg', document.getElementById('chk-transparent').checked ? '1' : '0');

  // Extended guard
  if (document.getElementById('row-extended-guard').style.display !== 'none') {
    p.set('extended_guard', document.getElementById('chk-extended-guard').checked ? '1' : '0');
  }
  // Start/stop
  if (document.getElementById('row-start-stop').style.display !== 'none') {
    p.set('disp_start_stop', document.getElementById('chk-start-stop').checked ? '1' : '0');
  }

  const typeID = selType.value;
  if (typeID === 'QR') {
    p.set('qr_error_level', document.getElementById('qr-ecc').value);
    p.set('qr_version', document.getElementById('qr-version').value);
  } else if (typeID === 'DataMatrix') {
    p.set('datamatrix_size', document.getElementById('dm-size').value);
  } else if (typeID === 'PDF417') {
    p.set('pdf417_error_level', document.getElementById('pdf417-error').value);
    p.set('pdf417_cols', document.getElementById('pdf417-cols').value);
    p.set('pdf417_rows', document.getElementById('pdf417-rows').value);
    p.set('pdf417_aspect_ratio', document.getElementById('pdf417-aspect').value);
  } else if (typeID === 'GS1DataBar14') {
    p.set('databar14_type', document.getElementById('databar14-type').value);
  } else if (typeID === 'GS1DataBarExpanded') {
    p.set('databar_expanded_type', document.getElementById('databar-exp-type').value);
    p.set('databar_expanded_cols', document.getElementById('databar-exp-cols').value);
  }
  return p;
}

function generate() {
  const code = document.getElementById('inp-code').value.trim();
  if (!code) {
    document.getElementById('preview-body').innerHTML = '<div class="empty-msg">Enter data first</div>';
    return;
  }

  const params = buildParams();
  const preview = document.getElementById('preview-body');
  const svgBox = document.getElementById('svg-source-box');
  svgBox.style.display = 'none';

  if (currentMode === 'pdf') {
    preview.innerHTML = `<iframe src="pdf?${params}"></iframe>`;
    return;
  }

  preview.innerHTML = '<div class="empty-msg">Generating...</div>';

  const endpoint = currentMode === 'svg' ? 'draw-svg' : currentMode === 'canvas' ? 'draw-canvas' : 'draw-base64';
  fetch(endpoint, {
    method: 'POST',
    headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
    body: params.toString()
  })
    .then(r => r.json())
    .then(data => {
      if (!data.ok) {
        preview.innerHTML = `<div class="error-msg">${data.error}</div>`;
        return;
      }
      if (currentMode === 'svg') {
        preview.innerHTML = `<div class="svg-wrap">${data.svg}</div>`;
        svgBox.textContent = data.svg;
        svgBox.style.display = 'block';
      } else {
        preview.innerHTML = `<img src="${data.base64}" alt="Barcode">`;
      }
    })
    .catch(err => {
      preview.innerHTML = `<div class="error-msg">${err}</div>`;
    });
}

// Init
onTypeChange();
</script>
</body>
</html>
