// Barcode.Go (C++ WASM Engine) - Easy 2 Steps
//
// QR code generation in just 2 steps using net/http + C++ WASM engine.
//
//	go run main.go
//	→ http://localhost:5702
package main

import (
	"encoding/json"
	"fmt"
	"log"
	"net/http"
	"text/template"

	wasm "barcode_pao_wasm"
)

var tmpl *template.Template

func init() {
	tmpl = template.Must(template.ParseFiles("templates/index.html"))
}

func main() {
	http.HandleFunc("/", handleIndex)
	http.HandleFunc("/draw-base64", handleDrawBase64)
	http.HandleFunc("/draw-svg", handleDrawSVG)

	fmt.Println("Barcode.Go (C++ WASM Engine) Easy 2 Steps")
	fmt.Println("→ http://localhost:5702")
	log.Fatal(http.ListenAndServe(":5702", nil))
}

func handleIndex(w http.ResponseWriter, r *http.Request) {
	tmpl.Execute(w, nil)
}

type apiResponse struct {
	OK     bool   `json:"ok"`
	Base64 string `json:"base64,omitempty"`
	SVG    string `json:"svg,omitempty"`
	Error  string `json:"error,omitempty"`
}

func writeJSON(w http.ResponseWriter, resp apiResponse) {
	w.Header().Set("Content-Type", "application/json")
	json.NewEncoder(w).Encode(resp)
}

func handleDrawBase64(w http.ResponseWriter, r *http.Request) {
	if r.Method != http.MethodPost {
		writeJSON(w, apiResponse{Error: "POST only"})
		return
	}
	r.ParseForm()
	code := r.FormValue("code")
	if code == "" {
		code = "https://www.pao.ac/"
	}

	// Step 1: Create barcode
	qr := wasm.NewQRCode(wasm.FormatPNG)
	// Step 2: Draw!
	b64, err := qr.Draw(code, 300)
	if err != nil {
		writeJSON(w, apiResponse{Error: err.Error()})
		return
	}
	writeJSON(w, apiResponse{OK: true, Base64: b64})
}

func handleDrawSVG(w http.ResponseWriter, r *http.Request) {
	if r.Method != http.MethodPost {
		writeJSON(w, apiResponse{Error: "POST only"})
		return
	}
	r.ParseForm()
	code := r.FormValue("code")
	if code == "" {
		code = "https://www.pao.ac/"
	}

	// Step 1: Create barcode (SVG mode)
	qr := wasm.NewQRCode(wasm.FormatSVG)
	// Step 2: Draw!
	svg, err := qr.Draw(code, 300)
	if err != nil {
		writeJSON(w, apiResponse{Error: err.Error()})
		return
	}
	writeJSON(w, apiResponse{OK: true, SVG: svg})
}
