// Barcode.Go (C++ WASM Engine) - All-in-One
//
// Full 18-type barcode REST API using net/http + C++ WASM engine.
//
//	go run main.go
//	→ http://localhost:5703
package main

import (
	"encoding/json"
	"fmt"
	"log"
	"net/http"
	"strconv"
	"text/template"

	wasm "barcode_pao_wasm"
)

// barcodeTypeDef defines a barcode type for the UI.
type barcodeTypeDef struct {
	ID      string `json:"id"`
	Label   string `json:"label"`
	Group   string `json:"group"`
	Dim     string `json:"dim"`
	Default string `json:"default"`
	W       int    `json:"w"`
	H       int    `json:"h"`
}

var barcodeTypes = []barcodeTypeDef{
	// 2D
	{ID: "QR", Label: "QR Code", Group: "2D Barcode", Dim: "2d", Default: "https://www.pao.ac/", W: 200, H: 200},
	{ID: "DataMatrix", Label: "DataMatrix", Group: "2D Barcode", Dim: "2d", Default: "Hello DataMatrix", W: 200, H: 200},
	{ID: "PDF417", Label: "PDF417", Group: "2D Barcode", Dim: "2d", Default: "Hello PDF417", W: 200, H: 100},
	// Special
	{ID: "GS1_128", Label: "GS1-128", Group: "Special", Dim: "1d", Default: "[01]04912345123459", W: 400, H: 100},
	{ID: "YubinCustomer", Label: "Yubin Customer", Group: "Special", Dim: "postal", Default: "1060032", W: 400, H: 60},
	// 1D
	{ID: "Code128", Label: "Code 128", Group: "1D Barcode", Dim: "1d", Default: "Hello-2026", W: 400, H: 100},
	{ID: "Code39", Label: "Code 39", Group: "1D Barcode", Dim: "1d", Default: "HELLO-123", W: 400, H: 100},
	{ID: "Code93", Label: "Code 93", Group: "1D Barcode", Dim: "1d", Default: "CODE93", W: 400, H: 100},
	{ID: "NW7", Label: "NW-7 / Codabar", Group: "1D Barcode", Dim: "1d", Default: "A123456B", W: 400, H: 100},
	{ID: "ITF", Label: "ITF", Group: "1D Barcode", Dim: "1d", Default: "123456", W: 400, H: 100},
	{ID: "Matrix2of5", Label: "Matrix 2 of 5", Group: "1D Barcode", Dim: "1d", Default: "1234", W: 400, H: 100},
	{ID: "NEC2of5", Label: "NEC 2 of 5", Group: "1D Barcode", Dim: "1d", Default: "1234", W: 400, H: 100},
	// GS1 DataBar
	{ID: "GS1DataBar14", Label: "GS1 DataBar 14", Group: "GS1 DataBar", Dim: "1d", Default: "0123456789012", W: 300, H: 80},
	{ID: "GS1DataBarLimited", Label: "GS1 DataBar Limited", Group: "GS1 DataBar", Dim: "1d", Default: "0123456789012", W: 300, H: 80},
	{ID: "GS1DataBarExpanded", Label: "GS1 DataBar Expanded", Group: "GS1 DataBar", Dim: "1d", Default: "[01]90012345678908", W: 400, H: 80},
	// JAN / UPC
	{ID: "JAN13", Label: "JAN-13 / EAN-13", Group: "JAN / UPC", Dim: "1d", Default: "490123456789", W: 300, H: 100},
	{ID: "JAN8", Label: "JAN-8 / EAN-8", Group: "JAN / UPC", Dim: "1d", Default: "1234567", W: 250, H: 100},
	{ID: "UPCA", Label: "UPC-A", Group: "JAN / UPC", Dim: "1d", Default: "01234567890", W: 300, H: 100},
	{ID: "UPCE", Label: "UPC-E", Group: "JAN / UPC", Dim: "1d", Default: "0123456", W: 250, H: 100},
}

var barcodeMap map[string]barcodeTypeDef
var tmpl *template.Template

func init() {
	barcodeMap = make(map[string]barcodeTypeDef)
	for _, bt := range barcodeTypes {
		barcodeMap[bt.ID] = bt
	}
	tmpl = template.Must(template.ParseFiles("templates/index.html"))
}

func main() {
	http.HandleFunc("/", handleIndex)
	http.HandleFunc("/api/types", handleTypes)
	http.HandleFunc("/draw-base64", handleDrawBase64)
	http.HandleFunc("/draw-svg", handleDrawSVG)

	fmt.Println("Barcode.Go (C++ WASM Engine) All-in-One")
	fmt.Println("→ http://localhost:5703")
	log.Fatal(http.ListenAndServe(":5703", nil))
}

type pageData struct {
	BarcodeTypesJSON string
}

func handleIndex(w http.ResponseWriter, r *http.Request) {
	jsonBytes, _ := json.Marshal(barcodeTypes)
	tmpl.Execute(w, pageData{BarcodeTypesJSON: string(jsonBytes)})
}

func handleTypes(w http.ResponseWriter, r *http.Request) {
	w.Header().Set("Content-Type", "application/json")
	json.NewEncoder(w).Encode(barcodeTypes)
}

type apiResponse struct {
	OK     bool   `json:"ok"`
	Base64 string `json:"base64,omitempty"`
	SVG    string `json:"svg,omitempty"`
	Error  string `json:"error,omitempty"`
}

func writeJSON(w http.ResponseWriter, resp apiResponse) {
	w.Header().Set("Content-Type", "application/json")
	json.NewEncoder(w).Encode(resp)
}

func getFormInt(r *http.Request, key string, def int) int {
	s := r.FormValue(key)
	if s == "" {
		return def
	}
	v, err := strconv.Atoi(s)
	if err != nil {
		return def
	}
	return v
}

func createAndDraw(typeID, code, outputFormat string, width, height int) (string, error) {
	switch typeID {
	case "QR":
		bc := wasm.NewQRCode(outputFormat)
		return bc.Draw(code, width)

	case "DataMatrix":
		bc := wasm.NewDataMatrix(outputFormat)
		return bc.Draw(code, width)

	case "PDF417":
		bc := wasm.NewPDF417(outputFormat)
		return bc.Draw(code, width, height)

	case "Code128":
		bc := wasm.NewCode128(outputFormat)
		return bc.Draw(code, width, height)

	case "Code39":
		bc := wasm.NewCode39(outputFormat)
		return bc.Draw(code, width, height)

	case "Code93":
		bc := wasm.NewCode93(outputFormat)
		return bc.Draw(code, width, height)

	case "NW7":
		bc := wasm.NewNW7(outputFormat)
		return bc.Draw(code, width, height)

	case "ITF":
		bc := wasm.NewITF(outputFormat)
		return bc.Draw(code, width, height)

	case "Matrix2of5":
		bc := wasm.NewMatrix2of5(outputFormat)
		return bc.Draw(code, width, height)

	case "NEC2of5":
		bc := wasm.NewNEC2of5(outputFormat)
		return bc.Draw(code, width, height)

	case "JAN13":
		bc := wasm.NewJAN13(outputFormat)
		return bc.Draw(code, width, height)

	case "JAN8":
		bc := wasm.NewJAN8(outputFormat)
		return bc.Draw(code, width, height)

	case "UPCA":
		bc := wasm.NewUPCA(outputFormat)
		return bc.Draw(code, width, height)

	case "UPCE":
		bc := wasm.NewUPCE(outputFormat)
		return bc.Draw(code, width, height)

	case "GS1_128":
		bc := wasm.NewGS1128(outputFormat)
		return bc.Draw(code, width, height)

	case "GS1DataBar14":
		bc := wasm.NewGS1DataBar14(outputFormat)
		return bc.Draw(code, width, height)

	case "GS1DataBarLimited":
		bc := wasm.NewGS1DataBarLimited(outputFormat)
		return bc.Draw(code, width, height)

	case "GS1DataBarExpanded":
		bc := wasm.NewGS1DataBarExpanded(outputFormat)
		return bc.Draw(code, width, height)

	case "YubinCustomer":
		bc := wasm.NewYubinCustomer(outputFormat)
		return bc.Draw(code, height)

	default:
		return "", fmt.Errorf("unknown barcode type: %s", typeID)
	}
}

func handleDrawBase64(w http.ResponseWriter, r *http.Request) {
	if r.Method != http.MethodPost {
		writeJSON(w, apiResponse{Error: "POST only"})
		return
	}
	r.ParseForm()
	typeID := r.FormValue("type")
	if typeID == "" {
		typeID = "QR"
	}
	code := r.FormValue("code")
	info, ok := barcodeMap[typeID]
	if !ok {
		info = barcodeMap["QR"]
		typeID = "QR"
	}
	if code == "" {
		code = info.Default
	}
	width := getFormInt(r, "width", info.W)
	height := getFormInt(r, "height", info.H)

	result, err := createAndDraw(typeID, code, wasm.FormatPNG, width, height)
	if err != nil {
		writeJSON(w, apiResponse{Error: err.Error()})
		return
	}
	writeJSON(w, apiResponse{OK: true, Base64: result})
}

func handleDrawSVG(w http.ResponseWriter, r *http.Request) {
	if r.Method != http.MethodPost {
		writeJSON(w, apiResponse{Error: "POST only"})
		return
	}
	r.ParseForm()
	typeID := r.FormValue("type")
	if typeID == "" {
		typeID = "QR"
	}
	code := r.FormValue("code")
	info, ok := barcodeMap[typeID]
	if !ok {
		info = barcodeMap["QR"]
		typeID = "QR"
	}
	if code == "" {
		code = info.Default
	}
	width := getFormInt(r, "width", info.W)
	height := getFormInt(r, "height", info.H)

	result, err := createAndDraw(typeID, code, wasm.FormatSVG, width, height)
	if err != nil {
		writeJSON(w, apiResponse{Error: err.Error()})
		return
	}
	writeJSON(w, apiResponse{OK: true, SVG: result})
}
