// Barcode.Go - Easy 2 Steps
//
// QR code generation in just 2 steps using net/http.
// 4 output modes: PNG, SVG, PDF (gopdf), Canvas (gg)
//
//	go run main.go
//	→ http://localhost:5700
package main

import (
	"bytes"
	"encoding/base64"
	"encoding/json"
	"fmt"
	"image/png"
	"log"
	"math"
	"math/rand"
	"net/http"
	"os"
	"text/template"
	"time"

	"github.com/fogleman/gg"
	barcode "github.com/pao-company/barcode-go"
	barcodeGg "github.com/pao-company/barcode-go/ggrenderer"
	barcodeGopdf "github.com/pao-company/barcode-go/gopdf"
	"github.com/signintech/gopdf"
)

// loadFont tries to load a TrueType font into the GoPdf instance.
func loadFont(pdf *gopdf.GoPdf) {
	fontPaths := []string{
		"C:/Windows/Fonts/arial.ttf",
		"/usr/share/fonts/dejavu-sans-fonts/DejaVuSans.ttf",
		"/usr/share/fonts/liberation-sans/LiberationSans-Regular.ttf",
		"/usr/share/fonts/truetype/dejavu/DejaVuSans.ttf",
	}
	for _, fp := range fontPaths {
		if _, err := os.Stat(fp); err == nil {
			if err := pdf.AddTTFFont("font", fp); err == nil {
				pdf.SetFont("font", "", 12)
				return
			}
		}
	}
}

var tmpl *template.Template

func init() {
	tmpl = template.Must(template.ParseFiles("templates/index.html"))
}

func main() {
	http.HandleFunc("/", handleIndex)
	http.HandleFunc("/draw-base64", handleDrawBase64)
	http.HandleFunc("/draw-svg", handleDrawSVG)
	http.HandleFunc("/draw-canvas", handleDrawCanvas)
	http.HandleFunc("/pdf", handlePDF)

	fmt.Println("Barcode.Go Easy 2 Steps")
	fmt.Println("→ http://localhost:5700")
	log.Fatal(http.ListenAndServe(":5700", nil))
}

func handleIndex(w http.ResponseWriter, r *http.Request) {
	tmpl.Execute(w, nil)
}

type apiResponse struct {
	OK     bool   `json:"ok"`
	Base64 string `json:"base64,omitempty"`
	SVG    string `json:"svg,omitempty"`
	Error  string `json:"error,omitempty"`
}

func writeJSON(w http.ResponseWriter, resp apiResponse) {
	w.Header().Set("Content-Type", "application/json")
	json.NewEncoder(w).Encode(resp)
}

func handleDrawBase64(w http.ResponseWriter, r *http.Request) {
	if r.Method != http.MethodPost {
		writeJSON(w, apiResponse{Error: "POST only"})
		return
	}
	r.ParseForm()
	code := r.FormValue("code")
	if code == "" {
		code = "https://www.pao.ac/"
	}

	// Step 1: Create barcode
	qr := barcode.NewQRCode(barcode.FormatPNG)
	// Step 2: Draw!
	err := qr.Draw(code, 300)
	if err != nil {
		writeJSON(w, apiResponse{Error: err.Error()})
		return
	}
	b64, err := qr.GetImageBase64()
	if err != nil {
		writeJSON(w, apiResponse{Error: err.Error()})
		return
	}
	writeJSON(w, apiResponse{OK: true, Base64: b64})
}

func handleDrawSVG(w http.ResponseWriter, r *http.Request) {
	if r.Method != http.MethodPost {
		writeJSON(w, apiResponse{Error: "POST only"})
		return
	}
	r.ParseForm()
	code := r.FormValue("code")
	if code == "" {
		code = "https://www.pao.ac/"
	}

	// Step 1: Create barcode (SVG mode)
	qr := barcode.NewQRCode(barcode.FormatSVG)
	// Step 2: Draw!
	err := qr.Draw(code, 300)
	if err != nil {
		writeJSON(w, apiResponse{Error: err.Error()})
		return
	}
	svg, err := qr.GetSVG()
	if err != nil {
		writeJSON(w, apiResponse{Error: err.Error()})
		return
	}
	writeJSON(w, apiResponse{OK: true, SVG: svg})
}

// handleDrawCanvas draws barcodes onto a gg canvas with fun graphics
func handleDrawCanvas(w http.ResponseWriter, r *http.Request) {
	if r.Method != http.MethodPost {
		writeJSON(w, apiResponse{Error: "POST only"})
		return
	}
	r.ParseForm()
	code := r.FormValue("code")
	if code == "" {
		code = "https://www.pao.ac/"
	}

	b64, err := drawCanvasFun(code)
	if err != nil {
		writeJSON(w, apiResponse{Error: err.Error()})
		return
	}
	writeJSON(w, apiResponse{OK: true, Base64: b64})
}

func drawCanvasFun(code string) (string, error) {
	W, H := 900, 520
	dc := gg.NewContext(W, H)

	// --- Night sky gradient ---
	for i := 0; i < H; i++ {
		t := float64(i) / float64(H)
		r := 30 + 20*t
		g := 60 + 40*t
		b := 120 + 80*t
		dc.SetRGBA255(int(r), int(g), int(b), 255)
		dc.DrawLine(0, float64(i), float64(W), float64(i))
		dc.Stroke()
	}

	// --- Scatter stars ---
	rng := rand.New(rand.NewSource(42))
	for i := 0; i < 25; i++ {
		sx := float64(rng.Intn(W - 20) + 10)
		sy := float64(rng.Intn(H - 20) + 10)
		sz := float64(rng.Intn(4) + 2)
		alpha := float64(rng.Intn(120)+80) / 255.0
		dc.SetRGBA(1, 1, 0.78, alpha)
		dc.DrawCircle(sx, sy, sz)
		dc.Fill()
	}

	// --- Elephant illustration ---
	ex, ey := 150.0, 290.0
	s := 1.4
	// Body
	dc.SetRGBA255(180, 190, 210, 220)
	dc.DrawEllipse(ex, ey, 55*s, 35*s)
	dc.Fill()
	dc.SetRGBA255(120, 130, 160, 255)
	dc.DrawEllipse(ex, ey, 55*s, 35*s)
	dc.Stroke()
	// Head
	dc.SetRGBA255(180, 190, 210, 220)
	headX := ex + 60*s
	headY := ey - 25*s
	dc.DrawEllipse(headX, headY, 30*s, 30*s)
	dc.Fill()
	dc.SetRGBA255(120, 130, 160, 255)
	dc.DrawEllipse(headX, headY, 30*s, 30*s)
	dc.Stroke()
	// Ear
	dc.SetRGBA255(160, 170, 195, 200)
	dc.DrawEllipse(ex+85*s, ey-35*s, 20*s, 25*s)
	dc.Fill()
	dc.SetRGBA255(120, 130, 160, 255)
	dc.DrawEllipse(ex+85*s, ey-35*s, 20*s, 25*s)
	dc.Stroke()
	// Eye
	dc.SetRGBA255(255, 255, 255, 255)
	dc.DrawCircle(ex+55*s, ey-35*s, 4*s)
	dc.Fill()
	dc.SetRGBA255(30, 30, 50, 255)
	dc.DrawCircle(ex+57*s, ey-33*s, 2*s)
	dc.Fill()
	// Trunk
	dc.SetRGBA255(140, 150, 175, 255)
	dc.SetLineWidth(8)
	dc.MoveTo(ex+85*s, ey-15*s)
	dc.LineTo(ex+100*s, ey+10*s)
	dc.LineTo(ex+95*s, ey+35*s)
	dc.LineTo(ex+80*s, ey+45*s)
	dc.Stroke()
	dc.SetLineWidth(1)
	// Legs
	for _, lx := range []float64{ex - 30, ex - 5, ex + 25, ex + 50} {
		dc.SetRGBA255(160, 170, 195, 220)
		dc.DrawRectangle(lx*s, ey+30*s, 20, 30*s)
		dc.Fill()
		dc.SetRGBA255(120, 130, 160, 255)
		dc.DrawRectangle(lx*s, ey+30*s, 20, 30*s)
		dc.Stroke()
	}
	// Tail
	dc.SetRGBA255(140, 150, 175, 255)
	dc.SetLineWidth(4)
	dc.DrawArc(ex-60*s, ey+5*s, 15*s, math.Pi*2/3, math.Pi*4/3)
	dc.Stroke()
	dc.SetLineWidth(1)

	// --- White card area ---
	cardX, cardY := 340.0, 40.0
	cardW, cardH := 520.0, 440.0
	dc.SetRGBA255(255, 255, 255, 230)
	dc.DrawRoundedRectangle(cardX, cardY, cardW, cardH, 14)
	dc.Fill()
	dc.SetRGBA255(200, 210, 230, 255)
	dc.SetLineWidth(2)
	dc.DrawRoundedRectangle(cardX, cardY, cardW, cardH, 14)
	dc.Stroke()
	dc.SetLineWidth(1)

	// --- Title text ---
	dc.SetRGBA255(30, 60, 120, 255)
	dc.DrawStringAnchored("Barcode.Go Canvas Demo", cardX+24, cardY+26, 0, 0.5)
	dc.SetRGBA255(100, 120, 160, 255)
	dc.DrawStringAnchored("gg + GgRenderer", cardX+24, cardY+50, 0, 0.5)

	// --- Draw barcodes via gg renderer ---
	renderer := barcodeGg.New()
	defer renderer.Close()

	// QR Code
	qr := barcode.NewQRCode(barcode.FormatPNG)
	qrSize := 220.0
	qrX := cardX + (cardW-qrSize)/2
	qrY := cardY + 75
	if err := renderer.DrawBarcode(dc, qr, code, qrX, qrY, qrSize, qrSize); err != nil {
		return "", err
	}

	// Code128
	c128 := barcode.NewCode128(barcode.FormatPNG)
	c128W, c128H := 380.0, 60.0
	c128X := cardX + (cardW-c128W)/2
	c128Y := cardY + 330
	if err := renderer.DrawBarcode(dc, c128, "BARCODE-2026", c128X, c128Y, c128W, c128H); err != nil {
		return "", err
	}

	// --- Data label ---
	dc.SetRGBA255(80, 100, 140, 255)
	label := code
	if len(label) > 40 {
		label = label[:40]
	}
	dc.DrawStringAnchored(fmt.Sprintf("QR: %s", label), cardX+24, cardY+315, 0, 0.5)

	// --- Footer ---
	dc.SetRGBA255(160, 170, 190, 255)
	dc.DrawStringAnchored("Generated by barcode-go + gg", cardX+24, cardY+cardH-20, 0, 0.5)

	// --- Encode to base64 ---
	img := dc.Image()
	var buf bytes.Buffer
	if err := png.Encode(&buf, img); err != nil {
		return "", err
	}
	b64 := base64.StdEncoding.EncodeToString(buf.Bytes())
	return "data:image/png;base64," + b64, nil
}

// handlePDF generates a DELIVERY NOTE invoice PDF
func handlePDF(w http.ResponseWriter, r *http.Request) {
	code := r.URL.Query().Get("code")
	if code == "" {
		code = "https://www.pao.ac/"
	}

	pdfBytes, err := generateInvoicePDF(code)
	if err != nil {
		http.Error(w, err.Error(), 500)
		return
	}
	w.Header().Set("Content-Type", "application/pdf")
	w.Header().Set("Content-Disposition", "inline; filename=invoice_sample.pdf")
	w.Write(pdfBytes)
}

func generateInvoicePDF(code string) ([]byte, error) {
	pdf := &gopdf.GoPdf{}
	pdf.Start(gopdf.Config{PageSize: *gopdf.PageSizeA4})
	pdf.AddPage()
	loadFont(pdf)

	pw, ph := gopdf.PageSizeA4.W, gopdf.PageSizeA4.H // 595.28, 841.89

	// Create renderer
	renderer := barcodeGopdf.New(pdf)
	defer renderer.Close()

	today := time.Now().Format("2006-01-02")
	invNo := fmt.Sprintf("INV-%s", time.Now().Format("2006-0102"))

	// ========== Header (blue) ==========
	pdf.SetFillColor(30, 64, 175) // #1E40AF
	pdf.RectFromUpperLeftWithStyle(0, 0, pw, 82, "F")
	// Accent line
	pdf.SetFillColor(124, 58, 237) // #7C3AED
	pdf.RectFromUpperLeftWithStyle(0, 82, pw, 4, "F")

	pdf.SetTextColor(255, 255, 255)
	pdf.SetFontSize(24)
	pdf.SetX(40)
	pdf.SetY(25)
	pdf.Cell(nil, "DELIVERY NOTE")

	pdf.SetFontSize(9)
	pdf.SetX(pw - 160)
	pdf.SetY(20)
	pdf.Cell(nil, "Pao@Office")
	pdf.SetX(pw - 160)
	pdf.SetY(33)
	pdf.Cell(nil, fmt.Sprintf("No: %s", invNo))
	pdf.SetX(pw - 160)
	pdf.SetY(46)
	pdf.Cell(nil, fmt.Sprintf("Date: %s", today))
	pdf.SetX(pw - 160)
	pdf.SetY(59)
	pdf.Cell(nil, "Tokyo, Japan")

	// ========== Recipient ==========
	pdf.SetTextColor(0, 0, 0)
	pdf.SetFontSize(11)
	pdf.SetX(40)
	pdf.SetY(105)
	pdf.Cell(nil, "Sample Corporation")

	// ========== Barcode area ==========
	y := 140.0
	pdf.SetFillColor(241, 245, 249) // #F1F5F9
	pdf.RectFromUpperLeftWithStyle(35, y, pw-70, 130, "F")

	pdf.SetTextColor(71, 85, 105)
	pdf.SetFontSize(8)
	pdf.SetX(50)
	pdf.SetY(y + 10)
	pdf.Cell(nil, "INVOICE BARCODE (Code128)")

	// Code128 barcode
	c128 := barcode.NewCode128(barcode.FormatPNG)
	renderer.DrawBarcode(c128, invNo, 50, y+25, 220, 40)

	pdf.SetFontSize(8)
	pdf.SetX(310)
	pdf.SetY(y + 10)
	pdf.Cell(nil, "QR CODE")

	// QR barcode
	qr := barcode.NewQRCode(barcode.FormatPNG)
	renderer.DrawBarcode(qr, code, 310, y+20, 100, 100)

	pdf.SetTextColor(148, 163, 184)
	pdf.SetFontSize(7)
	pdf.SetX(310)
	pdf.SetY(y + 122)
	label := code
	if len(label) > 40 {
		label = label[:40]
	}
	pdf.Cell(nil, fmt.Sprintf("Data: %s", label))

	// ========== Invoice table ==========
	y = 290.0
	// Header row
	pdf.SetFillColor(30, 64, 175)
	pdf.RectFromUpperLeftWithStyle(40, y, pw-80, 22, "F")
	pdf.SetTextColor(255, 255, 255)
	pdf.SetFontSize(9)
	headers := []struct{ text string; x float64 }{
		{"Item", 50}, {"Qty", 330}, {"Unit Price", 420}, {"Amount", 520},
	}
	for _, h := range headers {
		pdf.SetX(h.x)
		pdf.SetY(y + 6)
		pdf.Cell(nil, h.text)
	}

	// Data rows
	items := []struct{ name, qty, price, amount string }{
		{"Barcode.Go Pure Go Edition", "1", "\\33,000", "\\33,000"},
		{"Annual Support (3 years)", "1", "\\9,900", "\\9,900"},
	}
	pdf.SetTextColor(0, 0, 0)
	pdf.SetFontSize(9)
	for i, item := range items {
		ry := y + 22*float64(i+1)
		if i%2 == 0 {
			pdf.SetFillColor(248, 250, 252)
			pdf.RectFromUpperLeftWithStyle(40, ry, pw-80, 22, "F")
		}
		pdf.SetX(50)
		pdf.SetY(ry + 6)
		pdf.Cell(nil, item.name)
		pdf.SetX(340)
		pdf.SetY(ry + 6)
		pdf.Cell(nil, item.qty)
		pdf.SetX(420)
		pdf.SetY(ry + 6)
		pdf.Cell(nil, item.price)
		pdf.SetX(520)
		pdf.SetY(ry + 6)
		pdf.Cell(nil, item.amount)
	}

	// Total
	ry := y + 22*float64(len(items)+1) + 10
	pdf.SetStrokeColor(30, 64, 175)
	pdf.SetLineWidth(2)
	pdf.Line(350, ry, pw-40, ry)
	pdf.SetTextColor(30, 64, 175)
	pdf.SetFontSize(13)
	pdf.SetX(pw - 170)
	pdf.SetY(ry + 5)
	pdf.Cell(nil, "Total:  \\42,900")

	// ========== JAN-13 ==========
	ry += 40
	pdf.SetTextColor(71, 85, 105)
	pdf.SetFontSize(8)
	pdf.SetX(40)
	pdf.SetY(ry)
	pdf.Cell(nil, "PRODUCT BARCODE (JAN-13)")

	jan := barcode.NewJAN13(barcode.FormatPNG)
	renderer.DrawBarcode(jan, "490123456789", 40, ry+15, 180, 45)

	// ========== Footer ==========
	pdf.SetStrokeColor(226, 232, 240)
	pdf.SetLineWidth(0.5)
	pdf.Line(40, ph-45, pw-40, ph-45)
	pdf.SetTextColor(148, 163, 184)
	pdf.SetFontSize(7)
	pdf.SetX(40)
	pdf.SetY(ph - 35)
	pdf.Cell(nil, "Generated by Barcode.Go — barcode-go + GoPdfRenderer")
	pdf.SetX(pw - 80)
	pdf.SetY(ph - 35)
	pdf.Cell(nil, "pao.ac")

	// Output
	var buf bytes.Buffer
	_, err := pdf.WriteTo(&buf)
	if err != nil {
		return nil, err
	}
	return buf.Bytes(), nil
}
