package barcode

import (
	"strings"
	"testing"
)

func TestYubinCustomerEncode(t *testing.T) {
	y := NewYubinCustomer(FormatSVG)
	bars, err := y.Encode("1060002")
	if err != nil {
		t.Fatalf("Encode failed: %v", err)
	}
	// Total should be 67 bars
	if len(bars) != 67 {
		t.Errorf("expected 67 bars, got %d", len(bars))
	}
	// All bar values should be 1-4
	for i, b := range bars {
		if b < 1 || b > 4 {
			t.Errorf("bar[%d] = %d, expected 1-4", i, b)
		}
	}
}

func TestYubinCustomerCheckDigit(t *testing.T) {
	y := NewYubinCustomer(FormatSVG)
	_, err := y.Encode("1060002")
	if err != nil {
		t.Fatalf("Encode failed: %v", err)
	}
	if y.CheckDigit() < 0 || y.CheckDigit() > 18 {
		t.Errorf("check digit %d out of range 0-18", y.CheckDigit())
	}
}

func TestYubinCustomerWithAlpha(t *testing.T) {
	y := NewYubinCustomer(FormatSVG)
	bars, err := y.Encode("1060002-12-3F")
	if err != nil {
		t.Fatalf("Encode failed: %v", err)
	}
	if len(bars) != 67 {
		t.Errorf("expected 67 bars, got %d", len(bars))
	}
}

func TestYubinCustomerEmpty(t *testing.T) {
	y := NewYubinCustomer(FormatSVG)
	_, err := y.Encode("")
	if err == nil {
		t.Fatal("expected error for empty string")
	}
}

func TestYubinCustomerDrawSVG(t *testing.T) {
	y := NewYubinCustomer(FormatSVG)
	err := y.Draw("1060002", 400, 50)
	if err != nil {
		t.Fatalf("Draw failed: %v", err)
	}
	svg, err := y.GetSVG()
	if err != nil {
		t.Fatalf("GetSVG failed: %v", err)
	}
	if !strings.Contains(svg, "<svg") {
		t.Error("SVG output missing <svg tag")
	}
}

func TestYubinCustomerDrawPNG(t *testing.T) {
	y := NewYubinCustomer(FormatPNG)
	err := y.Draw("1060002", 400, 50)
	if err != nil {
		t.Fatalf("Draw failed: %v", err)
	}
	data := y.GetImageMemory()
	if len(data) == 0 {
		t.Fatal("PNG output is empty")
	}
	// PNG magic bytes
	if data[0] != 0x89 || data[1] != 0x50 {
		t.Error("output is not valid PNG")
	}
}
