package barcode

import (
	"fmt"
	"strings"
)

// Digit patterns (3 bars each): "1"=long, "2"=semi-upper, "3"=semi-lower, "4"=timing
var (
	yubinPtnN = []string{"144", "114", "132", "312", "123",
		"141", "321", "213", "231", "411"}
	yubinPtnC    = []string{"324", "342", "234", "432", "243", "423", "441", "111"}
	yubinStart   = "13"
	yubinStop    = "31"
	yubinPtnHI   = "414"
	yubinAscChrs = "0123456789-ABCDEFGHIJKLMNOPQRSTUVWXYZ"
)

func yubinGetPattern(index int) string {
	if index < 0 || index > 36 {
		return ""
	}
	if index <= 9 {
		return yubinPtnN[index]
	} else if index == 10 {
		return yubinPtnHI
	} else if index <= 20 {
		return yubinPtnC[0] + yubinPtnN[index-11]
	} else if index <= 30 {
		return yubinPtnC[1] + yubinPtnN[index-21]
	} else {
		return yubinPtnC[2] + yubinPtnN[index-31]
	}
}

func yubinGetCheckPattern(chkD int) string {
	if chkD < 0 || chkD > 18 {
		return ""
	}
	if chkD <= 9 {
		return yubinPtnN[chkD]
	} else if chkD == 10 {
		return yubinPtnHI
	} else {
		return yubinPtnC[chkD-11]
	}
}

func yubinPatternToCheckValue(pattern string) int {
	for j := 0; j < 10; j++ {
		if pattern == yubinPtnN[j] {
			return j
		}
	}
	if pattern == yubinPtnHI {
		return 10
	}
	for j := 0; j < 8; j++ {
		if pattern == yubinPtnC[j] {
			return 11 + j
		}
	}
	return -1
}

// YubinCustomer is the Japan Post Customer Barcode encoder.
type YubinCustomer struct {
	BarcodeBase1D
	bars       []int
	checkDigit int
}

// NewYubinCustomer creates a new YubinCustomer encoder.
func NewYubinCustomer(outputFormat string) *YubinCustomer {
	y := &YubinCustomer{checkDigit: -1}
	y.InitBase1D(outputFormat)
	return y
}

// Bars returns the bar pattern after encoding.
func (y *YubinCustomer) Bars() []int { return y.bars }

// CheckDigit returns the check digit value (0-18) after encoding.
func (y *YubinCustomer) CheckDigit() int { return y.checkDigit }

// Encode encodes the Japan Post Customer Barcode.
func (y *YubinCustomer) Encode(code string) ([]int, error) {
	if code == "" {
		return nil, fmt.Errorf("input code is empty")
	}

	upperCode := strings.ToUpper(code)

	var bars []int
	var chkStr strings.Builder

	// Start code
	for _, ch := range yubinStart {
		bars = append(bars, int(ch-'0'))
	}

	// Data part (20 character positions)
	codeLen := 0
	i := 0
	for i < len(upperCode) && codeLen < 20 {
		p := strings.IndexByte(yubinAscChrs, upperCode[i])
		if p < 0 {
			i++
			continue
		}

		pattern := yubinGetPattern(p)

		// Special case: at position 19, if pattern > 3 bars (alpha char), use CC1 only
		if codeLen == 19 && len(pattern) > 3 {
			for _, ch := range yubinPtnC[0] {
				bars = append(bars, int(ch-'0'))
				chkStr.WriteRune(ch)
			}
			codeLen++
			break
		}

		for _, ch := range pattern {
			bars = append(bars, int(ch-'0'))
			chkStr.WriteRune(ch)
		}

		if len(pattern) <= 3 {
			codeLen++
		} else {
			codeLen += 2
		}
		i++
	}

	// Pad with CC4 to fill 20 positions
	for codeLen < 20 {
		for _, ch := range yubinPtnC[3] {
			bars = append(bars, int(ch-'0'))
		}
		chkStr.WriteString(yubinPtnC[3])
		codeLen++
	}

	// Check digit calculation (modulo 19)
	chkString := chkStr.String()
	chkSum := 0
	for j := 0; j <= len(chkString)-3; j += 3 {
		value := yubinPatternToCheckValue(chkString[j : j+3])
		if value >= 0 {
			chkSum += value
		}
	}

	chkD := 19 - (chkSum % 19)
	if chkD == 19 {
		chkD = 0
	}
	y.checkDigit = chkD

	// Check digit bars
	chkPattern := yubinGetCheckPattern(chkD)
	for _, ch := range chkPattern {
		bars = append(bars, int(ch-'0'))
	}

	// Stop code
	for _, ch := range yubinStop {
		bars = append(bars, int(ch-'0'))
	}

	y.bars = bars
	return bars, nil
}

// Draw renders the YubinCustomer barcode (4-state bars).
func (y *YubinCustomer) Draw(code string, width, height int) error {
	bars, err := y.Encode(code)
	if err != nil {
		return err
	}
	if len(bars) == 0 {
		return fmt.Errorf("empty bar pattern")
	}

	if y.IsSVGOutput() {
		return y.drawSVG4State(bars, width, height)
	}
	return y.drawPNG4State(bars, width, height)
}
