package barcode

import (
	"fmt"
	"strconv"
)

var (
	upceOE   = []int{0x00, 0x0b, 0x0d, 0x0e, 0x13, 0x19, 0x1c, 0x15, 0x16, 0x1a}
	upceLeft = [2][]int{
		{0x27, 0x33, 0x1b, 0x21, 0x1d, 0x39, 0x05, 0x11, 0x09, 0x17}, // G-pattern
		{0x0d, 0x19, 0x13, 0x3d, 0x23, 0x31, 0x2f, 0x3b, 0x37, 0x0b}, // L-pattern
	}
)

func expandToUPCA(code string) string {
	if len(code) != 7 {
		return ""
	}
	c0, c6 := code[0], code[6]
	switch {
	case c0 == '0' && (c6 == '0' || c6 == '1' || c6 == '2'):
		return code[:3] + string(c6) + "0000" + code[3:6]
	case c0 == '0' && c6 == '3':
		return code[:4] + "00000" + code[4:6]
	case c0 == '0' && c6 == '4':
		return code[:5] + "00000" + code[5:6]
	case c0 == '0' && c6 >= '5' && c6 <= '9':
		return code[:6] + "0000" + string(c6)
	}
	return ""
}

// CalculateCheckDigitUPCE computes the UPC-E check digit via UPC-A expansion.
func CalculateCheckDigitUPCE(src string) string {
	var expanded string
	if len(src) == 7 {
		expanded = expandToUPCA(src)
		if expanded == "" {
			return ""
		}
	} else {
		expanded = src
	}

	odds := 0
	evens := 0
	odd := true
	for i := len(expanded); i > 0; i-- {
		n := int(expanded[i-1] - '0')
		if odd {
			odds += n
		} else {
			evens += n
		}
		odd = !odd
	}

	s := odds*3 + evens
	cd := s % 10
	if cd != 0 {
		cd = 10 - cd
	}
	return strconv.Itoa(cd)
}

// UPCE encoder.
type UPCE struct {
	BarcodeBase1D
	ExtendedGuard bool
}

// NewUPCE creates a new UPC-E encoder.
func NewUPCE(outputFormat string) *UPCE {
	u := &UPCE{ExtendedGuard: true}
	u.InitBase1D(outputFormat)
	return u
}

// Encode returns the bar/space width pattern for UPC-E.
func (u *UPCE) Encode(code string) ([]int, error) {
	if code == "" {
		return nil, fmt.Errorf("empty string")
	}

	displayCode := code
	if len(displayCode) == 6 {
		displayCode = "0" + displayCode
		displayCode += CalculateCheckDigitUPCE(displayCode)
	} else if len(displayCode) == 7 {
		displayCode += CalculateCheckDigitUPCE(displayCode)
	} else if len(displayCode) != 8 {
		return nil, fmt.Errorf("UPC-E requires 6, 7 or 8 digits")
	}

	for _, c := range displayCode {
		if c < '0' || c > '9' {
			return nil, fmt.Errorf("UPC-E barcode requires numeric digits only")
		}
	}

	return u.encodeUPCE(displayCode)
}

func (u *UPCE) encodeUPCE(src string) ([]int, error) {
	if len(src) != 8 {
		return nil, fmt.Errorf("EncodeUPCE: code must be 8 digits")
	}

	cd := int(src[7] - '0')
	data := src[1:7]

	pre := cd
	chkOE := 0x20

	// Build module sequence
	var modules []int

	// Start guard "101"
	modules = append(modules, 1, 0, 1)

	// Data 6 digits
	for i := 0; i < 6; i++ {
		d := int(data[i] - '0')
		flgOE := 0
		if (upceOE[pre] & (chkOE >> i)) != 0 {
			flgOE = 1
		}
		bits := upceLeft[flgOE][d]
		mask := 0x40

		for j := 0; j < 7; j++ {
			if (bits & (mask >> j)) != 0 {
				modules = append(modules, 1)
			} else {
				modules = append(modules, 0)
			}
		}
	}

	// End guard "010101"
	for j := 0; j < 6; j++ {
		if j%2 == 0 {
			modules = append(modules, 0)
		} else {
			modules = append(modules, 1)
		}
	}

	// Run-length encode
	if len(modules) == 0 || modules[0] != 1 {
		return nil, fmt.Errorf("EncodeUPCE: internal module sequence error")
	}

	var pattern []int
	current := modules[0]
	count := 1

	for k := 1; k < len(modules); k++ {
		if modules[k] == current {
			count++
		} else {
			pattern = append(pattern, count)
			current = modules[k]
			count = 1
		}
	}
	pattern = append(pattern, count)

	return pattern, nil
}

// Draw renders the UPC-E barcode with extended guard bars.
func (u *UPCE) Draw(code string, width, height int) error {
	pattern, err := u.Encode(code)
	if err != nil {
		return err
	}
	// Get display code
	displayCode := code
	if len(displayCode) == 6 {
		displayCode = "0" + displayCode
		displayCode += CalculateCheckDigitUPCE(displayCode)
	} else if len(displayCode) == 7 {
		displayCode += CalculateCheckDigitUPCE(displayCode)
	}

	if u.IsSVGOutput() {
		if u.ShowText && u.ExtendedGuard {
			return u.drawSVGUPCE(pattern, displayCode, width, height)
		}
		display := ""
		if u.ShowText {
			display = displayCode
		}
		return u.drawSVGBars(pattern, width, height, display)
	}
	return u.drawPNGJANUPC(pattern, displayCode, width, height, "UPCE")
}
