package barcode

import (
	"fmt"
	"strconv"
)

var (
	upcaLeftPatterns  = []int{0x0d, 0x19, 0x13, 0x3d, 0x23, 0x31, 0x2f, 0x3b, 0x37, 0x0b}
	upcaRightPatterns = []int{0x72, 0x66, 0x6c, 0x42, 0x5c, 0x4e, 0x50, 0x44, 0x48, 0x74}
)

// CalculateCheckDigitUPCA computes the UPC-A check digit (Modulus 10, weight 3-1).
func CalculateCheckDigitUPCA(src string) string {
	total := 0
	for i := 0; i < 11; i++ {
		d := int(src[i] - '0')
		if i%2 == 0 {
			total += d * 3
		} else {
			total += d
		}
	}
	cd := (10 - (total % 10)) % 10
	return strconv.Itoa(cd)
}

// UPCA encoder.
type UPCA struct {
	BarcodeBase1D
	ExtendedGuard bool
}

// NewUPCA creates a new UPC-A encoder.
func NewUPCA(outputFormat string) *UPCA {
	u := &UPCA{ExtendedGuard: true}
	u.InitBase1D(outputFormat)
	return u
}

// Encode returns the bar/space width pattern for UPC-A.
func (u *UPCA) Encode(code string) ([]int, error) {
	if code == "" {
		return nil, fmt.Errorf("empty string")
	}
	for _, c := range code {
		if c < '0' || c > '9' {
			return nil, fmt.Errorf("UPC-A barcode requires numeric digits only")
		}
	}

	s := code
	if len(s) == 12 {
		expected := CalculateCheckDigitUPCA(s[:11])
		if string(s[11]) != expected {
			return nil, fmt.Errorf("invalid check digit")
		}
	} else if len(s) == 11 {
		s = s + CalculateCheckDigitUPCA(s)
	} else {
		return nil, fmt.Errorf("UPC-A barcode requires 11 or 12 digits")
	}

	leftHalf := s[:6]
	rightHalf := s[6:12]

	var result []int
	// Start guard (101)
	result = append(result, 1, 1, 1)

	// Left 6 digits (L-pattern)
	for i := 0; i < 6; i++ {
		digit := int(leftHalf[i] - '0')
		pattern := upcaLeftPatterns[digit]
		result = append(result, bitsToRunlength(pattern, 7)...)
	}

	// Center guard (01010)
	result = append(result, 1, 1, 1, 1, 1)

	// Right 6 digits (R-pattern)
	for i := 0; i < 6; i++ {
		digit := int(rightHalf[i] - '0')
		pattern := upcaRightPatterns[digit]
		result = append(result, bitsToRunlength(pattern, 7)...)
	}

	// End guard (101)
	result = append(result, 1, 1, 1)

	return result, nil
}

// Draw renders the UPC-A barcode with extended guard bars.
func (u *UPCA) Draw(code string, width, height int) error {
	pattern, err := u.Encode(code)
	if err != nil {
		return err
	}
	s := code
	if len(s) == 11 {
		s = s + CalculateCheckDigitUPCA(s)
	}

	if u.IsSVGOutput() {
		if u.ShowText && u.ExtendedGuard {
			return u.drawSVGUPCA(pattern, s, width, height)
		}
		display := ""
		if u.ShowText {
			display = s
		}
		return u.drawSVGBars(pattern, width, height, display)
	}
	return u.drawPNGJANUPC(pattern, s, width, height, "UPCA")
}
