package barcode

// Shared RSS/DataBar utility algorithms.
// Common algorithms used by GS1 DataBar 14, DataBar Limited, and DataBar Expanded.

// getCombinations calculates nCr (binomial coefficient).
func getCombinations(n, r int) int {
	var minDenom, maxDenom int
	if n-r > r {
		minDenom = r
		maxDenom = n - r
	} else {
		minDenom = n - r
		maxDenom = r
	}

	val := 1
	j := 1
	for i := n; i > maxDenom; i-- {
		val *= i
		if j <= minDenom {
			val /= j
			j++
		}
	}
	for j <= minDenom {
		val /= j
		j++
	}
	return val
}

// getWidths is the RSS subset width algorithm (ISO 24724).
// Converts a value into an element width pattern.
func getWidths(val, n, elements, maxWidth, noNarrow int) []int {
	widths := make([]int, elements)
	narrowMask := 0

	for bar := 0; bar < elements-1; bar++ {
		elmWidth := 1
		narrowMask |= (1 << bar)
		subVal := 0

		for {
			subVal = getCombinations(n-elmWidth-1, elements-bar-2)

			if noNarrow == 0 && narrowMask == 0 &&
				n-elmWidth-(elements-bar-1) >= elements-bar-1 {
				subVal -= getCombinations(n-elmWidth-(elements-bar),
					elements-bar-2)
			}

			if elements-bar-1 > 1 {
				lessVal := 0
				mxwElement := n - elmWidth - (elements - bar - 2)
				for mxwElement > maxWidth {
					lessVal += getCombinations(n-elmWidth-mxwElement-1,
						elements-bar-3)
					mxwElement--
				}
				subVal -= lessVal * (elements - 1 - bar)
			} else if n-elmWidth > maxWidth {
				subVal--
			}

			val -= subVal
			if val < 0 {
				break
			}

			elmWidth++
			narrowMask &= ^(1 << bar)
		}

		val += subVal
		n -= elmWidth
		widths[bar] = elmWidth
	}

	widths[elements-1] = n
	return widths
}

// calculateGTINCheckDigit calculates GS1 GTIN-14 check digit from 13 digits.
func calculateGTINCheckDigit(src13 string) int {
	total := 0
	for i := 0; i < 13; i++ {
		weight := 3
		if i%2 != 0 {
			weight = 1
		}
		total += int(src13[i]-'0') * weight
	}
	return (10 - (total % 10)) % 10
}

// bin2pat converts a binary string to run-length encoded pattern string.
func bin2pat(binary string) string {
	if len(binary) == 0 {
		return ""
	}

	result := []byte{}
	count := 1
	lastChar := binary[0]

	for i := 1; i < len(binary); i++ {
		if binary[i] == lastChar {
			count++
		} else {
			result = append(result, byte('0'+count))
			count = 1
			lastChar = binary[i]
		}
	}
	result = append(result, byte('0'+count))

	return string(result)
}
