package barcode

// This file contains SVG/PNG rendering for 4-state barcodes (YubinCustomer).
// Generic 1D SVG rendering is in base_1d.go.

import (
	"fmt"
	"image"
	"image/draw"
	"math"
)

// drawSVG4State renders a 4-state postal barcode as SVG.
func (b *BarcodeBase1D) drawSVG4State(bars []int, width, height int) error {
	n := len(bars)
	if n == 0 {
		return fmt.Errorf("empty bar pattern")
	}
	pitch := float64(width) / float64(n)
	barW := math.Max(1, math.Round(pitch*0.45))
	h3 := float64(height) / 3.0

	b.svgBegin(width, height)
	b.svgRect(0, 0, float64(width), float64(height), b.backColor)

	for i, bt := range bars {
		bx := math.Round(float64(i)*pitch + (pitch-barW)/2.0)
		var by, bh float64
		switch bt {
		case 1:
			by, bh = 0, float64(height)
		case 2:
			by, bh = 0, math.Round(h3*2)
		case 3:
			by, bh = math.Round(h3), math.Round(h3*2)
		case 4:
			by, bh = math.Round(h3), math.Round(h3)
		default:
			continue
		}
		b.svgRect(bx, by, barW, bh, b.foreColor)
	}

	b.svgEnd()
	return nil
}

// drawPNG4State renders a 4-state postal barcode as PNG.
func (b *BarcodeBase1D) drawPNG4State(bars []int, width, height int) error {
	n := len(bars)
	if n == 0 {
		return fmt.Errorf("empty bar pattern")
	}

	img := image.NewNRGBA(image.Rect(0, 0, width, height))
	draw.Draw(img, img.Bounds(), &image.Uniform{b.backColor}, image.Point{}, draw.Src)

	pitch := float64(width) / float64(n)
	barW := int(math.Max(1, math.Round(pitch*0.45)))
	h3 := float64(height) / 3.0

	for i, bt := range bars {
		bx := int(math.Round(float64(i)*pitch + (pitch-float64(barW))/2.0))
		var by, bh int
		switch bt {
		case 1:
			by, bh = 0, height
		case 2:
			by, bh = 0, int(math.Round(h3*2))
		case 3:
			by, bh = int(math.Round(h3)), int(math.Round(h3*2))
		case 4:
			by, bh = int(math.Round(h3)), int(math.Round(h3))
		default:
			continue
		}
		fillRect(img, bx, by, bx+barW, by+bh, b.foreColor)
	}

	if IsTrialMode() {
		drawSampleOverlayPNG(img, 0, 0, width, height)
	}

	return b.encodeImage(img)
}
