package barcode

import (
	"bytes"
	"fmt"
	"image"
	"image/color"
	"image/draw"
	"image/jpeg"
	"image/png"
	"math"

	"golang.org/x/image/font"
	"golang.org/x/image/font/opentype"
	"golang.org/x/image/math/fixed"
)

// drawPNG1D renders a generic 1D barcode to PNG.
func (b *BarcodeBase1D) drawPNG1D(pattern []int, code string, width, height int) error {
	display := ""
	if b.ShowText {
		display = code
	}
	return b.renderBarsToPNG(pattern, width, height, display)
}

func (b *BarcodeBase1D) renderBarsToPNG(pattern []int, width, height int, text string) error {
	totalUnits := 0
	for _, v := range pattern {
		totalUnits += v
	}
	if totalUnits <= 0 {
		return fmt.Errorf("pattern has zero total units")
	}

	fontSize := int(math.Max(8, float64(height)*0.15*b.textFontScale))

	textH := 0
	if text != "" {
		textH = int(float64(fontSize) * 1.4)
	}
	barH := height - textH
	if barH <= 0 {
		barH = height
		text = ""
	}

	img := image.NewNRGBA(image.Rect(0, 0, width, height))
	// Fill background
	draw.Draw(img, img.Bounds(), &image.Uniform{b.backColor}, image.Point{}, draw.Src)

	unitW := float64(width) / float64(totalUnits)
	accum := 0.0
	isBar := true
	for _, units := range pattern {
		x1 := int(math.Round(accum * unitW))
		accum += float64(units)
		x2 := int(math.Round(accum * unitW))
		if isBar && x2 > x1 {
			fillRect(img, x1, 0, x2, barH, b.foreColor)
		}
		isBar = !isBar
	}

	// Draw text
	if text != "" {
		textY := barH + 2
		b.drawTextOnImage(img, text, 0, textY, width, fontSize)
	}

	// Trial mode watermark
	if IsTrialMode() {
		drawSampleOverlayPNG(img, 0, 0, width, height)
	}

	return b.encodeImage(img)
}

// drawTextOnImage draws evenly-distributed or centered text on the image.
func (b *BarcodeBase1D) drawTextOnImage(img *image.NRGBA, text string, x, y, width, fontSize int) {
	face := getDefaultFontFace(fontSize)
	if face == nil {
		return
	}
	defer face.Close()

	if b.TextEvenSpacing {
		drawTextEvenPNG(img, face, text, float64(x), float64(y), float64(width), fontSize, b.foreColor, b.textHorizontalSpacingScale)
	} else {
		drawTextCenteredPNG(img, face, text, x+width/2, y, b.foreColor)
	}
}

func drawTextEvenPNG(img *image.NRGBA, face font.Face, text string, x, y, width float64, fontSize int, c RGBA, hScale float64) {
	numChars := len(text)
	if numChars == 0 {
		return
	}

	marginRatio := 0.05
	if width < 100 {
		marginRatio = 0.03
	} else if width > 400 {
		marginRatio = 0.07
	}

	horizontalMargin := width * marginRatio * hScale
	textWidth := width - 2*horizontalMargin

	charSpacing := textWidth
	if numChars > 1 {
		charSpacing = textWidth / float64(numChars-1)
	}

	minSpacing := float64(fontSize) / 2
	if charSpacing < minSpacing {
		charSpacing = minSpacing
	}

	totalTextWidth := 0.0
	if numChars > 1 {
		totalTextWidth = charSpacing * float64(numChars-1)
	}
	startX := x + (width-totalTextWidth)/2

	for i := 0; i < numChars; i++ {
		charX := startX + float64(i)*charSpacing
		drawCharCenteredPNG(img, face, rune(text[i]), int(math.Round(charX)), int(math.Round(y)), c)
	}
}

func drawTextCenteredPNG(img *image.NRGBA, face font.Face, text string, cx, y int, c RGBA) {
	totalW := fixed.Int26_6(0)
	for _, ch := range text {
		adv, ok := face.GlyphAdvance(ch)
		if ok {
			totalW += adv
		}
	}
	startX := cx - totalW.Round()/2

	d := &font.Drawer{
		Dst:  img,
		Src:  image.NewUniform(c),
		Face: face,
		Dot:  fixed.P(startX, y+face.Metrics().Ascent.Round()),
	}
	d.DrawString(text)
}

func drawCharCenteredPNG(img *image.NRGBA, face font.Face, ch rune, cx, y int, c RGBA) {
	adv, ok := face.GlyphAdvance(ch)
	if !ok {
		return
	}
	startX := cx - adv.Round()/2

	d := &font.Drawer{
		Dst:  img,
		Src:  image.NewUniform(c),
		Face: face,
		Dot:  fixed.P(startX, y+face.Metrics().Ascent.Round()),
	}
	d.DrawString(string(ch))
}

func drawSampleOverlayPNG(img *image.NRGBA, x, y, width, height int) {
	fontSize := int(float64(height) * 0.12)
	if fontSize < 8 {
		fontSize = 8
	}
	if fontSize > 40 {
		fontSize = 40
	}
	margin := int(float64(height) * 0.01)
	if margin < 2 {
		margin = 2
	}
	text := getTrialText()
	red := RGBA{255, 0, 0, 255}

	face := getDefaultFontFace(fontSize)
	if face == nil {
		return
	}
	defer face.Close()

	d := &font.Drawer{
		Dst:  img,
		Src:  image.NewUniform(red),
		Face: face,
		Dot:  fixed.P(x+margin, y+margin+face.Metrics().Ascent.Round()),
	}
	d.DrawString(text)
}

// fillRect fills a rectangle on an NRGBA image.
func fillRect(img *image.NRGBA, x1, y1, x2, y2 int, c color.NRGBA) {
	rect := image.Rect(x1, y1, x2, y2).Intersect(img.Bounds())
	for py := rect.Min.Y; py < rect.Max.Y; py++ {
		for px := rect.Min.X; px < rect.Max.X; px++ {
			img.SetNRGBA(px, py, c)
		}
	}
}

// encodeImage encodes the image to PNG or JPEG and stores in imageBuffer.
func (b *BarcodeBase) encodeImage(img image.Image) error {
	var buf bytes.Buffer
	switch b.format {
	case FormatJPEG:
		if err := jpeg.Encode(&buf, img, &jpeg.Options{Quality: 95}); err != nil {
			return err
		}
	default:
		if err := png.Encode(&buf, img); err != nil {
			return err
		}
	}
	b.imageBuffer = buf.Bytes()
	return nil
}

// getDefaultFontFace returns a font.Face from the embedded Roboto font.
func getDefaultFontFace(size int) font.Face {
	data := getRobotoFontData()
	if data == nil {
		return nil
	}
	f, err := opentype.Parse(data)
	if err != nil {
		return nil
	}
	face, err := opentype.NewFace(f, &opentype.FaceOptions{
		Size:    float64(size),
		DPI:     72,
		Hinting: font.HintingFull,
	})
	if err != nil {
		return nil
	}
	return face
}
